#!/bin/bash
# Mine blocks in regtest mode
#
# Usage:
#   mine.sh [N]           Mine N blocks (default: 1)
#   mine.sh --address X   Mine to specific address
#
# Examples:
#   mine.sh             # Mine 1 block
#   mine.sh 6           # Mine 6 blocks
#   mine.sh 100         # Mine 100 blocks

set -e

BLOCKS="${1:-1}"
ADDRESS=""
BITCOIND_CONTAINER="bitcoind"

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --address)
            ADDRESS="$2"
            shift 2
            ;;
        [0-9]*)
            BLOCKS="$1"
            shift
            ;;
        -h|--help)
            echo "Usage: mine.sh [N] [--address ADDRESS]"
            echo ""
            echo "Mine N blocks in regtest mode (default: 1)"
            echo ""
            echo "Options:"
            echo "  --address ADDRESS   Mine to specific address"
            echo ""
            echo "Examples:"
            echo "  mine.sh           # Mine 1 block"
            echo "  mine.sh 6         # Mine 6 blocks (confirm channel)"
            echo "  mine.sh 100       # Mine 100 blocks (coinbase maturity)"
            exit 0
            ;;
        *)
            shift
            ;;
    esac
done

# Get mining address if not specified
if [ -z "$ADDRESS" ]; then
    ADDRESS=$(docker exec "$BITCOIND_CONTAINER" bitcoin-cli -regtest -rpcuser=bitcoin -rpcpassword=bitcoin getnewaddress)
fi

echo "Mining $BLOCKS block(s) to $ADDRESS..."

HASHES=$(docker exec "$BITCOIND_CONTAINER" bitcoin-cli -regtest -rpcuser=bitcoin -rpcpassword=bitcoin generatetoaddress "$BLOCKS" "$ADDRESS")

if [ "$BLOCKS" -eq 1 ]; then
    echo "Mined block: $(echo "$HASHES" | jq -r '.[0]')"
else
    echo "Mined $BLOCKS blocks"
    echo "First: $(echo "$HASHES" | jq -r '.[0]')"
    echo "Last:  $(echo "$HASHES" | jq -r '.[-1]')"
fi

# Show new height
HEIGHT=$(docker exec "$BITCOIND_CONTAINER" bitcoin-cli -regtest -rpcuser=bitcoin -rpcpassword=bitcoin getblockchaininfo | jq -r '.blocks')
echo "Block height: $HEIGHT"
