#!/bin/bash
# Initialize regtest environment with funded eclair wallet
#
# Usage:
#   regtest-setup.sh [options]
#
# Options:
#   --blocks N        Generate N initial blocks (default: 101)
#   --fund-amount N   Fund eclair with N BTC (default: 10)
#   --api-password P  Eclair API password (default: devpassword)

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

BLOCKS=101
FUND_AMOUNT=10
API_PASSWORD="devpassword"
BITCOIND_CONTAINER="bitcoind"
ECLAIR_CONTAINER="eclair"

# Bitcoin CLI helper
btc() {
    docker exec "$BITCOIND_CONTAINER" bitcoin-cli -regtest -rpcuser=bitcoin -rpcpassword=bitcoin "$@"
}

# Eclair CLI helper
ecl() {
    docker exec "$ECLAIR_CONTAINER" eclair-cli -p "$API_PASSWORD" "$@"
}

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --blocks)
            BLOCKS="$2"
            shift 2
            ;;
        --fund-amount)
            FUND_AMOUNT="$2"
            shift 2
            ;;
        --api-password)
            API_PASSWORD="$2"
            shift 2
            ;;
        -h|--help)
            echo "Usage: regtest-setup.sh [options]"
            echo ""
            echo "Options:"
            echo "  --blocks N        Generate N initial blocks (default: 101)"
            echo "  --fund-amount N   Fund eclair with N BTC (default: 10)"
            echo "  --api-password P  Eclair API password (default: devpassword)"
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

echo "=== Regtest Setup ==="
echo ""

# Check containers are running
echo "Checking containers..."
if ! docker ps --format '{{.Names}}' | grep -q "^${BITCOIND_CONTAINER}$"; then
    echo "Error: bitcoind container not running"
    echo "Start with: cd ~/.claude/skills/eclair/templates && docker-compose up -d"
    exit 1
fi

if ! docker ps --format '{{.Names}}' | grep -q "^${ECLAIR_CONTAINER}$"; then
    echo "Error: eclair container not running"
    echo "Start with: cd ~/.claude/skills/eclair/templates && docker-compose up -d"
    exit 1
fi

echo "  bitcoind: running"
echo "  eclair: running"
echo ""

# Wait for eclair to be ready
echo "Waiting for eclair to sync..."
for i in {1..30}; do
    if ecl getinfo > /dev/null 2>&1; then
        break
    fi
    sleep 1
done

# Get blockchain info
BLOCK_HEIGHT=$(btc getblockchaininfo | jq -r '.blocks')
echo "Current block height: $BLOCK_HEIGHT"

# Generate initial blocks if needed
if [ "$BLOCK_HEIGHT" -lt 100 ]; then
    echo ""
    echo "Generating $BLOCKS blocks for coinbase maturity..."
    MINER_ADDR=$(btc getnewaddress)
    btc generatetoaddress "$BLOCKS" "$MINER_ADDR" > /dev/null
    BLOCK_HEIGHT=$(btc getblockchaininfo | jq -r '.blocks')
    echo "New block height: $BLOCK_HEIGHT"
fi

# Get eclair address and fund it
echo ""
echo "Getting eclair address..."
ECLAIR_ADDR=$(ecl getnewaddress)
echo "Eclair address: $ECLAIR_ADDR"

echo ""
echo "Funding eclair with $FUND_AMOUNT BTC..."
TXID=$(btc sendtoaddress "$ECLAIR_ADDR" "$FUND_AMOUNT")
echo "Funding txid: $TXID"

# Mine a block to confirm
echo ""
echo "Mining block to confirm funding..."
MINER_ADDR=$(btc getnewaddress)
btc generatetoaddress 1 "$MINER_ADDR" > /dev/null

# Wait for eclair to see the funds
echo ""
echo "Waiting for eclair to see funds..."
sleep 2

# Show eclair status
echo ""
echo "=== Eclair Status ==="
echo ""
ecl getinfo | jq '{nodeId, alias, blockHeight, publicAddresses}'
echo ""
echo "On-chain balance:"
ecl onchainbalance | jq '.'
echo ""
echo "=== Ready for Lightning! ==="
echo ""
echo "Next steps:"
echo "  1. Connect to a peer:    eclair-cli connect --uri=<nodeId>@<host>:9735"
echo "  2. Open a channel:       eclair-cli open --nodeId=<nodeId> --fundingSatoshis=1000000"
echo "  3. Mine blocks:          docker exec bitcoind bitcoin-cli -regtest -rpcuser=bitcoin -rpcpassword=bitcoin generatetoaddress 6 \$(docker exec bitcoind bitcoin-cli -regtest -rpcuser=bitcoin -rpcpassword=bitcoin getnewaddress)"
echo ""
echo "Or use helper scripts:"
echo "  ~/.claude/skills/eclair/scripts/eclair-cli.sh getinfo"
echo "  ~/.claude/skills/eclair/scripts/mine.sh 6"
