# Apify Integration Guide

**Status:** Production Ready ✅
**Token Savings:** 90-98% vs traditional MCP approach
**Execution Time:** ~10 seconds typical

## Integration with PAI Skills

### Social Skill Integration

**Location:** `~/.opencode/skills/social/SKILL.md`

**Updated Section:** "Fetching Tweet Content"

The social skill now uses code-based Apify scripts instead of `mcp__apify` MCP tool.

**Trigger → Script Mapping:**

| User Says | Script to Run |
|-----------|---------------|
| "my latest tweet" | `get-latest-tweet.ts` |
| "my latest thread" | `get-latest-thread.ts` |
| "get tweets from @user" | `get-user-tweets.ts user 5` |
| "what has @user been talking about" | `get-user-tweets.ts user 10` |

**Example Workflow:**

1. User: "Turn my latest tweet into a LinkedIn post"
2. System runs: `bun ~/.opencode/filesystem-mcps/apify/get-latest-tweet.ts`
3. Script returns: Tweet text + metadata (~500 tokens)
4. System transforms tweet into LinkedIn format
5. **Token savings: 98%** (vs fetching unfiltered profile data)

### Research Skill Integration

**Use Case:** Monitor influential developers' Twitter activity

```bash
# Research what ThePrimeagen is discussing
bun ~/.opencode/filesystem-mcps/apify/get-user-tweets.ts ThePrimeagen 10

# Analyze Paul Graham's recent thoughts
bun ~/.opencode/filesystem-mcps/apify/get-user-tweets.ts paulg 20

# Track Simon Willison's posts
bun ~/.opencode/filesystem-mcps/apify/get-user-tweets.ts simonw 15
```

**Token Efficiency:**
- 10 tweets unfiltered: ~80,000 tokens
- 10 tweets filtered: ~8,000 tokens
- **Savings: 90%**

### Writing Skill Integration

**Use Case:** Generate blog content from Twitter discussions

```bash
# Get user's thread about AI topic
bun ~/.opencode/filesystem-mcps/apify/get-latest-thread.ts

# Expand thread into blog post format
# Token efficient: only thread content in context
```

## Available Scripts Summary

### 1. get-latest-tweet.ts
**Purpose:** User's most recent single tweet
**Usage:** `bun get-latest-tweet.ts`
**Returns:** Text, date, URL, engagement stats
**Tokens:** ~500

### 2. get-latest-thread.ts
**Purpose:** User's most recent Twitter thread
**Usage:** `bun get-latest-thread.ts`
**Returns:** All thread tweets chronologically
**Tokens:** ~5,500 (for 5-tweet thread)
**Savings:** 87-90% vs unfiltered

### 3. get-user-tweets.ts
**Purpose:** Any user's recent tweets
**Usage:** `bun get-user-tweets.ts <username> <limit>`
**Returns:** Recent tweets with metadata
**Tokens:** ~800 per tweet
**Savings:** 90-95% vs unfiltered

### 4. debug-tweet-structure.ts
**Purpose:** Inspect raw API response
**Usage:** `bun debug-tweet-structure.ts`
**Returns:** Full JSON structure + available fields
**Use:** Development/debugging only

## Migration from MCP

### Before (MCP Approach)

```typescript
// Step 1: Search for actors (~1,000 tokens)
mcp__Apify__search-actors("twitter scraper")

// Step 2: Call actor (~1,000 tokens)
mcp__Apify__call-actor(actorId, input)

// Step 3: Get output (~50,000 tokens unfiltered!)
mcp__Apify__get-actor-output(runId)

// Total: ~57,000 tokens
```

### After (Code-Based Approach)

```typescript
// All in one script, filtering in code
bun ~/.opencode/filesystem-mcps/apify/get-latest-tweet.ts

// Returns only filtered result: ~500 tokens
// Savings: 98.2%
```

## Best Practices

### DO:
✅ Use appropriate script for the task
✅ Let script filter data before returning
✅ Trust token savings calculations
✅ Run from `~/.opencode/filesystem-mcps/apify/` directory or use full path
✅ Check execution time (~10 seconds expected)

### DON'T:
❌ Fall back to MCP tools for Twitter operations
❌ Fetch unfiltered data into model context
❌ Re-implement filtering logic (use existing scripts)
❌ Skip error handling (scripts handle common errors)
❌ Ignore token savings metrics in output

## Performance Expectations

**Execution Time:**
- Actor search: Eliminated (hardcoded actor ID)
- Actor execution: ~10 seconds (Apify platform time)
- Data processing: <1 second (TypeScript filtering)
- **Total: ~10 seconds**

**Token Usage:**
- Single tweet: 500 tokens (vs 57,000 MCP)
- Thread (5 tweets): 5,500 tokens (vs 60,000 unfiltered)
- User tweets (10): 8,000 tokens (vs 80,000 unfiltered)

**Rate Limits:**
- Apify free tier: 100 actor runs/day
- Apify paid tier: Unlimited
- Current usage: Well within limits

## Error Handling

Scripts handle common errors automatically:

1. **Missing APIFY_TOKEN** → Clear error message with setup instructions
2. **Actor failure** → Reports status and exits cleanly
3. **No results** → Graceful message, no crash
4. **Network timeout** → Configurable timeout (120s default)

**Manual intervention rarely needed.**

## Future Enhancements

### Planned Features:

1. **Search tweets by topic**
   - `search-tweets.ts <username> <query> <limit>`
   - Example: Search user's tweets about "AI" from last month

2. **Thread detection improvements**
   - Better handling of quote tweets
   - Reply chain analysis
   - Thread continuity verification

3. **Engagement analytics**
   - Filter by minimum engagement threshold
   - Sort by engagement metrics
   - Engagement trend analysis

4. **Export formats**
   - JSON output for programmatic use
   - Markdown format for documentation
   - CSV for spreadsheet analysis

### Migration Candidates:

Other Apify actors worth implementing:
- Instagram scraping
- LinkedIn scraping
- YouTube data extraction
- Generic web scraping

**Same pattern applies:** Filter in code, 90%+ token savings expected.

## Documentation

**For Users:**
- Quick reference: `~/.opencode/filesystem-mcps/SCRIPTS-REFERENCE.md`
- Social skill: `~/.opencode/skills/social/SKILL.md`

**For Developers:**
- Implementation: `~/.opencode/filesystem-mcps/apify/README.md`
- Standards: `~/.opencode/filesystem-mcps/STANDARDS.md`
- Parent guide: `~/.opencode/filesystem-mcps/README.md`

## Support

**Common Questions:**

Q: Why not use MCP?
A: 90-98% token savings, faster execution, better control.

Q: What if script fails?
A: Check `APIFY_TOKEN` in `${PAI_DIR}/.env`, verify network, check Apify status.

Q: Can I add new actors?
A: Yes! Follow `STANDARDS.md` pattern, hardcode actor ID, filter in code.

Q: How do I debug?
A: Use `debug-tweet-structure.ts` to inspect raw data, check console output.

## Success Metrics

**Achieved:**
- ✅ 90-98% token reduction vs MCP
- ✅ ~10 second execution time
- ✅ Production integration in social skill
- ✅ 4 production-ready scripts
- ✅ Comprehensive documentation

**This is now the standard for all Twitter operations in PAI.**
