# Agent Patterns Reference

> Multi-agent orchestration patterns for v4.0 thinking framework

## Pattern Overview

| Pattern | Complexity | Agents | Execution |
|---------|------------|--------|-----------|
| **Direct** | Simple | Orchestrator only | Sequential |
| **Specialist** | Medium | Orchestrator + 1 Specialist | Sequential |
| **Parallel** | Complex | Orchestrator + 2-3 Specialists | Parallel |
| **Deep Dive** | Very Complex | Full pipeline + Parallel Workers | Hybrid |

---

## Pattern 1: Direct (Simple Problems)

```
User Request → [Orchestrator] → Result
                    │
                    └── Clear-Thought tool (single)
```

**When**:
- Single root cause
- 1-2 analysis steps
- Clear problem statement

**Implementation**:
```markdown
1. Orchestrator assesses complexity → Simple
2. Selects appropriate method (B Routine)
3. Applies Clear-Thought tool directly
4. Returns result with G4 quality gate
```

**Example**: "Why did the build fail?" → 5 Why with `sequentialthinking`

---

## Pattern 2: Specialist (Medium Problems)

```
User Request → [Orchestrator] → [Specialist] → [Orchestrator] → Result
                    │                │                │
                    │                └── Clear-Thought │
                    └── Complexity assessment          └── Integration
```

**When**:
- 3-5 contributing factors
- Some ambiguity in problem definition
- Requires specialized method execution

**Implementation**:
```markdown
1. Orchestrator assesses complexity → Medium
2. Dispatches ProblemDefiner OR MethodExecutor
3. Specialist executes with Clear-Thought tools
4. Orchestrator integrates with G2 + G4 gates
```

**Example**: "How can we improve API performance?" → ProblemDefiner + MethodExecutor (Fishbone)

---

## Pattern 3: Parallel (Complex Problems)

```
User Request → [Orchestrator] → [ProblemDefiner]
                    │                   │
                    │           ┌───────┴───────┐
                    │           ↓               ↓
                    │    [MethodExecutor]  [MethodExecutor]
                    │           │               │
                    │           └───────┬───────┘
                    │                   ↓
                    │          [StrategyArchitect]
                    │                   │
                    └───────────────────┘
                              ↓
                           Result
```

**When**:
- 5+ interdependent factors
- Multiple analysis perspectives needed
- Strategic synthesis required

**Implementation**:
```markdown
1. Orchestrator assesses complexity → Complex
2. ProblemDefiner decomposes into sub-problems
3. MethodExecutors analyze sub-problems in parallel
4. StrategyArchitect synthesizes findings
5. Orchestrator validates with all quality gates
```

**Example**: "Design a migration strategy for our monolith to microservices"

---

## Pattern 4: Deep Dive (Very Complex)

```
[Orchestrator]
      │
[ProblemDefiner]
      │
      ├─────────────────┬─────────────────┐
      ↓                 ↓                 ↓
[MethodExecutor]   [MethodExecutor]   [MethodExecutor]
      │                 │                 │
      ├── Parallel      ├── Parallel      ├── Parallel
      │   Workers       │   Workers       │   Workers
      │                 │                 │
      └─────────────────┴─────────────────┘
                        ↓
               [StrategyArchitect]
                        │
                        ├── Multi-persona synthesis
                        │
                        ↓
                   [Orchestrator]
                        │
                     Result
```

**When**:
- Systemic organizational problems
- Multi-domain analysis required
- Long-term strategic planning

**Example**: "Redesign our entire engineering culture and processes"

---

## Agent Communication Protocol

### Handoff Format

```json
{
  "from": "agent_id",
  "to": "next_agent_id",
  "payload": {
    "problem": "problem statement",
    "analysis": "findings so far",
    "confidence": 0.85,
    "uncertainties": ["list of unknowns"],
    "recommended_action": "next step"
  }
}
```

### Quality Gate Check Format

```json
{
  "gate": "G1|G2|G3|G4",
  "status": "pass|fail|warning",
  "score": 0.9,
  "issues": ["list if any"],
  "recommendation": "proceed|revise|escalate"
}
```

---

## Clear-Thought Tool Selection Guide

### By Analysis Type

| Analysis Type | Primary Tool | Alternative |
|---------------|--------------|-------------|
| Sequential reasoning | `sequentialthinking` | - |
| Multi-perspective | `collaborativereasoning` | `structuredargumentation` |
| Decision evaluation | `decisionframework` | - |
| Hypothesis testing | `scientificmethod` | - |
| Visual mapping | `visualreasoning` | - |
| Self-assessment | `metacognitivemonitoring` | - |
| Mental models | `mentalmodel` | - |

### By Method Category

| Category | Methods | Tools |
|----------|---------|-------|
| Root Cause | 5 Why, Fishbone | sequentialthinking, collaborativereasoning |
| Innovation | SCAMPER, TRIZ, Design Thinking | collaborativereasoning, mentalmodel |
| Strategy | SWOT, GAP Analysis | decisionframework, visualreasoning |
| Process | Pareto, PDCA, DMAIC | decisionframework, scientificmethod |
| Decision | OODA, Kepner-Tregoe | scientificmethod, decisionframework |
| Synthesis | Dialectic | structuredargumentation |

---

## Parallel Execution Patterns

### Fishbone Parallel Workers

```
[MethodExecutor: Fishbone]
         │
    ┌────┼────┬────┬────┬────┐
    ↓    ↓    ↓    ↓    ↓    ↓
 People Process Tech Env Materials Metrics
    │    │    │    │    │    │
    └────┴────┴────┴────┴────┘
              ↓
      Integration Agent
```

### SWOT Parallel Analysis

```
[MethodExecutor: SWOT]
         │
    ┌────┼────┬────┐
    ↓    ↓    ↓    ↓
   S    W    O    T
   │    │    │    │
   └────┴────┴────┘
         ↓
    2x2 Matrix
```

---

## Error Handling

| Error Type | Detection | Recovery |
|------------|-----------|----------|
| Agent timeout | No response in 60s | Retry with simplified prompt |
| Quality gate fail | Score < 0.6 | Revise and resubmit |
| Tool mismatch | Wrong output format | Re-select tool |
| Circular dependency | Same issue reappears | Escalate to user |

---

## Performance Optimization

| Optimization | Benefit | Trade-off |
|--------------|---------|-----------|
| Parallel agents | Faster complex analysis | Higher token usage |
| Skip ProblemDefiner | Faster for clear problems | May miss nuances |
| Simplified quality gates | Faster iteration | Lower confidence |
| Cache method selection | Faster repeated patterns | May miss context changes |

---

## Version History

| Version | Changes |
|---------|---------|
| v4.0 | Initial multi-agent architecture |
| v3.1 | Single-agent with Sequential Thinking |
