# API Integration Examples

Complete examples of API clients generated by the api-integration-builder skill.

## Example 1: Stripe Payment Integration

### User Request
"Create a Stripe API client for handling customers and payments"

### Generated Client

```typescript
// stripe-client.ts
import axios, { AxiosInstance } from 'axios';
import { AuthHandler } from './auth';
import { RateLimiter } from './rate-limiter';
import { RetryHandler } from './retry';
import type { ClientConfig, Customer, Charge, CreateCustomerData, CreateChargeData } from './types';

export class StripeClient {
  private axios: AxiosInstance;
  private rateLimiter: RateLimiter;
  private retryHandler: RetryHandler;

  constructor(config: ClientConfig) {
    this.rateLimiter = new RateLimiter({ requests: 100, per: 'second' });
    this.retryHandler = new RetryHandler({
      maxRetries: 3,
      initialDelay: 1000,
      backoffFactor: 2,
    });

    this.axios = axios.create({
      baseURL: 'https://api.stripe.com/v1',
      timeout: 30000,
      auth: {
        username: config.apiKey,
        password: '',
      },
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        'Stripe-Version': '2023-10-16',
      },
    });

    // Add interceptors for rate limiting and retry
    this.axios.interceptors.request.use(async (config) => {
      await this.rateLimiter.wait();
      return config;
    });

    this.axios.interceptors.response.use(
      (response) => response,
      async (error) => {
        if (this.retryHandler.shouldRetry(error)) {
          return this.retryHandler.retry(error);
        }
        throw this.handleError(error);
      }
    );
  }

  private handleError(error: any): Error {
    if (error.response?.data?.error) {
      const { message, type, code } = error.response.data.error;
      return new Error(`Stripe ${type}: ${message} (${code})`);
    }
    return error;
  }

  // Customer methods
  async createCustomer(data: CreateCustomerData): Promise<Customer> {
    const response = await this.axios.post('/customers', this.toFormData(data));
    return response.data;
  }

  async getCustomer(customerId: string): Promise<Customer> {
    const response = await this.axios.get(`/customers/${customerId}`);
    return response.data;
  }

  async listCustomers(params?: { limit?: number; starting_after?: string }): Promise<{
    data: Customer[];
    has_more: boolean;
  }> {
    const response = await this.axios.get('/customers', { params });
    return response.data;
  }

  async updateCustomer(customerId: string, data: Partial<CreateCustomerData>): Promise<Customer> {
    const response = await this.axios.post(`/customers/${customerId}`, this.toFormData(data));
    return response.data;
  }

  async deleteCustomer(customerId: string): Promise<{ deleted: boolean; id: string }> {
    const response = await this.axios.delete(`/customers/${customerId}`);
    return response.data;
  }

  // Charge methods
  async createCharge(data: CreateChargeData): Promise<Charge> {
    const response = await this.axios.post('/charges', this.toFormData(data));
    return response.data;
  }

  async getCharge(chargeId: string): Promise<Charge> {
    const response = await this.axios.get(`/charges/${chargeId}`);
    return response.data;
  }

  async refundCharge(chargeId: string, amount?: number): Promise<any> {
    const response = await this.axios.post('/refunds', this.toFormData({
      charge: chargeId,
      ...(amount && { amount }),
    }));
    return response.data;
  }

  // Helper method to convert JSON to form data (Stripe requirement)
  private toFormData(obj: any, prefix = ''): URLSearchParams {
    const params = new URLSearchParams();
    for (const [key, value] of Object.entries(obj)) {
      if (value === undefined || value === null) continue;
      const paramKey = prefix ? `${prefix}[${key}]` : key;
      if (typeof value === 'object' && !Array.isArray(value)) {
        const nested = this.toFormData(value, paramKey);
        nested.forEach((v, k) => params.append(k, v));
      } else {
        params.append(paramKey, String(value));
      }
    }
    return params;
  }
}
```

### Usage Example

```typescript
import { StripeClient } from './stripe-client';

const stripe = new StripeClient({
  apiKey: process.env.STRIPE_SECRET_KEY!,
});

// Create a customer
const customer = await stripe.createCustomer({
  email: 'customer@example.com',
  name: 'John Doe',
  metadata: { user_id: '12345' },
});

console.log(`Customer created: ${customer.id}`);

// Create a charge
const charge = await stripe.createCharge({
  amount: 2000, // $20.00
  currency: 'usd',
  source: 'tok_visa', // Test token
  customer: customer.id,
  description: 'Purchase from example.com',
});

console.log(`Charge successful: ${charge.id}`);

// List customers with pagination
const customers = await stripe.listCustomers({ limit: 10 });
console.log(`Found ${customers.data.length} customers`);

// Refund a charge
await stripe.refundCharge(charge.id);
console.log('Charge refunded');
```

---

## Example 2: SendGrid Email Integration

### User Request
"Set up SendGrid API for sending emails"

### Generated Client

```typescript
// sendgrid-client.ts
import axios, { AxiosInstance } from 'axios';
import { RateLimiter } from './rate-limiter';
import type { ClientConfig, EmailData, EmailResponse } from './types';

export class SendGridClient {
  private axios: AxiosInstance;
  private rateLimiter: RateLimiter;

  constructor(config: ClientConfig) {
    this.rateLimiter = new RateLimiter({ requests: 600, per: 'minute' });

    this.axios = axios.create({
      baseURL: 'https://api.sendgrid.com/v3',
      timeout: 30000,
      headers: {
        'Authorization': `Bearer ${config.apiKey}`,
        'Content-Type': 'application/json',
      },
    });

    this.axios.interceptors.request.use(async (config) => {
      await this.rateLimiter.wait();
      return config;
    });
  }

  async sendEmail(data: EmailData): Promise<EmailResponse> {
    const payload = {
      personalizations: [
        {
          to: [{ email: data.to, name: data.toName }],
          subject: data.subject,
        },
      ],
      from: { email: data.from, name: data.fromName },
      content: [
        {
          type: data.contentType || 'text/html',
          value: data.body,
        },
      ],
      reply_to: data.replyTo ? { email: data.replyTo } : undefined,
    };

    const response = await this.axios.post('/mail/send', payload);

    return {
      success: response.status === 202,
      messageId: response.headers['x-message-id'],
    };
  }

  async sendBulkEmails(emails: EmailData[]): Promise<EmailResponse[]> {
    return Promise.all(emails.map((email) => this.sendEmail(email)));
  }

  async sendTemplateEmail(data: {
    to: string;
    templateId: string;
    dynamicData: Record<string, any>;
  }): Promise<EmailResponse> {
    const payload = {
      personalizations: [
        {
          to: [{ email: data.to }],
          dynamic_template_data: data.dynamicData,
        },
      ],
      from: { email: 'noreply@example.com' },
      template_id: data.templateId,
    };

    const response = await this.axios.post('/mail/send', payload);

    return {
      success: response.status === 202,
      messageId: response.headers['x-message-id'],
    };
  }
}
```

### Usage Example

```typescript
import { SendGridClient } from './sendgrid-client';

const sendgrid = new SendGridClient({
  apiKey: process.env.SENDGRID_API_KEY!,
});

// Send single email
await sendgrid.sendEmail({
  to: 'user@example.com',
  toName: 'John Doe',
  from: 'hello@myapp.com',
  fromName: 'My App',
  subject: 'Welcome to My App!',
  body: '<h1>Welcome!</h1><p>Thanks for signing up.</p>',
  contentType: 'text/html',
});

// Send template email
await sendgrid.sendTemplateEmail({
  to: 'user@example.com',
  templateId: 'd-1234567890',
  dynamicData: {
    name: 'John',
    verification_link: 'https://myapp.com/verify/abc123',
  },
});

// Send bulk emails
await sendgrid.sendBulkEmails([
  { to: 'user1@example.com', from: 'hello@myapp.com', subject: 'Hello', body: 'Message 1' },
  { to: 'user2@example.com', from: 'hello@myapp.com', subject: 'Hello', body: 'Message 2' },
]);
```

---

## Example 3: GitHub API Client

### User Request
"Create GitHub API client with OAuth for managing repositories"

### Generated Client

```typescript
// github-client.ts
import axios, { AxiosInstance } from 'axios';
import { AuthHandler } from './auth';
import type { ClientConfig, Repository, Issue, PullRequest } from './types';

export class GitHubClient {
  private axios: AxiosInstance;
  private auth: AuthHandler;

  constructor(config: ClientConfig) {
    this.auth = new AuthHandler({
      type: 'oauth',
      clientId: config.oauthConfig!.clientId,
      clientSecret: config.oauthConfig!.clientSecret,
      tokenUrl: 'https://github.com/login/oauth/access_token',
    });

    this.axios = axios.create({
      baseURL: 'https://api.github.com',
      timeout: 30000,
      headers: {
        'Accept': 'application/vnd.github+json',
        'X-GitHub-Api-Version': '2022-11-28',
      },
    });

    this.axios.interceptors.request.use((config) => this.auth.addAuthHeaders(config));
  }

  // Repository methods
  async getRepository(owner: string, repo: string): Promise<Repository> {
    const response = await this.axios.get(`/repos/${owner}/${repo}`);
    return response.data;
  }

  async listRepositories(params?: { type?: 'all' | 'owner' | 'member'; sort?: 'created' | 'updated' | 'pushed' }): Promise<Repository[]> {
    const response = await this.axios.get('/user/repos', { params });
    return response.data;
  }

  async createRepository(data: { name: string; description?: string; private?: boolean }): Promise<Repository> {
    const response = await this.axios.post('/user/repos', data);
    return response.data;
  }

  // Issue methods
  async listIssues(owner: string, repo: string, params?: { state?: 'open' | 'closed' | 'all' }): Promise<Issue[]> {
    const response = await this.axios.get(`/repos/${owner}/${repo}/issues`, { params });
    return response.data;
  }

  async createIssue(owner: string, repo: string, data: { title: string; body?: string; labels?: string[] }): Promise<Issue> {
    const response = await this.axios.post(`/repos/${owner}/${repo}/issues`, data);
    return response.data;
  }

  // Pull Request methods
  async listPullRequests(owner: string, repo: string, params?: { state?: 'open' | 'closed' | 'all' }): Promise<PullRequest[]> {
    const response = await this.axios.get(`/repos/${owner}/${repo}/pulls`, { params });
    return response.data;
  }

  async createPullRequest(
    owner: string,
    repo: string,
    data: { title: string; head: string; base: string; body?: string }
  ): Promise<PullRequest> {
    const response = await this.axios.post(`/repos/${owner}/${repo}/pulls`, data);
    return response.data;
  }

  async mergePullRequest(owner: string, repo: string, pullNumber: number): Promise<{ merged: boolean }> {
    const response = await this.axios.put(`/repos/${owner}/${repo}/pulls/${pullNumber}/merge`);
    return response.data;
  }
}
```

### Usage Example

```typescript
import { GitHubClient } from './github-client';

const github = new GitHubClient({
  oauthConfig: {
    clientId: process.env.GITHUB_CLIENT_ID!,
    clientSecret: process.env.GITHUB_CLIENT_SECRET!,
  },
});

// List repositories
const repos = await github.listRepositories({ type: 'owner', sort: 'updated' });
console.log(`Found ${repos.length} repositories`);

// Create issue
const issue = await github.createIssue('owner', 'repo', {
  title: 'Bug: Login not working',
  body: 'Users are unable to log in after the latest update.',
  labels: ['bug', 'high-priority'],
});
console.log(`Issue created: #${issue.number}`);

// Create pull request
const pr = await github.createPullRequest('owner', 'repo', {
  title: 'Fix: Login bug',
  head: 'feature/fix-login',
  base: 'main',
  body: 'Fixes the login issue reported in #123',
});
console.log(`PR created: #${pr.number}`);

// Merge pull request
await github.mergePullRequest('owner', 'repo', pr.number);
console.log('PR merged successfully');
```

---

## Example 4: Custom REST API

### User Request
"Create a client for my custom REST API at api.myapp.com"

### Generated Client

```typescript
// myapp-client.ts
import axios, { AxiosInstance } from 'axios';
import { RetryHandler } from './retry';
import { RateLimiter } from './rate-limiter';
import type { ClientConfig, User, Product, Order } from './types';

export class MyAppClient {
  private axios: AxiosInstance;
  private retryHandler: RetryHandler;
  private rateLimiter: RateLimiter;

  constructor(config: ClientConfig) {
    this.retryHandler = new RetryHandler();
    this.rateLimiter = new RateLimiter({ requests: 1000, per: 'hour' });

    this.axios = axios.create({
      baseURL: config.baseURL || 'https://api.myapp.com/v1',
      timeout: 30000,
      headers: {
        'Authorization': `Bearer ${config.apiKey}`,
        'Content-Type': 'application/json',
      },
    });

    this.axios.interceptors.request.use(async (config) => {
      await this.rateLimiter.wait();
      return config;
    });

    this.axios.interceptors.response.use(
      (response) => response,
      async (error) => {
        if (this.retryHandler.shouldRetry(error)) {
          return this.retryHandler.retry(error);
        }
        throw error;
      }
    );
  }

  // User methods
  async getUser(userId: string): Promise<User> {
    const response = await this.axios.get(`/users/${userId}`);
    return response.data;
  }

  async updateUser(userId: string, data: Partial<User>): Promise<User> {
    const response = await this.axios.patch(`/users/${userId}`, data);
    return response.data;
  }

  // Product methods
  async listProducts(params?: { category?: string; limit?: number }): Promise<Product[]> {
    const response = await this.axios.get('/products', { params });
    return response.data;
  }

  async getProduct(productId: string): Promise<Product> {
    const response = await this.axios.get(`/products/${productId}`);
    return response.data;
  }

  // Order methods
  async createOrder(data: { userId: string; items: { productId: string; quantity: number }[] }): Promise<Order> {
    const response = await this.axios.post('/orders', data);
    return response.data;
  }

  async getOrder(orderId: string): Promise<Order> {
    const response = await this.axios.get(`/orders/${orderId}`);
    return response.data;
  }

  async listOrders(userId: string): Promise<Order[]> {
    const response = await this.axios.get(`/users/${userId}/orders`);
    return response.data;
  }
}
```

### Usage Example

```typescript
import { MyAppClient } from './myapp-client';

const client = new MyAppClient({
  baseURL: 'https://api.myapp.com/v1',
  apiKey: process.env.MYAPP_API_KEY!,
});

// Get user
const user = await client.getUser('user_123');
console.log(`User: ${user.name} (${user.email})`);

// List products
const products = await client.listProducts({ category: 'electronics', limit: 10 });
console.log(`Found ${products.length} products`);

// Create order
const order = await client.createOrder({
  userId: 'user_123',
  items: [
    { productId: 'prod_456', quantity: 2 },
    { productId: 'prod_789', quantity: 1 },
  ],
});
console.log(`Order created: ${order.id} - Total: $${order.total}`);
```

---

## Common Patterns

### Pattern 1: Pagination

```typescript
async *paginateAll<T>(
  fetchFn: (params: { limit: number; offset: number }) => Promise<{ data: T[]; hasMore: boolean }>
): AsyncGenerator<T> {
  let offset = 0;
  const limit = 100;
  let hasMore = true;

  while (hasMore) {
    const response = await fetchFn({ limit, offset });
    for (const item of response.data) {
      yield item;
    }
    hasMore = response.hasMore;
    offset += limit;
  }
}

// Usage
for await (const customer of client.paginateAll(client.listCustomers)) {
  console.log(customer);
}
```

### Pattern 2: Batch Operations

```typescript
async batchCreate<T>(items: T[], batchSize: number = 10): Promise<T[]> {
  const results: T[] = [];

  for (let i = 0; i < items.length; i += batchSize) {
    const batch = items.slice(i, i + batchSize);
    const batchResults = await Promise.all(batch.map((item) => this.create(item)));
    results.push(...batchResults);

    // Add delay between batches to respect rate limits
    if (i + batchSize < items.length) {
      await new Promise((resolve) => setTimeout(resolve, 1000));
    }
  }

  return results;
}
```

### Pattern 3: Webhook Verification

```typescript
import crypto from 'crypto';

function verifyWebhook(payload: string, signature: string, secret: string): boolean {
  const hmac = crypto.createHmac('sha256', secret);
  const digest = 'sha256=' + hmac.update(payload).digest('hex');
  return crypto.timingSafeEqual(Buffer.from(signature), Buffer.from(digest));
}

// Usage in Express
app.post('/webhook', (req, res) => {
  const signature = req.headers['x-signature'] as string;
  const payload = JSON.stringify(req.body);

  if (!verifyWebhook(payload, signature, process.env.WEBHOOK_SECRET!)) {
    return res.status(401).send('Invalid signature');
  }

  // Process webhook
  console.log('Webhook received:', req.body);
  res.sendStatus(200);
});
```

See main [SKILL.md](SKILL.md) for generation workflow and [reference.md](reference.md) for API patterns.
