#!/bin/bash

# Code Refactoring Skill - Codebase Analyzer
# Scans entire codebase for refactoring opportunities and generates audit report

# Colors for output
RED='\033[0;31m'
YELLOW='\033[1;33m'
GREEN='\033[0;32m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Default values
DIRECTORY="${1:-.}"
MIN_LINES=150
OUTPUT_FILE="refactoring-audit-$(date +%Y%m%d-%H%M%S).md"

# Counters
CRITICAL_COUNT=0
HIGH_COUNT=0
MEDIUM_COUNT=0
GOOD_COUNT=0
TOTAL_FILES=0

# Arrays to store file info
declare -a CRITICAL_FILES
declare -a HIGH_FILES
declare -a MEDIUM_FILES

echo -e "${BLUE}╔════════════════════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}║${NC}  ${CYAN}Code Refactoring Skill - Codebase Analyzer${NC}            ${BLUE}║${NC}"
echo -e "${BLUE}╚════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo "Scanning directory: $DIRECTORY"
echo "Minimum threshold: $MIN_LINES lines"
echo ""

# Function to analyze file
analyze_file() {
    local file=$1
    local lines=$(wc -l < "$file" 2>/dev/null)
    local filename=$(basename "$file")
    local relative_path=$(echo "$file" | sed "s|^$DIRECTORY/||")

    ((TOTAL_FILES++))

    # Get last modified date
    local last_modified=""
    if [[ "$OSTYPE" == "darwin"* ]]; then
        last_modified=$(stat -f "%Sm" -t "%Y-%m-%d" "$file")
    else
        last_modified=$(stat -c "%y" "$file" 2>/dev/null | cut -d' ' -f1)
    fi

    # Categorize by size
    if [ $lines -gt 300 ]; then
        ((CRITICAL_COUNT++))
        CRITICAL_FILES+=("$relative_path|$lines|$last_modified")
        echo -e "${RED}🛑 CRITICAL${NC} $relative_path (${lines} lines)"
    elif [ $lines -gt 200 ]; then
        ((HIGH_COUNT++))
        HIGH_FILES+=("$relative_path|$lines|$last_modified")
        echo -e "${YELLOW}🚨 HIGH${NC}     $relative_path (${lines} lines)"
    elif [ $lines -gt 150 ]; then
        ((MEDIUM_COUNT++))
        MEDIUM_FILES+=("$relative_path|$lines|$last_modified")
        echo -e "${YELLOW}⚠️  MEDIUM${NC}   $relative_path (${lines} lines)"
    else
        ((GOOD_COUNT++))
    fi
}

# Scan JavaScript/TypeScript files
echo -e "${CYAN}Scanning JavaScript/TypeScript files...${NC}"
while IFS= read -r file; do
    analyze_file "$file"
done < <(find "$DIRECTORY" -type f \( -name "*.tsx" -o -name "*.ts" -o -name "*.jsx" -o -name "*.js" \) 2>/dev/null)

# Scan Python files
echo ""
echo -e "${CYAN}Scanning Python files...${NC}"
while IFS= read -r file; do
    analyze_file "$file"
done < <(find "$DIRECTORY" -type f -name "*.py" 2>/dev/null)

# Calculate statistics
NEEDS_ATTENTION=$((CRITICAL_COUNT + HIGH_COUNT + MEDIUM_COUNT))
TOTAL_SCANNED=$TOTAL_FILES

# Generate report
echo ""
echo -e "${BLUE}╔════════════════════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}║${NC}  ${CYAN}Generating Audit Report...${NC}                             ${BLUE}║${NC}"
echo -e "${BLUE}╚════════════════════════════════════════════════════════════╝${NC}"
echo ""

# Create markdown report
cat > "$OUTPUT_FILE" << EOF
# Code Refactoring Audit Report

**Date:** $(date +"%Y-%m-%d %H:%M:%S")
**Directory:** \`$DIRECTORY\`
**Generated By:** Code Refactoring Skill (analyze-codebase.sh)

---

## Executive Summary

This report identifies files that exceed recommended size thresholds and should be considered for refactoring.

### Statistics

| Metric | Count | Percentage |
|--------|-------|------------|
| 🛑 Critical files (>300 lines) | $CRITICAL_COUNT | $(awk "BEGIN {printf \"%.1f%%\", ($CRITICAL_COUNT/$TOTAL_SCANNED)*100}") |
| 🚨 High priority (200-300 lines) | $HIGH_COUNT | $(awk "BEGIN {printf \"%.1f%%\", ($HIGH_COUNT/$TOTAL_SCANNED)*100}") |
| ⚠️ Medium priority (150-200 lines) | $MEDIUM_COUNT | $(awk "BEGIN {printf \"%.1f%%\", ($MEDIUM_COUNT/$TOTAL_SCANNED)*100}") |
| ✅ Healthy files (<150 lines) | $GOOD_COUNT | $(awk "BEGIN {printf \"%.1f%%\", ($GOOD_COUNT/$TOTAL_SCANNED)*100}") |
| **Total files scanned** | **$TOTAL_SCANNED** | **100%** |
| **Files needing attention** | **$NEEDS_ATTENTION** | **$(awk "BEGIN {printf \"%.1f%%\", ($NEEDS_ATTENTION/$TOTAL_SCANNED)*100}")**|

---

## 🛑 Critical Files (>300 lines)

**Immediate action required. These files should be refactored within 1 week.**

EOF

# Add critical files
if [ ${#CRITICAL_FILES[@]} -gt 0 ]; then
    echo "| # | File | Lines | Last Modified | Priority |" >> "$OUTPUT_FILE"
    echo "|---|------|-------|---------------|----------|" >> "$OUTPUT_FILE"

    counter=1
    IFS=$'\n' sorted_critical=($(sort -t'|' -k2 -rn <<<"${CRITICAL_FILES[*]}"))
    for file_info in "${sorted_critical[@]}"; do
        IFS='|' read -r path lines modified <<< "$file_info"
        echo "| $counter | \`$path\` | $lines | $modified | P0 |" >> "$OUTPUT_FILE"
        ((counter++))
    done
else
    echo "*No critical files found. Great job!* ✅" >> "$OUTPUT_FILE"
fi

echo "" >> "$OUTPUT_FILE"
echo "---" >> "$OUTPUT_FILE"
echo "" >> "$OUTPUT_FILE"

# Add high priority files
cat >> "$OUTPUT_FILE" << EOF
## 🚨 High Priority Files (200-300 lines)

**Should be refactored within 1 month.**

EOF

if [ ${#HIGH_FILES[@]} -gt 0 ]; then
    echo "| # | File | Lines | Last Modified | Priority |" >> "$OUTPUT_FILE"
    echo "|---|------|-------|---------------|----------|" >> "$OUTPUT_FILE"

    counter=1
    IFS=$'\n' sorted_high=($(sort -t'|' -k2 -rn <<<"${HIGH_FILES[*]}"))
    for file_info in "${sorted_high[@]}"; do
        IFS='|' read -r path lines modified <<< "$file_info"
        echo "| $counter | \`$path\` | $lines | $modified | P1 |" >> "$OUTPUT_FILE"
        ((counter++))
    done
else
    echo "*No high priority files found.* ✅" >> "$OUTPUT_FILE"
fi

echo "" >> "$OUTPUT_FILE"
echo "---" >> "$OUTPUT_FILE"
echo "" >> "$OUTPUT_FILE"

# Add medium priority files
cat >> "$OUTPUT_FILE" << EOF
## ⚠️ Medium Priority Files (150-200 lines)

**Monitor these files. Refactor before making major changes.**

EOF

if [ ${#MEDIUM_FILES[@]} -gt 0 ]; then
    echo "| # | File | Lines | Last Modified | Priority |" >> "$OUTPUT_FILE"
    echo "|---|------|-------|---------------|----------|" >> "$OUTPUT_FILE"

    counter=1
    IFS=$'\n' sorted_medium=($(sort -t'|' -k2 -rn <<<"${MEDIUM_FILES[*]}"))
    for file_info in "${sorted_medium[@]:0:20}"; do  # Limit to top 20
        IFS='|' read -r path lines modified <<< "$file_info"
        echo "| $counter | \`$path\` | $lines | $modified | P2 |" >> "$OUTPUT_FILE"
        ((counter++))
    done

    if [ ${#MEDIUM_FILES[@]} -gt 20 ]; then
        echo "" >> "$OUTPUT_FILE"
        echo "*Showing top 20 of ${#MEDIUM_FILES[@]} medium priority files.*" >> "$OUTPUT_FILE"
    fi
else
    echo "*No medium priority files found.* ✅" >> "$OUTPUT_FILE"
fi

# Add recommendations
cat >> "$OUTPUT_FILE" << EOF

---

## 📋 Recommended Action Plan

### Phase 1: Critical Files (Weeks 1-2)

**Goal:** Bring all critical files under 300 lines

EOF

if [ $CRITICAL_COUNT -gt 0 ]; then
    echo "**Estimated time:** $(awk "BEGIN {print $CRITICAL_COUNT * 6}") hours (avg 6 hours per file)" >> "$OUTPUT_FILE"
    echo "" >> "$OUTPUT_FILE"
    echo "**Priority order:**" >> "$OUTPUT_FILE"

    counter=1
    IFS=$'\n' sorted_critical=($(sort -t'|' -k2 -rn <<<"${CRITICAL_FILES[*]}"))
    for file_info in "${sorted_critical[@]}"; do
        IFS='|' read -r path lines modified <<< "$file_info"
        echo "$counter. \`$path\` ($lines lines)" >> "$OUTPUT_FILE"
        ((counter++))
    done
else
    echo "✅ No critical files to address!" >> "$OUTPUT_FILE"
fi

cat >> "$OUTPUT_FILE" << EOF

### Phase 2: High Priority Files (Weeks 3-6)

**Goal:** Reduce high priority files by 50%

EOF

if [ $HIGH_COUNT -gt 0 ]; then
    echo "**Estimated time:** $(awk "BEGIN {print int($HIGH_COUNT * 4 / 2)}") hours (50% of files, avg 4 hours each)" >> "$OUTPUT_FILE"
else
    echo "✅ No high priority files to address!" >> "$OUTPUT_FILE"
fi

cat >> "$OUTPUT_FILE" << EOF

### Phase 3: Ongoing Maintenance (Months 2-3)

**Goal:** Address medium priority files opportunistically

- Refactor medium priority files when making other changes
- Establish file size monitoring
- Set up pre-commit hooks to prevent new large files

---

## 🎯 Success Metrics

Track these metrics monthly:

| Metric | Current | Target (3 months) |
|--------|---------|-------------------|
| Critical files | $CRITICAL_COUNT | 0 |
| High priority files | $HIGH_COUNT | <$(awk "BEGIN {print int($HIGH_COUNT / 2)}") |
| Medium priority files | $MEDIUM_COUNT | <$(awk "BEGIN {print int($MEDIUM_COUNT / 2)}") |
| Average file size | TBD | <150 lines |

---

## 🔧 Next Steps

1. **Review this report** with your team
2. **Prioritize critical files** for immediate refactoring
3. **Schedule refactoring sprints** based on estimated time
4. **Set up prevention**:
   - Enable code-refactoring skill for auto-checks
   - Add size limits to code review checklist
   - Run quarterly audits

---

## 📊 Detailed Analysis Commands

To analyze specific files from this report:

\`\`\`bash
# Analyze a specific file
bash ~/.claude/skills/code-refactoring/scripts/check-size.sh path/to/file

# Get more details about a file
wc -l path/to/file
git log --oneline path/to/file | head -5
\`\`\`

To regenerate this report:

\`\`\`bash
bash ~/.claude/skills/code-refactoring/scripts/analyze-codebase.sh $DIRECTORY
\`\`\`

---

**Report saved to:** \`$OUTPUT_FILE\`

**For refactoring patterns and templates, see:**
- \`REFERENCE.md\` - Language-specific refactoring patterns
- \`FORMS.md\` - Refactoring checklists and templates
EOF

# Print summary to console
echo ""
echo -e "${GREEN}╔════════════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║${NC}  ${CYAN}Audit Complete!${NC}                                         ${GREEN}║${NC}"
echo -e "${GREEN}╚════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${CYAN}Summary:${NC}"
echo -e "  Total files scanned: ${BLUE}$TOTAL_SCANNED${NC}"
echo -e "  Files needing attention: ${YELLOW}$NEEDS_ATTENTION${NC} ($(awk "BEGIN {printf \"%.0f%%\", ($NEEDS_ATTENTION/$TOTAL_SCANNED)*100}"))"
echo ""
echo -e "  ${RED}🛑 Critical (>300 lines):${NC} $CRITICAL_COUNT files"
echo -e "  ${YELLOW}🚨 High (200-300 lines):${NC} $HIGH_COUNT files"
echo -e "  ${YELLOW}⚠️  Medium (150-200 lines):${NC} $MEDIUM_COUNT files"
echo -e "  ${GREEN}✅ Healthy (<150 lines):${NC} $GOOD_COUNT files"
echo ""
echo -e "${GREEN}Report saved to:${NC} ${BLUE}$OUTPUT_FILE${NC}"
echo ""
echo -e "${CYAN}Next steps:${NC}"
echo "  1. Review the detailed report: cat $OUTPUT_FILE"
echo "  2. Prioritize critical files for refactoring"
echo "  3. Use code-refactoring skill for individual file analysis"
echo ""

# Calculate estimated time
TOTAL_HOURS=$(awk "BEGIN {print ($CRITICAL_COUNT * 6) + ($HIGH_COUNT * 4) + ($MEDIUM_COUNT * 2)}")
echo -e "${YELLOW}Estimated total refactoring time: ${TOTAL_HOURS} hours${NC}"
echo ""
