#!/bin/bash
# File size checker for feature-orchestrator
# Helps prevent accidental complexity growth
#
# Research Citation:
# "Reviewing 200–400 lines of code at a time detects up to 90% of defects"
# — Microsoft Research, Code Review Best Practices

# Usage:
# ./check-file-size.sh <file-path>
# ./check-file-size.sh src/components/UserProfile.tsx

if [ -z "$1" ]; then
    echo "Usage: $0 <file-path>"
    echo "Example: $0 src/components/UserProfile.tsx"
    exit 1
fi

FILE=$1

if [ ! -f "$FILE" ]; then
    echo "❌ Error: File not found: $FILE"
    exit 1
fi

# Get line count
LINE_COUNT=$(wc -l < "$FILE")
FILE_TYPE=$(echo "$FILE" | sed 's/.*\.//')

echo "================================================"
echo "📏 File Size Analysis"
echo "================================================"
echo ""
echo "📄 File: $FILE"
echo "📊 Lines: $LINE_COUNT"
echo "🏷️  Type: $FILE_TYPE"
echo ""

# Determine thresholds based on file type
if [[ "$FILE_TYPE" == "tsx" ]] || [[ "$FILE_TYPE" == "jsx" ]]; then
    # React components
    IDEAL=200
    WARNING=300
    CRITICAL=400
    FILE_CATEGORY="React Component"
elif [[ "$FILE_TYPE" == "ts" ]] || [[ "$FILE_TYPE" == "js" ]]; then
    # Services/utilities
    IDEAL=250
    WARNING=400
    CRITICAL=500
    FILE_CATEGORY="Service/Utility"
elif [[ "$FILE_TYPE" == "py" ]]; then
    # Python files
    IDEAL=250
    WARNING=400
    CRITICAL=500
    FILE_CATEGORY="Python Module"
else
    # General files
    IDEAL=250
    WARNING=400
    CRITICAL=500
    FILE_CATEGORY="General"
fi

echo "📁 Category: $FILE_CATEGORY"
echo "🎯 Ideal: < $IDEAL lines"
echo "⚠️  Warning: $WARNING lines"
echo "🛑 Critical: $CRITICAL lines"
echo ""

# Assess status
if [ $LINE_COUNT -lt $IDEAL ]; then
    echo "✅ STATUS: GOOD"
    echo "   File size is within ideal range"
    echo ""
    echo "💡 Keep it this way!"
    echo "   - Maintain single responsibility"
    echo "   - Extract early if adding major features"
    exit 0

elif [ $LINE_COUNT -lt $WARNING ]; then
    echo "⚠️  STATUS: WARNING"
    echo "   File is getting large, plan extraction"
    echo ""
    echo "📋 Recommended Actions:"
    echo "   1. Identify extraction candidates:"
    if [[ "$FILE_CATEGORY" == "React Component" ]]; then
        echo "      - Large data arrays (>20 lines) → data file"
        echo "      - Sub-components (modals, forms)"
        echo "      - Custom hooks (4+ useState)"
    elif [[ "$FILE_CATEGORY" == "Python Module" ]]; then
        echo "      - Large classes → split into multiple classes"
        echo "      - Config/constants → separate config file"
        echo "      - Helper functions → utilities module"
    else
        echo "      - Repeated logic → extract to functions"
        echo "      - Config/constants → separate file"
        echo "      - Complex sections → separate modules"
    fi
    echo ""
    echo "   2. Before next addition, extract first"
    echo "   3. Keep additions small (< 100 lines)"
    echo ""
    echo "🎓 Research Insight:"
    echo "   'Code reviews of 200-400 LOC detect 90% of defects'"
    echo "   Keeping files smaller = easier to review & maintain"
    exit 0

elif [ $LINE_COUNT -lt $CRITICAL ]; then
    echo "🚨 STATUS: ALERT"
    echo "   File exceeded warning threshold"
    echo ""
    echo "⚠️  RECOMMENDED: Extract before adding more code"
    echo ""
    echo "📋 Action Plan:"
    echo "   1. Review the file for extraction points"
    echo "   2. Create refactoring plan:"
    if [[ "$FILE_CATEGORY" == "React Component" ]]; then
        echo "      ├─ Extract data to [ComponentName]-data.tsx"
        echo "      ├─ Extract sub-components to [ComponentName]/[SubComponent].tsx"
        echo "      └─ Extract hooks to hooks/use[HookName].ts"
    elif [[ "$FILE_CATEGORY" == "Python Module" ]]; then
        echo "      ├─ Split large classes into multiple files"
        echo "      ├─ Extract config to config.py"
        echo "      └─ Extract utilities to utils.py"
    else
        echo "      ├─ Extract by responsibility"
        echo "      ├─ Move config to separate file"
        echo "      └─ Create utility modules"
    fi
    echo ""
    echo "   3. Invoke code-refactoring skill if needed"
    echo "   4. Get user approval"
    echo "   5. Refactor incrementally"
    echo "   6. Then add new features"
    echo ""
    echo "💡 Better to refactor now than accumulate tech debt!"
    exit 0

else
    echo "🛑 STATUS: CRITICAL"
    echo "   File MUST be refactored immediately"
    echo ""
    echo "❌ DO NOT add more code to this file"
    echo ""
    echo "🚨 REQUIRED ACTIONS:"
    echo "   1. Stop feature work"
    echo "   2. Invoke code-refactoring skill"
    echo "   3. Get user approval for refactoring"
    echo "   4. Break file into smaller modules:"
    if [[ "$FILE_CATEGORY" == "React Component" ]]; then
        echo "      Target: Multiple files < 200 lines each"
        echo "      - Main component: < 200 lines"
        echo "      - Data file: [ComponentName]-data.tsx"
        echo "      - Sub-components: [ComponentName]/[SubComponent].tsx"
        echo "      - Hooks: hooks/use[HookName].ts"
    elif [[ "$FILE_CATEGORY" == "Python Module" ]]; then
        echo "      Target: Multiple files < 250 lines each"
        echo "      - Split into package with __init__.py"
        echo "      - Separate classes into files"
        echo "      - Extract config and constants"
    else
        echo "      Target: Multiple files < 250 lines each"
        echo "      - Split by responsibility"
        echo "      - Extract shared utilities"
        echo "      - Organize into modules"
    fi
    echo ""
    echo "   5. Test after each extraction"
    echo "   6. Only then resume feature work"
    echo ""
    echo "🎓 Research Evidence:"
    echo "   'Large files correlate with more bugs' (Microsoft Study)"
    echo "   'Files >300 LOC have 2x defect density' (Industry Data)"
    echo ""
    echo "💰 Cost Analysis:"
    echo "   Refactor now: 2-4 hours"
    echo "   Fix bugs later: 10-20 hours + user frustration"
    echo ""
    echo "✅ Refactoring is an investment, not a cost!"
    exit 1
fi
