@echo off
REM Validation script for feature-orchestrator incremental implementation (Windows)
REM Based on CI/CD best practices and research from Google, Microsoft
REM
REM Research Citation:
REM "Small problems are easier to fix than big problems, and frequent commits
REM make bugs easier to identify because there's less code to sort through."
REM — Harness.io, Continuous Integration Best Practices (2024)

setlocal enabledelayedexpansion

echo ===============================================================================
echo 🚀 Feature Orchestrator - Increment Validation
echo ===============================================================================
echo.

set ALL_PASSED=1

REM ============================================================================
REM Step 1: Linting
REM ============================================================================
echo 🔍 Step 1/3: Linting code...
echo    Purpose: Ensure code style consistency
echo    Research: Consistent style reduces cognitive load (Google Research)
echo.

call npm run lint >nul 2>&1
if %ERRORLEVEL% EQU 0 (
    echo ✅ PASSED
    echo    ℹ️  All linting rules passed
) else (
    echo ❌ FAILED
    echo    ⚠️  Run 'npm run lint' to see details
    echo    💡 Tip: Run 'npm run lint:fix' to auto-fix issues
    set ALL_PASSED=0
)

echo.

REM ============================================================================
REM Step 2: Type Checking
REM ============================================================================
echo 🔧 Step 2/3: Type checking...
echo    Purpose: Catch type errors before runtime
echo    Research: Type safety reduces bugs by 15%% (Microsoft Study)
echo.

call npm run type-check >nul 2>&1
if %ERRORLEVEL% EQU 0 (
    echo ✅ PASSED
    echo    ℹ️  No type errors found
) else (
    echo ❌ FAILED
    echo    ⚠️  Run 'npm run type-check' to see details
    echo    💡 Tip: Check TypeScript configuration (tsconfig.json)
    set ALL_PASSED=0
)

echo.

REM ============================================================================
REM Step 3: Tests
REM ============================================================================
echo 🧪 Step 3/3: Running tests...
echo    Purpose: Verify functionality and prevent regressions
echo    Research: Test after each increment catches bugs early (CI Research)
echo.

REM Check if test script exists
npm run test --version >nul 2>&1
if %ERRORLEVEL% EQU 0 (
    call npm test >nul 2>&1
    if !ERRORLEVEL! EQU 0 (
        echo ✅ PASSED
        echo    ℹ️  All tests passed
    ) else (
        echo ❌ FAILED
        echo    ⚠️  Run 'npm test' to see details
        echo    💡 Tip: Fix failing tests before proceeding to next increment
        set ALL_PASSED=0
    )
) else (
    echo ⚠️  SKIPPED
    echo    ℹ️  No test script found (check package.json)
    echo    💡 Tip: Add tests for better code quality
)

echo.
echo ===============================================================================

REM ============================================================================
REM Final Summary
REM ============================================================================
if %ALL_PASSED% EQU 1 (
    echo 🎉 SUCCESS: All validation checks passed!
    echo.
    echo ✅ You can proceed to the next increment
    echo.
    echo 📋 Next steps:
    echo    1. Mark current increment as 'completed' in TodoWrite
    echo    2. Commit changes (if appropriate)
    echo    3. Update next increment to 'in_progress'
    echo    4. Begin work on next increment
    echo.
    exit /b 0
) else (
    echo ❌ FAILURE: Some validation checks failed
    echo.
    echo ⚠️  DO NOT proceed to next increment
    echo.
    echo 📋 Required actions:
    echo    1. Review failed checks above
    echo    2. Fix all issues
    echo    3. Re-run this validation script
    echo    4. Only proceed after ALL checks pass
    echo.
    echo 🎓 Research-backed insight:
    echo    'Small problems are easier to fix than big problems.'
    echo    Fix issues now to save hours of debugging later!
    echo.
    exit /b 1
)
