# Feature Orchestrator - Complete Workflow Examples

This file contains detailed examples of the feature-orchestrator skill in action.

---

## Complete Workflow Example: User Authentication System

**User Request:** "Implement user authentication system"

### Phase 1: Gap Analysis (2-3 minutes)

**AI Response:**
```markdown
Starting gap analysis for authentication system...

🔍 Searching for existing implementations:
- Searching for "auth", "authentication", "login"...
- Checking components/services/utils...

📋 Gap Analysis Results:

Existing Code Found:
- None found in src/components/
- None found in src/services/
- No authentication hooks found

Reusable Libraries Available:
- NextAuth.js (recommended for Next.js projects)
- Supabase Auth (if using Supabase)
- Custom JWT approach

Genuine Gap Identified:
✅ No authentication system exists
✅ Need to implement from scratch

Recommended Approach:
Use NextAuth.js for production-ready auth with:
- Multiple providers (Google, GitHub, Email)
- Session management
- CSRF protection
- Built-in security

Shall I proceed with NextAuth.js implementation plan?
```

**User:** "Yes, proceed"

---

### Phase 2: Implementation Planning (3-5 minutes)

**AI Response:**
```markdown
Creating implementation plan...

## Implementation Plan: NextAuth.js Authentication

### Objective
Implement secure authentication system with email/password and OAuth providers

### Technical Approach
- NextAuth.js v5 with App Router
- Session-based authentication
- Prisma for user storage (optional: use existing DB)
- Protected routes with middleware

### Implementation Steps

1. [ ] Install dependencies and configure NextAuth (~20 min)
   - Install next-auth, bcrypt, types
   - Create auth.ts config
   - Set up environment variables
   - Test: Server starts without errors

2. [ ] Create auth API routes (~30 min)
   - Create /api/auth/[...nextauth]/route.ts
   - Configure providers (Email, Google)
   - Set up session strategy
   - Test: Auth routes respond correctly

3. [ ] Implement login/signup UI (~45 min)
   - Create LoginForm component
   - Create SignupForm component
   - Add form validation
   - Add loading states
   - Test: Forms submit correctly

4. [ ] Add session management (~30 min)
   - Create useAuth hook
   - Add SessionProvider to layout
   - Create session context
   - Test: Session persists across pages

5. [ ] Protect routes with middleware (~20 min)
   - Create middleware.ts
   - Define protected routes
   - Add redirect logic
   - Test: Protected routes require auth

6. [ ] Add logout functionality (~15 min)
   - Create logout button
   - Handle session cleanup
   - Redirect to home
   - Test: Logout clears session

### Testing Strategy
- Unit tests: Form validation, hooks
- Integration tests: Auth flow (signup → login → logout)
- E2E tests: Complete user journey
- Security: CSRF tokens, session expiry

### Performance Considerations
- React.memo() for auth forms (prevent re-renders)
- useMemo() for session context value
- Dynamic import for auth components

### Rollback Plan
- Feature flag: NEXT_PUBLIC_ENABLE_AUTH
- Keep existing public access as fallback
- Document migration path

Estimated Total Time: ~2.5 hours

Creating TodoWrite tracking...
✅ 6 todos created

Would you like me to proceed with implementation?
```

**User:** "Yes"

---

### Phase 3: Review Gate (1-2 minutes)

**AI Response:**
```markdown
Checking if agent review needed...

Feature Type: Authentication system
- Complex logic: ✅ Yes (>100 lines)
- UI components: ✅ Yes (forms)
- Security critical: ✅ Yes

Decision: Invoke both agents in parallel

Invoking:
1. critic-agent → Security review, code quality
2. ui-ux-designer → Form UX, accessibility

[Agent reviews run in parallel...]

📋 Review Results:

Critic Agent:
✅ Use next-auth (industry standard)
⚠️ Add rate limiting on auth endpoints
✅ Use httpOnly cookies for session
⚠️ Add CSRF token validation
✅ Hash passwords with bcrypt (rounds=12)

UI/UX Designer:
✅ Forms in left-side for visibility (80/20 rule)
⚠️ Add clear error messages (not generic "Login failed")
✅ Show password strength indicator
⚠️ Support keyboard navigation (tab order)
✅ Add "Remember me" checkbox

Updating plan with feedback...
✅ Plan updated with 2 additional steps (rate limiting, CSRF)

Proceeding with implementation...
```

---

### Phase 4: Incremental Execution (2.5 hours)

**AI Response (Per Step):**
```markdown
📝 Step 1/6: Install dependencies and configure NextAuth

[in_progress] Install dependencies and configure NextAuth

Running: npm install next-auth bcrypt @types/bcrypt
✅ Dependencies installed

Creating: src/auth.ts
✅ Auth config created

Creating: .env.local.example
✅ Environment template added

Testing increment:
✅ npm run lint → PASSED
✅ npm run type-check → PASSED
✅ npm run dev → Server starts successfully

[completed] Install dependencies and configure NextAuth

---

📝 Step 2/6: Create auth API routes

[in_progress] Create auth API routes

Creating: src/app/api/auth/[...nextauth]/route.ts
✅ API route created with providers

Testing increment:
✅ npm run lint → PASSED
✅ npm run type-check → PASSED
✅ Manual test: http://localhost:3000/api/auth/signin → ✅ Auth page loads

[completed] Create auth API routes

---

[Continues for all 6 steps...]

---

✅ All steps completed!

Final verification:
✅ All todos marked complete
✅ All tests passing
✅ Auth flow working end-to-end
✅ Protected routes enforcing authentication
✅ No breaking changes to existing features

Authentication system implementation complete! 🎉

Would you like me to:
1. Create documentation for the auth system?
2. Add E2E tests for auth flows?
3. Set up monitoring for auth events?
```

---

## Example 2: Simple Feature - Add Dark Mode Toggle

**User Request:** "Add dark mode toggle to settings"

### Phase 1: Gap Analysis (1 minute)

```markdown
🔍 Searching for existing dark mode implementations...

Found:
- src/components/ThemeToggle.tsx (already exists!)
- src/context/ThemeContext.tsx (theme system in place)
- Tailwind dark mode configured (class-based)

✅ Dark mode infrastructure exists
⚠️ Not visible in settings page

Recommendation:
- Reuse existing ThemeToggle component
- Add to settings page (no new code needed)
- Estimated time: 5 minutes

Proceed?
```

**User:** "Yes"

### Phase 2: Implementation Planning (1 minute)

```markdown
## Implementation Plan: Add Dark Mode to Settings

### Objective
Make existing dark mode toggle visible in settings page

### Technical Approach
- Import existing ThemeToggle component
- Add to settings page UI
- No new code required (reuse existing)

### Implementation Steps
1. [ ] Import ThemeToggle into settings page (~2 min)
2. [ ] Add to settings UI with label (~2 min)
3. [ ] Test dark mode switching (~1 min)

Estimated Total Time: 5 minutes

Should I proceed?
```

**User:** "Yes"

### Phase 3: Review Gate

```markdown
Feature Type: Simple UI addition
- Complex logic: ❌ No (just imports)
- UI components: ✅ Yes (but reusing existing)
- Lines of code: ~5 lines

Decision: Skip agent review (too simple, reusing existing component)

Proceeding with implementation...
```

### Phase 4: Incremental Execution (5 minutes)

```markdown
📝 Step 1/3: Import ThemeToggle

[in_progress] Import ThemeToggle into settings page

Modified: src/app/settings/page.tsx
- Added import: ThemeToggle
✅ Import added

Testing:
✅ npm run lint → PASSED
✅ npm run type-check → PASSED

[completed] Import ThemeToggle into settings page

---

📝 Step 2/3: Add to settings UI

[in_progress] Add to settings UI with label

Modified: src/app/settings/page.tsx
- Added section: "Appearance"
- Added ThemeToggle component with label
✅ UI updated

Testing:
✅ npm run lint → PASSED
✅ npm run type-check → PASSED
✅ Manual test: Settings page renders correctly

[completed] Add to settings UI with label

---

📝 Step 3/3: Test dark mode switching

[in_progress] Test dark mode switching

Manual testing:
✅ Toggle switches theme
✅ Preference persists across page loads
✅ No console errors
✅ Smooth transition animation

[completed] Test dark mode switching

---

✅ All steps completed!

Dark mode toggle now visible in settings!
Total time: 5 minutes (as estimated)
```

---

## Example 3: When Gap Analysis Prevents Redundancy

**User Request:** "Add a loading spinner component"

### Phase 1: Gap Analysis (1 minute)

```markdown
🔍 Searching for existing loading implementations...

Found:
- src/components/ui/loading.tsx
  - LoadingSpinner component (already exists!)
  - LoadingDots component
  - LoadingSkeleton component
  - 3 spinner variants (default, small, large)

📋 Gap Analysis Results:

Existing Code:
✅ LoadingSpinner component exists with multiple sizes
✅ Already implements accessibility (aria-label)
✅ Already supports className prop for customization
✅ Used in 15 places across codebase

Recommendation:
❌ DO NOT create duplicate component
✅ Reuse existing: import { LoadingSpinner } from '@/components/ui/loading'

Where do you want to use the spinner?
I'll show you how to import the existing one.
```

**User:** "I want to add it to the profile page while data loads"

```markdown
Perfect! Here's how to use the existing LoadingSpinner:

## Implementation Plan: Add Loading State to Profile

### Approach
- Import existing LoadingSpinner component
- Add loading state to profile page
- Show spinner while fetching user data

### Steps
1. [ ] Import LoadingSpinner (~1 min)
2. [ ] Add loading state (~2 min)
3. [ ] Test loading experience (~2 min)

Estimated Total Time: 5 minutes

Proceed?
```

**Result:** Prevented duplicate component creation, reused existing code!

---

## Example 4: When Review Gate Catches Issues

**User Request:** "Add user profile editing with image upload"

### Phases 1-2: Gap Analysis & Planning (Completed)

### Phase 3: Review Gate

```markdown
Checking if agent review needed...

Feature Type: User profile editing
- Complex logic: ✅ Yes (file upload, validation)
- UI components: ✅ Yes (forms, image preview)
- Security critical: ✅ Yes (file uploads)

Decision: Invoke both agents in parallel

Invoking:
1. critic-agent → Security review, code quality
2. ui-ux-designer → Form UX, accessibility

[Reviews complete...]

📋 Review Results:

Critic Agent:
🚨 SECURITY ISSUES FOUND:
❌ No file type validation (allows any file upload)
❌ No file size limit (DoS risk)
❌ Direct upload to public folder (security risk)
❌ No virus scanning mentioned

REQUIRED CHANGES:
1. Add whitelist: ['image/jpeg', 'image/png', 'image/webp']
2. Max size: 5MB
3. Upload to secure storage (S3/CloudFlare) not public folder
4. Generate signed URLs for access
5. Add image optimization (resize, compress)

UI/UX Designer:
⚠️ Image upload without preview (users can't verify)
⚠️ No crop/zoom functionality (profile pictures need framing)
✅ Form layout good

UPDATING PLAN...

Added steps:
- Step 2b: Add file validation (type, size)
- Step 3b: Configure S3 storage
- Step 4b: Add image preview and crop
- Step 5b: Add image optimization

Original: 4 steps, 1.5 hours
Updated: 8 steps, 3 hours

This review just prevented a CRITICAL SECURITY VULNERABILITY!

Proceed with updated plan?
```

**Result:** Agent review caught security issues before implementation!

---

## Key Takeaways from Examples

1. **Gap Analysis Prevents Redundancy**
   - Example 3 shows how searching first prevents duplicate components
   - Saves time and maintains consistency

2. **Planning Scales with Complexity**
   - Simple features: 3 steps, 5 minutes (dark mode)
   - Complex features: 6+ steps, 2+ hours (auth system)

3. **Review Gate is Conditional**
   - Simple additions: Skip review (dark mode)
   - Security-critical: Always review (auth, file upload)

4. **Incremental Execution Keeps Build Working**
   - Every step is tested before proceeding
   - Can stop at any increment if issues arise
   - Each commit is deployable

5. **Reuse Over Reinvent**
   - Example 2: Reused ThemeToggle (5 min)
   - Example 3: Prevented duplicate LoadingSpinner
   - Always search first!

6. **Agent Reviews Catch Issues Early**
   - Example 4: Security vulnerabilities caught before coding
   - Cheaper to fix in planning than in production

---

## When to Use Each Phase

### Always Use All 4 Phases:
- ✅ New feature implementation
- ✅ Complex components (>100 lines)
- ✅ Security-critical features
- ✅ Database schema changes
- ✅ API endpoint creation

### Can Skip Phase 3 (Review):
- ✅ Simple UI additions (<50 lines)
- ✅ Reusing existing components
- ✅ Configuration changes
- ✅ Content/text updates

### Never Skip Phases 1, 2, 4:
- Gap analysis always required (prevent redundancy)
- Planning always required (incremental approach)
- Incremental execution always required (never break build)

---

## Success Metrics

After using this workflow, you should see:

- ✅ Zero duplicate implementations
- ✅ All features have implementation plans
- ✅ No surprise breaking changes
- ✅ Test pass rate >95% on first try
- ✅ Code review feedback declining
- ✅ Component sizes within limits
- ✅ Feature delivery time predictable

If you're not seeing these results, the workflow isn't being followed correctly.
