#!/bin/bash
# Validation script for feature-orchestrator incremental implementation
# Based on CI/CD best practices and research from Google, Microsoft
#
# Research Citation:
# "Small problems are easier to fix than big problems, and frequent commits
# make bugs easier to identify because there's less code to sort through."
# — Harness.io, Continuous Integration Best Practices (2024)

set -e  # Exit on first error

echo "🚀 Feature Orchestrator - Increment Validation"
echo "================================================"
echo ""

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Track overall status
ALL_PASSED=true

# Function to print status
print_status() {
    if [ $1 -eq 0 ]; then
        echo -e "${GREEN}✅ PASSED${NC}"
    else
        echo -e "${RED}❌ FAILED${NC}"
        ALL_PASSED=false
    fi
}

echo "📋 Running validation suite..."
echo ""

# ============================================================================
# Step 1: Linting
# ============================================================================
echo "🔍 Step 1/3: Linting code..."
echo "   Purpose: Ensure code style consistency"
echo "   Research: Consistent style reduces cognitive load (Google Research)"
echo ""

if npm run lint > /dev/null 2>&1; then
    print_status 0
    echo "   ℹ️  All linting rules passed"
else
    print_status 1
    echo "   ⚠️  Run 'npm run lint' to see details"
    echo "   💡 Tip: Run 'npm run lint:fix' to auto-fix issues"
fi

echo ""

# ============================================================================
# Step 2: Type Checking
# ============================================================================
echo "🔧 Step 2/3: Type checking..."
echo "   Purpose: Catch type errors before runtime"
echo "   Research: Type safety reduces bugs by 15% (Microsoft Study)"
echo ""

if npm run type-check > /dev/null 2>&1; then
    print_status 0
    echo "   ℹ️  No type errors found"
else
    print_status 1
    echo "   ⚠️  Run 'npm run type-check' to see details"
    echo "   💡 Tip: Check TypeScript configuration (tsconfig.json)"
fi

echo ""

# ============================================================================
# Step 3: Tests
# ============================================================================
echo "🧪 Step 3/3: Running tests..."
echo "   Purpose: Verify functionality and prevent regressions"
echo "   Research: Test after each increment catches bugs early (CI Research)"
echo ""

# Check if test script exists
if npm run test --dry-run > /dev/null 2>&1; then
    if npm test > /dev/null 2>&1; then
        print_status 0
        echo "   ℹ️  All tests passed"
    else
        print_status 1
        echo "   ⚠️  Run 'npm test' to see details"
        echo "   💡 Tip: Fix failing tests before proceeding to next increment"
    fi
else
    echo -e "${YELLOW}⚠️  SKIPPED${NC}"
    echo "   ℹ️  No test script found (check package.json)"
    echo "   💡 Tip: Add tests for better code quality"
fi

echo ""
echo "================================================"

# ============================================================================
# Final Summary
# ============================================================================
if [ "$ALL_PASSED" = true ]; then
    echo -e "${GREEN}🎉 SUCCESS: All validation checks passed!${NC}"
    echo ""
    echo "✅ You can proceed to the next increment"
    echo ""
    echo "📋 Next steps:"
    echo "   1. Mark current increment as 'completed' in TodoWrite"
    echo "   2. Commit changes (if appropriate)"
    echo "   3. Update next increment to 'in_progress'"
    echo "   4. Begin work on next increment"
    echo ""
    exit 0
else
    echo -e "${RED}❌ FAILURE: Some validation checks failed${NC}"
    echo ""
    echo "⚠️  DO NOT proceed to next increment"
    echo ""
    echo "📋 Required actions:"
    echo "   1. Review failed checks above"
    echo "   2. Fix all issues"
    echo "   3. Re-run this validation script"
    echo "   4. Only proceed after ALL checks pass"
    echo ""
    echo "🎓 Research-backed insight:"
    echo "   'Small problems are easier to fix than big problems.'"
    echo "   Fix issues now to save hours of debugging later!"
    echo ""
    exit 1
fi
