# Feature Orchestrator - v2.0

**Research-backed feature implementation workflow for Claude Code**

[![Version](https://img.shields.io/badge/version-2.0-blue.svg)](https://github.com/mgbotoe/claude-code-share)
[![License](https://img.shields.io/badge/license-CC%20BY%204.0-green.svg)](https://creativecommons.org/licenses/by/4.0/)
[![Research-Backed](https://img.shields.io/badge/research-backed-brightgreen.svg)](#research-foundation)

> Enforce CI/CD best practices, prevent code duplication, ensure incremental delivery based on Google, Microsoft, and IEEE research.

---

## 📋 Table of Contents

- [Overview](#overview)
- [What's New in v2.0](#whats-new-in-v20)
- [Quick Start](#quick-start)
- [Research Foundation](#research-foundation)
- [File Structure](#file-structure)
- [Usage](#usage)
- [Templates](#templates)
- [Automation Scripts](#automation-scripts)
- [Integration](#integration)
- [Examples](#examples)
- [Contributing](#contributing)
- [License](#license)

---

## Overview

The Feature Orchestrator skill enforces a research-backed, four-phase workflow for implementing features in software projects:

1. **Gap Analysis** - Search for existing code before writing new code
2. **Implementation Planning** - Break features into incremental, testable steps
3. **Review Gate** - Invoke AI agents for quality/design review
4. **Incremental Execution** - Build in small increments, testing after each

### The Problem It Solves

Common development issues:
- ❌ Unknowingly duplicating existing code
- ❌ Implementing features without a plan
- ❌ Large commits that break the build
- ❌ Discovering oversized files "by accident"
- ❌ Skipping code review until it's too late

### The Solution

✅ Search for existing code FIRST (prevents duplication)
✅ Plan before coding (ensures incremental delivery)
✅ Agent review for complex features (catches issues early)
✅ Test after each increment (never break the build)
✅ Monitor file sizes proactively (prevents complexity)

---

## What's New in v2.0

### Major Improvements

#### 🔬 Research-Backed Practices
- Every recommendation backed by Google, Microsoft, or IEEE research
- Direct quotes from academic studies
- Measurable metrics and thresholds

#### ⚡ Progressive Disclosure (57% Faster)
- Main skill.md reduced from 5,800 to 2,500 tokens
- Details loaded on-demand from supporting files
- 2.3x faster initial loading

#### 📋 Reusable Templates
- `gap-analysis-template.md` - Structured search process
- `implementation-plan-template.md` - Complete planning framework
- `increment-checklist-template.md` - Per-increment verification

#### 🤖 Automation Scripts
- `validate-increment.sh/bat` - One-command validation
- `check-file-size.sh` - Proactive complexity monitoring
- Research citations included in output

### Comparison

| Metric | v1.0 | v2.0 | Improvement |
|--------|------|------|-------------|
| Token load | 5,800 | 2,500 | 57% reduction |
| Research sources | 0 | 15+ | Added |
| Templates | 0 | 3 | Added |
| Automation | 0 | 3 scripts | Added |

---

## Quick Start

### Installation

This skill is already installed if you're using it! It's located at:
```
~/.claude/plugins/marketplaces/custom-skills/feature-orchestrator/
```

### Basic Usage

The skill auto-invokes when you request implementing a feature:

```
User: "implement user authentication"
Claude: [Automatically invokes feature-orchestrator skill]
        Phase 1: Searching for existing authentication code...
        Phase 2: Creating implementation plan...
        Phase 3: Invoking code review agents...
        Phase 4: Building incrementally with testing...
```

### Manual Invocation

You can also explicitly request it:
```
User: "Use feature-orchestrator to help me add dark mode"
```

---

## Research Foundation

This workflow is backed by industry research:

### Academic Sources

**Google Research (2013):**
> "We conducted an exploratory investigation of modern code review through 12 interviews, a survey with 44 respondents, and the analysis of review logs for 9 million reviewed changes."

**Microsoft Research (2013):**
> "Reviewing 200–400 lines of code (LOC) at a time detects up to 90% of defects, with defect detection rates dropping beyond this amount."

**IEEE / ResearchGate (2024):**
> "Research shows that CI leads to higher levels of throughput, more stable systems, and higher quality software."

### Industry Standards

- **SonarQube:** < 5% code duplication target
- **Atlassian:** Trunk-based development patterns
- **Harness.io:** CI/CD best practices (2024)
- **Scrum.org:** Incremental delivery research (2024)

**For complete citations:** See `REFERENCE.md`

---

## File Structure

```
feature-orchestrator/
├── README.md                               # This file
├── skill.md                                # Main skill (2,500 tokens)
├── REFERENCE.md                            # Detailed procedures with research
├── EXAMPLES.md                             # Real-world scenarios
├── resources/
│   ├── gap-analysis-template.md           # Search & reuse template
│   ├── implementation-plan-template.md    # Planning framework
│   └── increment-checklist-template.md    # Per-increment verification
└── scripts/
    ├── validate-increment.sh              # Validation (Linux/Mac)
    ├── validate-increment.bat             # Validation (Windows)
    └── check-file-size.sh                 # File size monitoring
```

### Progressive Loading

```
Initial Load:    skill.md (2,500 tokens)
On-Demand:       REFERENCE.md (~6,000 tokens)
                 EXAMPLES.md (~3,500 tokens)
                 Templates (~2,000 tokens each)
```

---

## Usage

### The Four-Phase Workflow

#### Phase 1: Gap Analysis (2-3 minutes)

**Purpose:** Search for existing code before writing new code

**Process:**
```bash
# Search for similar implementations
glob "**/*[keyword]*.tsx"
grep "[keyword]" --output_mode files_with_matches

# If found: Plan to reuse/extend
# If not found: Create new with plan
```

**Template:** `resources/gap-analysis-template.md`

**Research:** DRY principle - < 5% duplication target (SonarQube)

---

#### Phase 2: Implementation Planning (3-5 minutes)

**Purpose:** Break feature into incremental, testable steps

**Required Elements:**
- Objective (what, why, success criteria)
- Technical approach (architecture, data flow)
- Incremental steps (30-60 min each, < 100 LOC each)
- Testing strategy (unit, integration, E2E)
- Performance considerations (React.memo(), etc.)
- Rollback plan (feature flags, gradual rollout)

**Template:** `resources/implementation-plan-template.md`

**Research:** Agile INVEST criteria - Small, Testable steps

---

#### Phase 3: Review Gate (1-2 minutes, conditional)

**Purpose:** Invoke AI agents for quality/design review

**Decision Matrix:**
| Criteria | Threshold | Action |
|----------|-----------|--------|
| Lines of code | > 100 | Invoke critic-agent |
| Security critical | Auth, payments | Always review |
| User-facing UI | Any | Invoke ui-ux-designer |
| Simple addition | < 50 lines | Skip review |

**Research:** Google code reviews < 4 hour median latency

---

#### Phase 4: Incremental Execution (varies)

**Purpose:** Implement in small, testable increments

**For EACH increment:**
```markdown
1. Mark todo as "in_progress"
2. Implement (< 100 lines)
3. Test (ALL must pass):
   - npm run lint
   - npm run type-check
   - npm run test
4. Mark todo as "completed"
5. Commit (if appropriate)
6. Only proceed if tests pass
```

**Template:** `resources/increment-checklist-template.md`

**Research:** CI with frequent commits reduces issues (ResearchGate)

---

## Templates

### gap-analysis-template.md

**Use when:** Starting gap analysis (Phase 1)

**Contents:**
- Search strategy
- Results documentation
- Reusability assessment
- DRY principle compliance
- User confirmation

**How to use:**
```bash
# Copy template
cp resources/gap-analysis-template.md gap-analysis-[feature-name].md

# Fill in sections
# - Search keywords
# - Search results
# - Reuse opportunities
# - Recommendations
```

---

### implementation-plan-template.md

**Use when:** Planning implementation (Phase 2)

**Contents:**
- Objective & success criteria
- Technical approach
- Incremental steps (detailed)
- Testing strategy
- Performance considerations
- Security checklist
- Rollback plan
- Time estimation
- Risk assessment

**How to use:**
```bash
# Copy template
cp resources/implementation-plan-template.md plan-[feature-name].md

# Fill in all sections
# - Define objective
# - Break into steps
# - Add test strategy
# - Include rollback plan
```

---

### increment-checklist-template.md

**Use when:** Completing each increment (Phase 4)

**Contents:**
- Pre-implementation checks
- Code quality verification
- Testing requirements
- Accessibility checks
- Commit guidelines
- Post-implementation cleanup
- Decision point (proceed or fix)

**How to use:**
```bash
# Use for each increment
# Check off items as you complete them
# Verify ALL pass before proceeding
```

---

## Automation Scripts

### validate-increment.sh / .bat

**Purpose:** Automate the 3-step validation process

**What it does:**
```bash
1. npm run lint        # Code style
2. npm run type-check  # Type safety
3. npm test           # Functionality
```

**Usage:**
```bash
# Linux/Mac
./scripts/validate-increment.sh

# Windows
./scripts/validate-increment.bat
```

**Output:**
```
🚀 Feature Orchestrator - Increment Validation
================================================

🔍 Step 1/3: Linting code...
✅ PASSED

🔧 Step 2/3: Type checking...
✅ PASSED

🧪 Step 3/3: Running tests...
✅ PASSED

🎉 SUCCESS: All validation checks passed!

✅ You can proceed to the next increment
```

**Features:**
- ✅ Research citations in output
- ✅ Clear pass/fail status
- ✅ Actionable recommendations
- ✅ Exit codes for CI/CD integration

---

### check-file-size.sh

**Purpose:** Proactively monitor file complexity

**What it does:**
- Analyzes file line count
- Determines status (Good, Warning, Alert, Critical)
- Provides context-aware recommendations
- Cites research on file size vs. bugs

**Usage:**
```bash
./scripts/check-file-size.sh src/components/UserProfile.tsx
```

**Output:**
```
================================================
📏 File Size Analysis
================================================

📄 File: src/components/UserProfile.tsx
📊 Lines: 285
🏷️  Type: tsx
📁 Category: React Component
🎯 Ideal: < 200 lines
⚠️  Warning: 300 lines
🛑 Critical: 400 lines

⚠️  STATUS: WARNING
   File is getting large, plan extraction

📋 Recommended Actions:
   1. Identify extraction candidates:
      - Large data arrays (>20 lines) → data file
      - Sub-components (modals, forms)
      - Custom hooks (4+ useState)

   2. Before next addition, extract first
   3. Keep additions small (< 100 lines)

🎓 Research Insight:
   'Code reviews of 200-400 LOC detect 90% of defects'
   Keeping files smaller = easier to review & maintain
```

**Thresholds:**
- **React components:** Ideal < 200, Warning 300, Critical 400
- **Services/utilities:** Ideal < 250, Warning 400, Critical 500
- **Python modules:** Ideal < 250, Warning 400, Critical 500

---

## Integration

### With Other Skills

**code-refactoring:**
- Triggered when files exceed size limits
- Provides extraction recommendations
- Coordinates incremental refactoring

**ui-ux-audit:**
- Invoked during Phase 3 for UI features
- Provides design critique
- Checks accessibility compliance

**devops-deployment:**
- Used after Phase 4 completion
- Deployment checklists
- Production readiness verification

**qa-testing:**
- Referenced in Phase 2 for test strategy
- Comprehensive test coverage
- Test automation guidance

### With CLAUDE.md

This skill enforces ALL mandatory CLAUDE.md rules:

1. ✅ Search for existing code FIRST
2. ✅ Plan before implementing
3. ✅ Incremental implementation
4. ✅ Test between increments
5. ✅ React.memo() for performance
6. ✅ Coordinate agents
7. ✅ TodoWrite tracking
8. ✅ Component size limits

---

## Examples

### Example 1: Complete Auth System (2.5 hours)

```
User: "Implement user authentication system"

Phase 1: Gap Analysis (3 min)
- Searched for existing auth code
- Found: Nothing
- Recommendation: Use NextAuth.js

Phase 2: Planning (5 min)
- Created 6-step plan
- Each step 15-45 min
- Total estimate: 2.5 hours
- Created TodoWrite tracking

Phase 3: Review Gate (2 min)
- Invoked critic-agent (security review)
- Invoked ui-ux-designer (form UX)
- Incorporated feedback: Added rate limiting, CSRF

Phase 4: Execution (2.5 hours)
- Step 1: Install deps & config (20 min) ✅
- Step 2: Auth API routes (30 min) ✅
- Step 3: Login/Signup UI (45 min) ✅
- Step 4: Session management (30 min) ✅
- Step 5: Protected routes (20 min) ✅
- Step 6: Logout functionality (15 min) ✅

Result: Auth system complete, all tests passing!
```

**See `EXAMPLES.md` for complete walkthrough**

---

### Example 2: Simple Feature (5 minutes)

```
User: "Add dark mode toggle to settings"

Phase 1: Gap Analysis (1 min)
- Found: ThemeToggle component already exists!
- Recommendation: Reuse existing component

Phase 2: Planning (1 min)
- 3 simple steps
- Total estimate: 5 minutes

Phase 3: Review Gate
- Skipped (< 50 lines, reusing existing)

Phase 4: Execution (5 min)
- Step 1: Import ThemeToggle (2 min) ✅
- Step 2: Add to settings UI (2 min) ✅
- Step 3: Test functionality (1 min) ✅

Result: Dark mode toggle in settings, 5 minutes!
Gap analysis saved hours by reusing existing component.
```

---

### Example 3: Preventing Duplication

```
User: "Add a loading spinner component"

Phase 1: Gap Analysis (1 min)
- Searched for "loading", "spinner"
- Found: LoadingSpinner already exists!
  - Multiple sizes (small, default, large)
  - Accessibility built-in
  - Used in 15 places

Recommendation: DO NOT create duplicate!
- Reuse: import { LoadingSpinner } from '@/components/ui/loading'

User: "Oh, I didn't know that existed! Yes, let's use that."

Result: Duplication prevented, consistency maintained!
```

**See `EXAMPLES.md` for more scenarios**

---

## Contributing

### How to Contribute

1. **Fork the repository** on GitHub
2. **Create a feature branch** (`git checkout -b feature/improvement`)
3. **Make your changes** following the structure
4. **Test thoroughly** (use the skill on real features)
5. **Submit a pull request** with detailed description

### Contribution Guidelines

- **Research-backed changes only** - Cite sources
- **Maintain progressive disclosure** - Keep main skill.md < 3,000 tokens
- **Add examples** - Real-world scenarios in EXAMPLES.md
- **Update templates** - Keep templates comprehensive
- **Test automation** - Scripts must work on Linux/Mac/Windows

### Reporting Issues

Found a bug or have a suggestion?

1. Check existing issues first
2. Create new issue with:
   - Clear description
   - Steps to reproduce (if bug)
   - Expected vs. actual behavior
   - Your environment (OS, Claude version)

---

## License

**Creative Commons Attribution 4.0 International (CC BY 4.0)**

You are free to:
- **Share** - Copy and redistribute the material
- **Adapt** - Remix, transform, and build upon the material

Under the following terms:
- **Attribution** - You must give appropriate credit, provide a link to the license, and indicate if changes were made

**Attribution Required:**
```
Feature Orchestrator Skill by Madina Gbotoe (https://madinagbotoe.com/)
GitHub: https://github.com/mgbotoe/claude-code-share
```

Full license: https://creativecommons.org/licenses/by/4.0/

---

## Credits

**Created by:** Madina Gbotoe
**Website:** https://madinagbotoe.com/
**GitHub:** https://github.com/mgbotoe/claude-code-share
**Project:** AI-Enhanced Professional Portfolio

### Research Sources

**Academic:**
- Google Research: "Modern Code Review" (2013)
- Microsoft Research: "Expectations of Modern Code Review" (2013)
- IEEE: "Continuous Integration Research" (2024)
- Scrum.org: "Incremental Delivery Research" (2024)

**Industry:**
- SonarQube: Code quality standards
- Atlassian: Trunk-based development
- Harness.io: CI/CD best practices (2024)
- MetriDev: Code duplication research (2024)

**Books:**
- "The DevOps Handbook" by Gene Kim, et al.
- "Accelerate" by Nicole Forsgren, et al.
- "Building Maintainable Software" by O'Reilly

---

## Version History

- **v2.0** (Nov 2025): Research-backed edition with progressive disclosure, templates, automation
- **v1.0** (Oct 2025): Initial version based on CLAUDE.md

**For detailed changelog:** See `REFERENCE.md` → Version History

---

## Support

**Documentation:**
- Main workflow: `skill.md`
- Detailed procedures: `REFERENCE.md`
- Examples: `EXAMPLES.md`
- Templates: `resources/` directory

**Questions or Issues:**
- GitHub Issues: https://github.com/mgbotoe/claude-code-share/issues
- Website: https://madinagbotoe.com/

---

## Acknowledgments

Thank you to:
- Google, Microsoft, IEEE for publishing their research
- SonarQube, Atlassian, Harness.io for industry standards
- The Claude Code community for feedback and testing
- All contributors who helped improve this skill

---

**Ready to build better features? Start with gap analysis! 🚀**

**Research-backed. Industry-proven. Battle-tested.**
