# Skill Builder Reference

Comprehensive reference for creating Claude Code Skills.

## Frontmatter Specification

### name (required)
- **Type**: string
- **Max Length**: 64 characters
- **Format**: lowercase letters, numbers, hyphens only
- **Pattern**: `^[a-z0-9-]+$`

**Valid Examples:**
- `excel-automation`
- `code-reviewer`
- `api-doc-generator`
- `test-suite-builder`

**Invalid Examples:**
- `Excel_Automation` (uppercase and underscore)
- `code reviewer` (space)
- `api.doc.generator` (dots)
- `Test-Suite!` (uppercase and special char)

### description (required)
- **Type**: string
- **Min Length**: 1 character
- **Max Length**: 1024 characters
- **Purpose**: Help Claude decide when to invoke the skill

**Good Descriptions (with natural language keywords):**
```yaml
description: Auto-invoked when user wants to fix broken code, not working, crashed, or error message. Also triggers on technical terms like debug, stack trace, error analysis, production incidents.
```

```yaml
description: Auto-invoked when storing data, saving information, organizing data, or database is slow. Also triggers on database design, schema, query performance, migrations.
```

```yaml
description: Auto-invoked when user wants to put app online, make it live, publish site, or launch app. Also triggers on deployment, CI/CD, Docker, Kubernetes, production.
```

**Why This Works:**
- ✅ Natural language: "fix broken code" (how non-technical users speak)
- ✅ Technical terms: "stack trace" (how developers speak)
- ✅ Both audiences can trigger the skill

**Poor Descriptions:**
```yaml
description: Excel stuff
# Too vague, unclear when to use
```

```yaml
description: Handles database schema migrations and query optimization
# Only technical jargon - non-technical users won't trigger this
```

```yaml
description: A comprehensive, enterprise-grade, production-ready solution for automated generation of Excel workbooks with advanced formula support, complex chart creation, multi-sheet management, conditional formatting, data validation, pivot tables, and extensive customization capabilities for business intelligence and data analysis workflows
# Too long, exceeds clarity needs
```

## Natural Language Keywords for Non-Technical Users

### Why This Matters

**CRITICAL INSIGHT**: Not all users are developers. Many are:
- Product managers who say "put this online" not "deploy to production"
- Designers who say "change the colors" not "update design tokens"
- Business users who say "save this data" not "implement database schema"
- Content creators who say "my site crashed" not "analyze stack trace"

**Your skill MUST be accessible to BOTH audiences:**
- ✅ Natural language (how people naturally speak)
- ✅ Technical jargon (how developers speak)

### The Dual-Keyword Strategy

**Template Pattern:**
```yaml
description: Auto-invoked when user wants to [NATURAL LANGUAGE PHRASES]. Also triggers on technical terms like [TECHNICAL JARGON].
```

**Example:**
```yaml
description: Auto-invoked when user wants to fix broken code, not working, crashed, or error message. Also triggers on technical terms like debug, stack trace, error analysis, production incidents.
```

### Natural Language Keyword Categories

#### 1. Problem/Issue Keywords (What's Wrong?)
**Natural**: broken, not working, crashed, error, failed, stuck, won't load, stopped
**Technical**: exception, stack trace, runtime error, null pointer, undefined reference

**Example:**
```yaml
# Debugger skill
description: Auto-invoked when something's broken, not working, error message, crashed, fix this, why isn't working, or app stopped working. Also triggers on debugging, stack trace, production incidents.
```

#### 2. Action/Intent Keywords (What Do I Want?)
**Natural**: save, store, organize, put online, make live, publish, change, update, fix
**Technical**: persist, deploy, migrate, refactor, optimize, synchronize

**Example:**
```yaml
# DevOps skill
description: Auto-invoked when user wants to put app online, make it live, publish site, launch app, or go live. Also triggers on deployment, CI/CD, container orchestration, infrastructure setup.
```

#### 3. Data/Content Keywords (Working with Information)
**Natural**: save data, store information, slow database, where to save, organize data
**Technical**: schema design, query optimization, normalization, indexing, migrations

**Example:**
```yaml
# Database skill
description: Auto-invoked when storing data, saving information, organizing data, database is slow, or where to save data. Also triggers on database design, schema, query performance, migrations.
```

#### 4. Quality/Review Keywords (Is This Good?)
**Natural**: check my code, is this secure, look over code, is this good, find problems
**Technical**: code review, security audit, OWASP, static analysis, vulnerability scan

**Example:**
```yaml
# Code reviewer skill
description: Auto-invoked when user wants to check my code, is this secure, review code, look over code, or is this good. Also triggers on code review, security audit, PR analysis, OWASP.
```

#### 5. Creation/Generation Keywords (Make Something)
**Natural**: write docs, explain code, create guide, document this, make instructions
**Technical**: API documentation, OpenAPI, technical writing, changelog, ADR

**Example:**
```yaml
# Documentation skill
description: Auto-invoked when user wants to write docs, create documentation, explain code, write README, or make user guide. Also triggers on technical documentation, API docs, OpenAPI.
```

#### 6. Improvement Keywords (Make It Better)
**Natural**: too long, simplify, clean up code, make easier to read, organize better
**Technical**: refactor, extract method, reduce complexity, DRY, SOLID principles

**Example:**
```yaml
# Refactoring skill
description: Auto-invoked when file is too long, code needs simplifying, clean up code, organize better, or make code easier to read. Also triggers on refactor, extract, split component.
```

#### 7. Intelligence/AI Keywords (Smart Features)
**Natural**: chatbot, smart search, AI helper, answer questions, intelligent assistant
**Technical**: RAG, LLM integration, vector database, embeddings, semantic search

**Example:**
```yaml
# AI/ML skill
description: Auto-invoked when user wants to add chatbot, smart search, AI helper, answer questions automatically, or intelligent assistant. Also triggers on RAG, vector database, LLM integration.
```

#### 8. Visual/Design Keywords (How It Looks)
**Natural**: change design, new look, redesign, change colors, update fonts, rebrand
**Technical**: design system, design tokens, theme migration, CSS-in-JS, styling

**Example:**
```yaml
# Design system skill
description: Auto-invoked when user wants to change entire design, new look everywhere, redesign everything, change all colors/fonts, or rebrand. Also triggers on design system migration, theme changes.
```

### Common Anti-Patterns to Avoid

#### ❌ WRONG: Technical Jargon Only
```yaml
description: Handles database schema migrations and query optimization using normalization techniques and index strategies
```
**Problem**: Non-technical users won't say "schema migrations" or "normalization"

#### ✅ CORRECT: Natural + Technical
```yaml
description: Auto-invoked when storing data, saving information, organizing data, or database is slow. Also triggers on database design, schema, query performance, migrations.
```

#### ❌ WRONG: Too Vague
```yaml
description: Database stuff and data things
```
**Problem**: Not specific enough for Claude to match user requests

#### ✅ CORRECT: Specific Natural Language
```yaml
description: Auto-invoked when storing data, saving information, organizing data, database is slow, or where to save data.
```

#### ❌ WRONG: Natural Language Only (Missing Technical)
```yaml
description: Auto-invoked when user wants to put stuff online or make it work
```
**Problem**: Developers using technical terms won't trigger the skill

#### ✅ CORRECT: Both Audiences Covered
```yaml
description: Auto-invoked when user wants to put app online, make it live, publish site. Also triggers on deployment, CI/CD, Docker, Kubernetes.
```

### Testing Your Keywords

**Ask yourself:**

1. **Non-Technical User Test**:
   - Would a product manager/designer/business user use these words?
   - Can someone explain their problem WITHOUT technical jargon?

2. **Technical User Test**:
   - Would a developer recognize these technical terms?
   - Are industry-standard terms included?

3. **Coverage Test**:
   - Do keywords cover different ways to express the same need?
   - "save data" vs "store information" vs "organize data"

4. **Specificity Test**:
   - Are keywords specific enough to match the skill's purpose?
   - Avoid overly generic terms like "stuff", "things", "work"

### Validation Scoring

**Natural Language Keywords Check (3 points):**
- **3 points**: 2+ natural language keywords (✅ Accessible to non-technical users)
- **2 points**: 1 natural language keyword (⚠️ Partially accessible)
- **1 point**: Only technical jargon (🚫 Inaccessible to non-technical users)
- **0 points**: No clear keywords

**Passing threshold**: 80% (24/29 points total)

### Real-World Examples from Custom Skills

#### Example 1: ai-ml-implementation
```yaml
description: Research-backed AI/ML feature implementation. Auto-invoked when user wants to add chatbot, smart search, AI helper, answer questions automatically, chat with data, intelligent assistant, AI-powered search, conversational AI, question answering, or any AI features. Also triggers on technical terms like RAG, vector database, embeddings, LLM integration, semantic search, knowledge base.
```
**Natural**: chatbot, smart search, AI helper, answer questions automatically
**Technical**: RAG, vector database, embeddings, LLM integration
**Why It Works**: Both PM ("add a chatbot") and developer ("implement RAG") can trigger it

#### Example 2: debugger
```yaml
description: Production debugging, log analysis, stack trace interpretation, and root cause analysis. Auto-invoked when something's broken, not working, error message, crashed, fix this, why isn't working, bug, something wrong, or app stopped working. Also triggers on technical terms like debugging, stack trace, production incidents, CI/CD failures, error analysis.
```
**Natural**: broken, not working, crashed, fix this, app stopped working
**Technical**: stack trace, production incidents, CI/CD failures
**Why It Works**: User frustration language ("this is broken!") and dev terms ("stack trace") both work

#### Example 3: git-manager
```yaml
description: Git workflow standardization, commit conventions, branch strategies, and PR management. Auto-invoked when user wants to save changes, save work, track changes, commit code, create pull request, or merge code. Also triggers on technical terms like git commit, PR, pull request, branching, merge, version control, Conventional Commits, semantic versioning.
```
**Natural**: save changes, save work, track changes
**Technical**: git commit, PR, Conventional Commits, semantic versioning
**Why It Works**: New developer ("save my work") and experienced dev ("create PR") both trigger it

### Quick Reference Checklist

When writing descriptions:
- [ ] Includes 2+ natural language keywords (how non-technical users speak)
- [ ] Includes 2+ technical keywords (how developers speak)
- [ ] Uses "Auto-invoked when user wants to..." pattern
- [ ] Separates natural and technical with "Also triggers on technical terms like..."
- [ ] Covers different ways to express the same intent
- [ ] Specific enough to match skill's purpose
- [ ] Would pass validation (3 points for natural language keywords)

## File Organization Patterns

### Minimal Skill (Single File)
```
skill-name/
└── SKILL.md
```
Use when: Simple, self-contained instructions with no supporting resources needed.

### Basic Skill (With Reference)
```
skill-name/
├── SKILL.md
└── REFERENCE.md
```
Use when: Instructions need detailed reference info that shouldn't load initially.

### Standard Skill (With Templates)
```
skill-name/
├── SKILL.md
├── REFERENCE.md
└── FORMS.md
```
Use when: Skill generates structured output using templates.

### Advanced Skill (Full Structure)
```
skill-name/
├── SKILL.md
├── REFERENCE.md
├── FORMS.md
├── scripts/
│   ├── process.py
│   └── validate.sh
└── resources/
    ├── templates/
    │   ├── base.txt
    │   └── advanced.txt
    └── data/
        └── reference.json
```
Use when: Complex workflows requiring executable code and multiple resources.

## Progressive Disclosure Details

### Level 1: Metadata (Always Loaded)
**Token Cost**: ~100 tokens per skill
**Content**: YAML frontmatter only
**Purpose**: Skill discovery and matching

```yaml
---
name: example-skill
description: What it does and when to use it
---
```

### Level 2: Instructions (Loaded When Triggered)
**Token Cost**: <5,000 tokens (keep it concise)
**Content**: SKILL.md body
**Purpose**: Core procedures and workflow

Keep instructions focused:
- Core workflow only
- Reference other files for details
- Use examples sparingly
- Link to REFERENCE.md for specs

### Level 3: Resources (Loaded As Needed)
**Token Cost**: 0 (accessed via bash, doesn't consume context)
**Content**: Supporting files
**Purpose**: Detailed info, templates, scripts

Access pattern:
```bash
# Read reference when needed
cat ~/.claude/skills/skill-name/REFERENCE.md

# Load template when formatting output
cat ~/.claude/skills/skill-name/resources/template.txt

# Execute script when processing data
python ~/.claude/skills/skill-name/scripts/process.py
```

## Validation Rules

### Name Validation
```python
import re

def is_valid_skill_name(name):
    if len(name) > 64:
        return False
    if not re.match(r'^[a-z0-9-]+$', name):
        return False
    return True
```

### Description Validation
```python
def is_valid_description(desc):
    if len(desc) == 0 or len(desc) > 1024:
        return False
    return True
```

### Structure Validation
```bash
# Check required files
test -f SKILL.md || echo "Missing SKILL.md"

# Check frontmatter exists
head -1 SKILL.md | grep -q "^---$" || echo "Missing frontmatter"

# Verify name and description in frontmatter
grep -q "^name: " SKILL.md || echo "Missing name field"
grep -q "^description: " SKILL.md || echo "Missing description field"
```

## Runtime Constraints

### No Internet Access
Skills cannot:
- Make HTTP requests
- Access external APIs
- Download resources
- Fetch remote data

Workaround: Pre-include all necessary data in resources/

### No Package Installation
Skills cannot:
- Run `pip install`
- Run `npm install`
- Install system packages
- Download dependencies at runtime

Workaround: Only use pre-installed packages available in Claude Code environment

### Available Pre-installed Packages
Common packages available:
- **Python**: requests, pandas, numpy, json, csv, re, os, sys
- **Node.js**: Basic built-ins
- **Shell**: Standard Unix utilities

Verify availability before relying on specific packages.

## Best Practice Examples

### Good: Progressive Disclosure
```markdown
# Data Analyzer Skill

## Process
1. Read input data
2. For processing rules, see REFERENCE.md section 3.2
3. Apply template from FORMS.md
4. Run scripts/analyze.py if complex processing needed
```

### Bad: Everything Embedded
```markdown
# Data Analyzer Skill

## Process
1. Read input data
2. Apply these 47 detailed rules:
   - Rule 1: [500 words of explanation]
   - Rule 2: [500 words of explanation]
   ...
3. Use this template:
   [Entire template embedded here]
4. Processing algorithm:
   [Entire script code pasted here]
```

### Good: Clear Trigger Conditions
```markdown
## When to Use
Invoke this skill when users:
- Request code review
- Ask to check code for issues
- Want security analysis
- Need best practices validation
```

### Bad: Vague Triggers
```markdown
## When to Use
Use this skill for code stuff.
```

## Example Skills

### Simple: Code Formatter
```markdown
---
name: code-formatter
description: Format code files according to language-specific style guides when user requests code formatting
---

# Code Formatter

## When to Use
- User asks to format code
- User mentions style guide compliance
- Code formatting is requested

## Process
1. Identify language from file extension
2. Apply appropriate formatter (prettier, black, gofmt, etc.)
3. Show diff and apply changes
4. Confirm formatting completed
```

### Medium: Test Generator
```markdown
---
name: test-generator
description: Generate unit tests for code functions when user requests test creation or test coverage improvements
---

# Test Generator

## When to Use
- User requests tests for a function/class
- User wants to improve test coverage
- User mentions unit testing

## Process
1. Analyze target code to understand functionality
2. Identify test cases (see REFERENCE.md for test patterns)
3. Generate test code using template from FORMS.md
4. Explain test coverage

## Supporting Files
- REFERENCE.md: Test patterns and edge cases
- FORMS.md: Test templates for each language
```

### Complex: API Documentation Generator
```markdown
---
name: api-doc-generator
description: Generate comprehensive API documentation from code comments and endpoint definitions for REST APIs
---

# API Documentation Generator

## When to Use
- User requests API documentation
- Need to document REST endpoints
- OpenAPI/Swagger generation requested

## Workflow
1. Scan codebase for API endpoints using scripts/scan-endpoints.sh
2. Extract comments and parameters
3. For documentation format, see REFERENCE.md sections 2-4
4. Generate using templates from FORMS.md
5. Run scripts/validate-docs.py to verify completeness

## File Structure
- scripts/scan-endpoints.sh: Find API endpoints
- scripts/validate-docs.py: Verify documentation
- REFERENCE.md: OpenAPI specification details
- FORMS.md: Documentation templates
- resources/examples/: Sample API docs
```

## Troubleshooting

### Skill Not Triggering
**Possible Causes:**
- Description doesn't match user request
- Name contains invalid characters
- SKILL.md has malformed frontmatter
- File is in wrong directory

**Solutions:**
- Make description more specific to use cases
- Validate name against rules
- Check YAML frontmatter syntax
- Verify file location (`.claude/skills/` or `~/.claude/skills/`)

### Instructions Too Long
**Symptoms:**
- Slow skill loading
- Token budget issues

**Solutions:**
- Move detailed content to REFERENCE.md
- Extract templates to FORMS.md
- Use file references instead of embedding
- Keep SKILL.md under 5k tokens

### Resources Not Loading
**Possible Causes:**
- Incorrect file paths
- Files not in skill directory
- Permission issues

**Solutions:**
- Use absolute paths: `~/.claude/skills/skill-name/file.txt`
- Verify files exist with `ls` command
- Check file permissions
