# Specialized Shader Features

This document details the implementation of specialized shader variants such as Fur, Tessellation, and Gem.

## 1. Fur
Simulates volumetric fur using the "Shell" technique (multi-pass geometry extrusion).

### Properties
*   **_FurVector**: Direction and length of the fur.
*   **_FurGravity**: Gravity influence on fur strands.
*   **_FurLayerNum**: Number of layers (shells) to render.
*   **_FurNoiseMask**: Noise texture defining strand density/shape.
*   **_FurLengthMask**: Mask to control fur length per vertex.
*   **_FurAO**: Ambient occlusion between strands.

### Implementation Call
**Vertex/Geometry Shader**: `Assets/lilToon/Shader/Includes/lil_common_vert_fur.hlsl`
**Fragment Shader**: `Assets/lilToon/Shader/Includes/lil_pass_forward_fur.hlsl`

### Principles & Math
**Geometry Shader Extrusion (Shells)**
The Geometry Shader takes a triangle and emits multiple copies (shells) of it, each displaced further along the normal/fur vector.

**Pseudo-Code (`geom` shader):**
```hlsl
void geom(triangle v2g input[3], inout TriangleStream<v2f> outStream) {
    // Determine number of layers
    int layers = _FurLayerNum;
    
    for (int i = 0; i < layers; i++) {
        float factor = (float)i / (float)layers;
        
        // 1. Interpolate Vertex Data
        v2f output = Interpolate(input);
        
        // 2. Displace Vertex Position
        // Move vertex along Fur Vector
        float3 displacement = FurVector * factor;
        output.positionWS += displacement;
        
        // Adjust for Gravity
        // displacement.y -= _FurGravity * length(displacement);
        
        // 3. Pass Layer Index
        output.furLayer = factor; // 0.0 to 1.0
        
        outStream.Append(output);
    }
}
```

**Fragment Shader (Strand Alpha)**
Uses a noise texture to discard pixels, creating the appearance of strands. The alpha threshold increases for outer layers, making tips thinner.

**Function (`OVERRIDE_FUR`):**
```hlsl
float furLayer = input.furLayer;
float furNoise = Sample(_FurNoiseMask);

// Alpha Threshold Calculation
// As furLayer goes 0->1, the threshold increases
float threshold = lerp(0.0, 1.0, furLayer); 

// Discard if noise value is below current layer threshold
if (furNoise < threshold) discard;

// Apply Fur AO (darken roots)
col.rgb *= lerp(1.0, 1.0 - _FurAO, 1.0 - furLayer);
```

## 2. Tessellation
Uses DX11 Hull and Domain shaders to subdivide geometry for smoother silhouettes and displacement mapping.

### Properties
*   **_TessEdge**: Target edge length (in pixels) for subdivision.
*   **_TessStrength**: overall multiplier.
*   **_TessFactorMax**: Maximum subdivision level (optimization).

### Principles & Math
**Hull Shader**: Calculates how much to tessellate each edge based on distance/screen size.
**Domain Shader**: Interpolates vertex data for the new vertices generated by the tessellator.

**Edge Factor Calculation (`lilCalcEdgeTessFactor`):**
```hlsl
float lilCalcEdgeTessFactor(float3 wpos0, float3 wpos1) {
    float edgeLength = distance(wpos0, wpos1);
    float distToCam = distance(Center(wpos0, wpos1), CameraPos);
    
    // Tessellate more if edge is long in screen space
    return edgeLength * ScreenHeight / (_TessEdge * distToCam);
}
```

## 3. Refraction
Simulates light bending through transparent materials like glass or water.

### Properties
*   **_RefractionStrength**: Distortion amount.
*   **_RefractionFresnel**: Varies distortion strength at edges.

### Principles & Math
Samples the opaque scene texture (GrabPass or CameraOpaqueTexture) using distorted screen UVs.

**Implementation (`OVERRIDE_REFRACTION`):**
```hlsl
// 1. Calculate Distortion Offset
float2 offset = NormalViewSpace.xy * _RefractionStrength;

// 2. Apply Fresnel (optional)
// Distort less at center, more at edges? or vice-versa
offset *= pow(NdotV, _RefractionFresnel);

// 3. Sample Scene Color
float2 uv = ScreenUV + offset;
float3 sceneColor = Sample(SceneTexture, uv);

// 4. Blend
col.rgb = lerp(col.rgb, sceneColor, Transparency);
```

## 4. Gem
Specialized shader for gemstones combining refraction, internal reflection, and volumetric coloring.

### Principles
*   **Multi-Layer Parallax**: Simulates internal facets by rendering "fake" interior geometry using parallax mapping.
*   **CubeMap Reflection**: High-contrast environment reflections.
*   **Chromatic Aberration**: Splits RGB channels during refraction sampling to simulate dispersion.

**Pseudo-Code:**
```hlsl
// Internal Facets (Parallax)
float3 rayDir = normalize(ViewDirTangent);
float2 internalUV = uv + rayDir.xy * _GemParallaxDepth;
float3 internalColor = Sample(_GemInteriorTex, internalUV);

// Dispersion
float3 refractColor;
refractColor.r = Sample(Scene, uv + offset * 1.0);
refractColor.g = Sample(Scene, uv + offset * 1.02);
refractColor.b = Sample(Scene, uv + offset * 1.04);
```
