# Vulnerability Scanner Skill

## Overview

The Vulnerability Scanner skill provides comprehensive security vulnerability scanning for dependencies and code. It integrates with CVE databases, security platforms, and SAST tools to identify, assess, and prioritize security risks during migration planning.

## Quick Start

### Prerequisites

- Target codebase with dependency manifests
- At least one scanning tool installed:
  - Snyk CLI (recommended)
  - Trivy
  - npm audit (for Node.js)
  - OWASP Dependency-Check

### Basic Usage

1. **Navigate to target project**
   ```bash
   cd /path/to/target/project
   ```

2. **Run vulnerability scan**
   ```bash
   # Scan dependencies
   snyk test --json > vuln-report.json

   # Or with Trivy
   trivy fs --format json --output vuln-report.json .
   ```

3. **Review results**
   - `security-report/vulnerabilities.json` - All vulnerabilities
   - `security-report/critical.md` - Critical issues requiring immediate action
   - `security-report/remediation.md` - Fix recommendations

## Features

### Vulnerability Detection

| Category | Description | Tools |
|----------|-------------|-------|
| Dependency CVEs | Known vulnerabilities in packages | Snyk, Trivy, npm audit |
| Security Anti-patterns | Insecure code patterns | Semgrep, SAST tools |
| Secrets Detection | Hardcoded credentials | Trivy, git-secrets |
| License Risk | Risky or unknown licenses | Snyk, FOSSA |

### Severity Classification

| Level | CVSS Score | SLA | Example |
|-------|------------|-----|---------|
| Critical | 9.0-10.0 | 7 days | Remote code execution |
| High | 7.0-8.9 | 30 days | Authentication bypass |
| Medium | 4.0-6.9 | 90 days | Information disclosure |
| Low | 0.1-3.9 | Best effort | Minor data leak |

### Exploitability Assessment

The skill evaluates:

- **Attack Vector**: Network, adjacent, local, physical
- **Attack Complexity**: Low vs high
- **Privileges Required**: None, low, high
- **User Interaction**: None required vs required
- **Exploit Maturity**: Proof of concept, functional, high

## Configuration

### Project Configuration

Create `.vulnerability-scanner.json`:

```json
{
  "scanScope": "full",
  "severityThreshold": "medium",
  "failOnSeverity": "critical",
  "databases": [
    "nvd",
    "ghsa",
    "osv"
  ],
  "scanners": {
    "dependencies": true,
    "secrets": true,
    "sast": true
  },
  "excludeVulnerabilities": [
    "CVE-2021-12345"
  ],
  "waivers": [
    {
      "id": "CVE-2021-12345",
      "reason": "Not exploitable - internal only service",
      "approvedBy": "security-team",
      "expiresAt": "2026-06-01"
    }
  ],
  "policy": {
    "maxCritical": 0,
    "maxHigh": 5,
    "requirePatchWithin": {
      "critical": "7d",
      "high": "30d",
      "medium": "90d"
    }
  },
  "notifications": {
    "newCritical": ["security@company.com"],
    "policyViolation": ["security@company.com", "dev-lead@company.com"]
  },
  "reporting": {
    "formats": ["json", "sarif", "markdown"],
    "outputDir": "./security-report"
  }
}
```

### CI/CD Integration

```yaml
# GitHub Actions example
- name: Security Scan
  run: |
    snyk test --severity-threshold=high
    trivy fs --severity CRITICAL,HIGH --exit-code 1 .
```

## Output Examples

### Vulnerability Report (JSON)

```json
{
  "scanId": "scan-20260124-143022",
  "timestamp": "2026-01-24T14:30:22Z",
  "summary": {
    "totalVulnerabilities": 23,
    "critical": 2,
    "high": 5,
    "medium": 12,
    "low": 4,
    "fixable": 19,
    "riskScore": 78
  },
  "vulnerabilities": [
    {
      "id": "CVE-2024-38816",
      "title": "Path Traversal in Spring Framework",
      "severity": "critical",
      "cvss": {
        "score": 9.8,
        "vector": "CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H"
      },
      "package": {
        "name": "spring-webmvc",
        "version": "5.3.18",
        "ecosystem": "maven"
      },
      "fixedVersions": "5.3.34, 6.0.17, 6.1.4",
      "patchAvailable": true,
      "exploitability": {
        "hasKnownExploit": true,
        "exploitMaturity": "functional",
        "attackVector": "network"
      },
      "remediation": {
        "recommendation": "Upgrade to spring-webmvc >= 5.3.34",
        "upgradeTarget": "5.3.34"
      }
    }
  ]
}
```

### Critical Issues Report (Markdown)

```markdown
# Critical Security Issues

## Summary
- **Total Critical**: 2
- **Immediate Action Required**: Yes
- **Estimated Remediation Time**: 4 hours

## Critical Vulnerabilities

### 1. CVE-2024-38816 - Spring Framework Path Traversal

**Severity**: Critical (CVSS 9.8)
**Package**: spring-webmvc@5.3.18
**Exploit Available**: Yes

**Description**: A path traversal vulnerability in Spring Framework allows attackers
to access files outside the intended directory.

**Remediation**:
```bash
# Update pom.xml
<dependency>
  <groupId>org.springframework</groupId>
  <artifactId>spring-webmvc</artifactId>
  <version>5.3.34</version>
</dependency>
```

**References**:
- [CVE-2024-38816](https://nvd.nist.gov/vuln/detail/CVE-2024-38816)
- [Spring Security Advisory](https://spring.io/security/cve-2024-38816)
```

### SARIF Output

```json
{
  "$schema": "https://raw.githubusercontent.com/oasis-tcs/sarif-spec/master/Schemata/sarif-schema-2.1.0.json",
  "version": "2.1.0",
  "runs": [
    {
      "tool": {
        "driver": {
          "name": "vulnerability-scanner",
          "version": "1.0.0",
          "rules": [
            {
              "id": "CVE-2024-38816",
              "shortDescription": {
                "text": "Path Traversal in Spring Framework"
              },
              "defaultConfiguration": {
                "level": "error"
              }
            }
          ]
        }
      },
      "results": [
        {
          "ruleId": "CVE-2024-38816",
          "level": "error",
          "message": {
            "text": "spring-webmvc@5.3.18 is vulnerable to path traversal"
          }
        }
      ]
    }
  ]
}
```

## Integration with Babysitter SDK

### Using in a Process

```javascript
import { defineTask } from '@a5c-ai/babysitter-sdk';

export const scanVulnerabilitiesTask = defineTask('scan-vulnerabilities', (args, ctx) => ({
  kind: 'skill',
  title: 'Security Vulnerability Scan',
  skill: {
    name: 'vulnerability-scanner',
    context: {
      targetPath: args.projectPath,
      scanScope: args.scanScope || 'full',
      severityThreshold: args.severityThreshold || 'medium',
      failOnSeverity: args.failOnSeverity || 'critical'
    }
  },
  io: {
    inputJsonPath: `tasks/${ctx.effectId}/input.json`,
    outputJsonPath: `tasks/${ctx.effectId}/result.json`
  }
}));
```

### Process Integration

This skill is used by these migration processes:

1. **dependency-analysis-updates** - Security audit
2. **security-remediation-migration** - Vulnerability fixing
3. **cloud-migration** - Security compliance check
4. **legacy-codebase-assessment** - Security posture

## CLI Examples

### Snyk Scanning

```bash
# Test for vulnerabilities
snyk test

# Test with severity threshold
snyk test --severity-threshold=high

# Generate JSON report
snyk test --json > snyk-report.json

# Monitor for new vulnerabilities
snyk monitor
```

### Trivy Scanning

```bash
# Scan filesystem
trivy fs .

# Scan with specific severities
trivy fs --severity CRITICAL,HIGH .

# Generate SARIF output
trivy fs --format sarif --output trivy.sarif .

# Scan specific image
trivy image myapp:latest
```

### npm audit

```bash
# Run audit
npm audit

# Generate JSON report
npm audit --json > npm-audit.json

# Fix automatically
npm audit fix

# Force fix (may include breaking changes)
npm audit fix --force
```

### OWASP Dependency-Check

```bash
# Run scan
dependency-check --project "MyApp" --scan .

# Generate multiple formats
dependency-check --project "MyApp" --scan . --format ALL
```

## Troubleshooting

### Common Issues

**Rate limiting**
```
Error: API rate limit exceeded
```
Solution: Use API token or reduce scan frequency

**False positives**
```
Warning: Vulnerability not applicable to this context
```
Solution: Add to waivers with justification

**Missing advisory data**
```
Warning: No advisory data available for package
```
Solution: Check alternative databases or report to maintainers

### Debug Mode

Enable verbose logging:

```json
{
  "debug": true,
  "logLevel": "verbose",
  "showApiCalls": true
}
```

## Best Practices

1. **Shift Left**: Scan in development, not just production
2. **Automate**: Integrate into CI/CD pipelines
3. **Prioritize**: Focus on exploitable, internet-facing vulnerabilities
4. **Document Waivers**: Maintain auditable waiver records
5. **Monitor**: Set up alerts for new critical vulnerabilities
6. **Track Metrics**: Monitor mean time to remediation (MTTR)

## Related Documentation

- [SKILL.md](./SKILL.md) - Full skill specification
- [Snyk Documentation](https://docs.snyk.io/)
- [Trivy Documentation](https://aquasecurity.github.io/trivy/)
- [NVD](https://nvd.nist.gov/)

## Version History

| Version | Date | Changes |
|---------|------|---------|
| 1.0.0 | 2026-01-24 | Initial release |
