/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { zod } from '../third_party/index.js';
import { ToolCategory } from './categories.js';
import { defineTool } from './ToolDefinition.js';
const FILTERABLE_RESOURCE_TYPES = [
    'document',
    'stylesheet',
    'image',
    'media',
    'font',
    'script',
    'texttrack',
    'xhr',
    'fetch',
    'prefetch',
    'eventsource',
    'websocket',
    'manifest',
    'signedexchange',
    'ping',
    'cspviolationreport',
    'preflight',
    'fedcm',
    'other',
];
export const listNetworkRequests = defineTool({
    name: 'list_network_requests',
    description: `List all requests for the currently selected page since the last navigation.`,
    annotations: {
        category: ToolCategory.NETWORK,
        readOnlyHint: true,
    },
    schema: {
        pageSize: zod
            .number()
            .int()
            .positive()
            .optional()
            .describe('Maximum number of requests to return. When omitted, returns all requests.'),
        pageIdx: zod
            .number()
            .int()
            .min(0)
            .optional()
            .describe('Page number to return (0-based). When omitted, returns the first page.'),
        resourceTypes: zod
            .array(zod.enum(FILTERABLE_RESOURCE_TYPES))
            .optional()
            .describe('Filter requests to only return requests of the specified resource types. When omitted or empty, returns all requests.'),
        includePreservedRequests: zod
            .boolean()
            .default(false)
            .optional()
            .describe('Set to true to return the preserved requests over the last 3 navigations.'),
    },
    handler: async (request, response, context) => {
        const data = await context.getDevToolsData();
        response.attachDevToolsData(data);
        const reqid = data?.cdpRequestId
            ? context.resolveCdpRequestId(data.cdpRequestId)
            : undefined;
        response.setIncludeNetworkRequests(true, {
            pageSize: request.params.pageSize,
            pageIdx: request.params.pageIdx,
            resourceTypes: request.params.resourceTypes,
            includePreservedRequests: request.params.includePreservedRequests,
            networkRequestIdInDevToolsUI: reqid,
        });
    },
});
export const getNetworkRequest = defineTool({
    name: 'get_network_request',
    description: `Gets a network request by an optional reqid, if omitted returns the currently selected request in the DevTools Network panel.`,
    annotations: {
        category: ToolCategory.NETWORK,
        readOnlyHint: true,
    },
    schema: {
        reqid: zod
            .number()
            .optional()
            .describe('The reqid of the network request. If omitted returns the currently selected request in the DevTools Network panel.'),
    },
    handler: async (request, response, context) => {
        if (request.params.reqid) {
            response.attachNetworkRequest(request.params.reqid);
        }
        else {
            const data = await context.getDevToolsData();
            response.attachDevToolsData(data);
            const reqid = data?.cdpRequestId
                ? context.resolveCdpRequestId(data.cdpRequestId)
                : undefined;
            if (reqid) {
                response.attachNetworkRequest(reqid);
            }
            else {
                response.appendResponseLine(`Nothing is currently selected in the DevTools Network panel.`);
            }
        }
    },
});
