/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import assert from 'node:assert';
import os from 'node:os';
import path from 'node:path';
import { describe, it } from 'node:test';
import { executablePath } from 'puppeteer';
import { ensureBrowserConnected, launch } from '../src/browser.js';
describe('browser', () => {
    it('cannot launch multiple times with the same profile', async () => {
        const tmpDir = os.tmpdir();
        const folderPath = path.join(tmpDir, `temp-folder-${crypto.randomUUID()}`);
        const browser1 = await launch({
            headless: true,
            isolated: false,
            userDataDir: folderPath,
            executablePath: executablePath(),
            devtools: false,
        });
        try {
            try {
                const browser2 = await launch({
                    headless: true,
                    isolated: false,
                    userDataDir: folderPath,
                    executablePath: executablePath(),
                    devtools: false,
                });
                await browser2.close();
                assert.fail('not reached');
            }
            catch (err) {
                assert.strictEqual(err.message, `The browser is already running for ${folderPath}. Use --isolated to run multiple browser instances.`);
            }
        }
        finally {
            await browser1.close();
        }
    });
    it('launches with the initial viewport', async () => {
        const tmpDir = os.tmpdir();
        const folderPath = path.join(tmpDir, `temp-folder-${crypto.randomUUID()}`);
        const browser = await launch({
            headless: true,
            isolated: false,
            userDataDir: folderPath,
            executablePath: executablePath(),
            viewport: {
                width: 1501,
                height: 801,
            },
            devtools: false,
        });
        try {
            const [page] = await browser.pages();
            const result = await page.evaluate(() => {
                return { width: window.innerWidth, height: window.innerHeight };
            });
            assert.deepStrictEqual(result, {
                width: 1501,
                height: 801,
            });
        }
        finally {
            await browser.close();
        }
    });
    it('connects to an existing browser with userDataDir', async () => {
        const tmpDir = os.tmpdir();
        const folderPath = path.join(tmpDir, `temp-folder-${crypto.randomUUID()}`);
        const browser = await launch({
            headless: true,
            isolated: false,
            userDataDir: folderPath,
            executablePath: executablePath(),
            devtools: false,
            args: ['--remote-debugging-port=0'],
        });
        try {
            const connectedBrowser = await ensureBrowserConnected({
                userDataDir: folderPath,
                devtools: false,
            });
            assert.ok(connectedBrowser);
            assert.ok(connectedBrowser.connected);
            connectedBrowser.disconnect();
        }
        finally {
            await browser.close();
        }
    });
});
