# Safe Migration Patterns

Patterns for writing safe, reversible database migrations in Rails.

## Basic Migration Structure

```ruby
class CreatePosts < ActiveRecord::Migration[7.1]
  def change
    create_table :posts do |t|
      t.string :title, null: false
      t.text :content
      t.references :user, null: false, foreign_key: true
      t.boolean :published, default: false, null: false
      t.datetime :published_at
      t.timestamps
    end

    add_index :posts, :published_at
    add_index :posts, [:user_id, :published]
  end
end
```

## Reversible Migrations

### Using change (Preferred)

Rails can automatically reverse these:

```ruby
def change
  create_table :posts
  add_column :users, :admin, :boolean
  add_index :posts, :slug
  add_reference :posts, :category, foreign_key: true
  rename_column :users, :name, :full_name
  change_column_default :posts, :published, from: nil, to: false
end
```

### Using reversible

For complex operations:

```ruby
def change
  reversible do |dir|
    dir.up do
      execute <<~SQL
        CREATE TYPE post_status AS ENUM ('draft', 'published', 'archived')
      SQL
    end

    dir.down do
      execute <<~SQL
        DROP TYPE post_status
      SQL
    end
  end
end
```

### Using up/down

When change isn't sufficient:

```ruby
def up
  change_column :posts, :content, :text
end

def down
  change_column :posts, :content, :string
end
```

## Column Types

### Common Types

```ruby
t.string :name           # VARCHAR(255)
t.text :content          # TEXT
t.integer :count         # INTEGER
t.bigint :large_number   # BIGINT
t.float :price           # FLOAT
t.decimal :amount, precision: 10, scale: 2  # DECIMAL(10,2)
t.boolean :active        # BOOLEAN
t.date :birth_date       # DATE
t.datetime :published_at # DATETIME/TIMESTAMP
t.time :start_time       # TIME
t.binary :data           # BLOB
t.json :metadata         # JSON (native)
t.jsonb :settings        # JSONB (PostgreSQL)
t.uuid :external_id      # UUID
```

### References/Foreign Keys

```ruby
# Standard reference
t.references :user, foreign_key: true

# With null allowed
t.references :category, foreign_key: true, null: true

# Polymorphic
t.references :commentable, polymorphic: true

# Custom foreign key name
t.references :author, foreign_key: { to_table: :users }

# Index options
t.references :user, index: { unique: true }
```

## Adding Columns Safely

### Non-null Column with Default

```ruby
# Safe: default value provided
add_column :users, :admin, :boolean, default: false, null: false

# For large tables, do it in steps:
# 1. Add column allowing null
add_column :users, :role, :string

# 2. Backfill in batches (separate migration or rake task)
User.in_batches.update_all(role: 'member')

# 3. Add constraints
change_column_null :users, :role, false
change_column_default :users, :role, 'member'
```

### Adding Index Without Downtime

```ruby
# For large tables, use CONCURRENTLY (PostgreSQL)
class AddIndexToPostsTitle < ActiveRecord::Migration[7.1]
  disable_ddl_transaction!

  def change
    add_index :posts, :title, algorithm: :concurrently
  end
end
```

## Modifying Columns

### Changing Type

```ruby
# Safe direction
change_column :products, :price, :decimal, precision: 10, scale: 2

# Requires data migration
def up
  add_column :users, :status_new, :integer
  User.update_all("status_new = CASE status WHEN 'active' THEN 1 ELSE 0 END")
  remove_column :users, :status
  rename_column :users, :status_new, :status
end
```

### Renaming

```ruby
rename_column :users, :name, :full_name
rename_table :posts, :articles
```

## Indexes

### Adding Indexes

```ruby
# Single column
add_index :posts, :slug, unique: true

# Composite
add_index :posts, [:user_id, :created_at]

# Partial (PostgreSQL)
add_index :posts, :published_at, where: "published = true"

# Expression (PostgreSQL)
add_index :users, 'lower(email)', unique: true

# With specific name
add_index :posts, :title, name: 'idx_posts_title'
```

### Index Best Practices

```ruby
# Always index foreign keys
add_index :posts, :user_id

# Index columns used in WHERE clauses
add_index :posts, :status

# Index columns used in ORDER BY
add_index :posts, :created_at

# Composite index for common queries
add_index :posts, [:user_id, :status, :created_at]
```

## Constraints

### NOT NULL

```ruby
# On new column
add_column :users, :email, :string, null: false

# On existing column (with default for existing rows)
change_column_null :users, :name, false, 'Unknown'
```

### Check Constraints (PostgreSQL)

```ruby
add_check_constraint :products, "price >= 0", name: "products_price_positive"
add_check_constraint :orders, "quantity > 0", name: "orders_quantity_positive"
```

### Foreign Keys

```ruby
# Add
add_foreign_key :posts, :users
add_foreign_key :posts, :users, on_delete: :cascade
add_foreign_key :posts, :users, on_delete: :nullify

# Remove
remove_foreign_key :posts, :users
```

## Data Migrations

### Inline (Small Tables)

```ruby
def up
  add_column :users, :slug, :string

  User.find_each do |user|
    user.update_column(:slug, user.name.parameterize)
  end

  add_index :users, :slug, unique: true
  change_column_null :users, :slug, false
end
```

### Background Job (Large Tables)

```ruby
# migration
def change
  add_column :users, :normalized_email, :string
  add_index :users, :normalized_email
end

# Separate rake task or job
namespace :data do
  task normalize_emails: :environment do
    User.in_batches(of: 1000) do |batch|
      batch.update_all("normalized_email = LOWER(TRIM(email))")
    end
  end
end
```

## Safe Migration Checklist

### Before Deployment
- [ ] Migration is reversible
- [ ] No downtime required (or scheduled)
- [ ] Indexes added for new foreign keys
- [ ] Large table changes use batching

### Dangerous Operations
- Removing columns (use `ignored_columns` first)
- Renaming columns (add new, migrate data, remove old)
- Changing column types
- Adding NOT NULL to existing column

### Safe Pattern for Removing Columns

```ruby
# Step 1: Add to ignored_columns (deploy)
class User < ApplicationRecord
  self.ignored_columns = [:old_column]
end

# Step 2: Remove column (next deploy)
class RemoveOldColumnFromUsers < ActiveRecord::Migration[7.1]
  def change
    remove_column :users, :old_column, :string
  end
end
```

## PostgreSQL-Specific

```ruby
# Enable extension
enable_extension 'pgcrypto'
enable_extension 'uuid-ossp'

# UUID primary key
create_table :posts, id: :uuid do |t|
  t.string :title
end

# JSONB column
add_column :users, :preferences, :jsonb, default: {}
add_index :users, :preferences, using: :gin

# Array column
add_column :posts, :tags, :string, array: true, default: []
add_index :posts, :tags, using: :gin

# Enum type
create_enum :post_status, ['draft', 'published', 'archived']
add_column :posts, :status, :enum, enum_type: :post_status, default: 'draft'
```

## Testing Migrations

```ruby
# test/migrations/create_posts_test.rb
require 'test_helper'

class CreatePostsTest < ActiveSupport::TestCase
  def setup
    @migration = CreatePosts.new
  end

  test "migration is reversible" do
    @migration.migrate(:up)
    assert Post.table_exists?

    @migration.migrate(:down)
    refute Post.table_exists?
  end
end
```
