# ActiveRecord Validation Patterns

Patterns for implementing model validations in Rails.

## Built-in Validators

### Presence

```ruby
validates :name, presence: true
validates :email, presence: { message: "is required" }

# For associations
validates :user, presence: true  # Validates associated object exists
```

### Uniqueness

```ruby
validates :email, uniqueness: true
validates :email, uniqueness: { case_sensitive: false }
validates :slug, uniqueness: { scope: :category_id }
validates :email, uniqueness: {
  scope: :account_id,
  message: "is already registered for this account"
}
```

**Important:** Always add a database unique index:
```ruby
add_index :users, :email, unique: true
add_index :posts, [:slug, :category_id], unique: true
```

### Format

```ruby
validates :email, format: { with: URI::MailTo::EMAIL_REGEXP }
validates :phone, format: {
  with: /\A\d{10}\z/,
  message: "must be 10 digits"
}
validates :username, format: {
  with: /\A[a-z0-9_]+\z/,
  message: "can only contain lowercase letters, numbers, and underscores"
}
```

### Length

```ruby
validates :name, length: { minimum: 2 }
validates :bio, length: { maximum: 500 }
validates :password, length: { in: 8..128 }
validates :pin, length: { is: 4 }
validates :summary, length: {
  maximum: 200,
  too_long: "%{count} characters is the maximum"
}
```

### Numericality

```ruby
validates :age, numericality: { only_integer: true }
validates :price, numericality: { greater_than: 0 }
validates :quantity, numericality: {
  only_integer: true,
  greater_than_or_equal_to: 0
}
validates :discount, numericality: {
  greater_than_or_equal_to: 0,
  less_than_or_equal_to: 100
}
```

### Inclusion/Exclusion

```ruby
STATUSES = %w[pending active archived].freeze
validates :status, inclusion: { in: STATUSES }

validates :subdomain, exclusion: {
  in: %w[www admin api],
  message: "%{value} is reserved"
}
```

### Comparison

```ruby
validates :end_date, comparison: { greater_than: :start_date }
validates :age, comparison: { greater_than_or_equal_to: 18 }
```

## Conditional Validations

```ruby
# With if/unless
validates :card_number, presence: true, if: :paid_with_card?
validates :reason, presence: true, unless: :approved?

# With Proc
validates :password, presence: true, if: -> { new_record? || password_changed? }

# Multiple conditions
validates :bio, length: { maximum: 500 },
          if: :bio_present?,
          unless: :admin?

# With on: option
validates :password, presence: true, on: :create
validates :email, uniqueness: true, on: :update
```

## Custom Validations

### Custom Method

```ruby
class Order < ApplicationRecord
  validate :delivery_date_in_future
  validate :has_items, on: :create

  private

  def delivery_date_in_future
    return unless delivery_date.present?

    if delivery_date <= Date.current
      errors.add(:delivery_date, "must be in the future")
    end
  end

  def has_items
    if items.empty?
      errors.add(:base, "Order must have at least one item")
    end
  end
end
```

### Custom Validator Class

```ruby
# app/validators/email_validator.rb
class EmailValidator < ActiveModel::EachValidator
  def validate_each(record, attribute, value)
    return if value.blank?

    unless value =~ URI::MailTo::EMAIL_REGEXP
      record.errors.add(attribute, options[:message] || "is not a valid email")
    end

    if options[:disposable] == false && disposable_email?(value)
      record.errors.add(attribute, "cannot be a disposable email")
    end
  end

  private

  def disposable_email?(email)
    domain = email.split('@').last
    DisposableDomains.include?(domain)
  end
end

# Usage
class User < ApplicationRecord
  validates :email, email: { disposable: false }
end
```

### Reusable Validation Concern

```ruby
# app/models/concerns/sluggable.rb
module Sluggable
  extend ActiveSupport::Concern

  included do
    validates :slug, presence: true,
              uniqueness: true,
              format: { with: /\A[a-z0-9-]+\z/ }

    before_validation :generate_slug, on: :create
  end

  private

  def generate_slug
    self.slug ||= name&.parameterize
  end
end

# Usage
class Post < ApplicationRecord
  include Sluggable
end
```

## Validation Contexts

```ruby
class User < ApplicationRecord
  validates :terms_accepted, acceptance: true, on: :registration
  validates :admin_note, presence: true, on: :admin_update
end

# Usage
user.save(context: :registration)
user.valid?(:admin_update)
```

## Validation Errors

### Accessing Errors

```ruby
user.valid?
user.errors.full_messages  # ["Email can't be blank", "Name is too short"]
user.errors[:email]        # ["can't be blank", "is invalid"]
user.errors.of_kind?(:email, :blank)  # true
user.errors.added?(:email, :blank)    # true
```

### Custom Error Messages

```ruby
validates :email, presence: { message: "is required" }
validates :age, numericality: {
  greater_than: 18,
  message: "you must be at least 18 years old"
}

# With interpolation
validates :name, length: {
  minimum: 3,
  message: "must be at least %{count} characters"
}

# In locale file
# config/locales/en.yml
en:
  activerecord:
    errors:
      models:
        user:
          attributes:
            email:
              taken: "is already registered"
```

## Strict Validations

Raise exception instead of adding errors:

```ruby
validates :token, presence: true, strict: true
# Raises ActiveModel::StrictValidationFailed

validates :api_key, presence: true, strict: ApiKeyError
# Raises custom exception
```

## Database Constraints

Always complement validations with database constraints:

```ruby
# Migration
class AddConstraintsToUsers < ActiveRecord::Migration[7.1]
  def change
    change_column_null :users, :email, false
    add_index :users, :email, unique: true

    # Check constraint (PostgreSQL)
    add_check_constraint :users, "age >= 0", name: "users_age_positive"

    # Foreign key
    add_foreign_key :posts, :users, on_delete: :cascade
  end
end
```

## Skipping Validations

Use cautiously:

```ruby
user.save(validate: false)
User.insert_all([...])  # Bulk insert without validations
user.update_column(:admin, true)  # Single column, no validations/callbacks
user.update_columns(admin: true, role: 'admin')
```

## Best Practices

### Do
- Add database constraints for critical validations
- Use built-in validators when possible
- Extract complex validations to custom validators
- Test validations thoroughly
- Use meaningful error messages

### Don't
- Don't rely solely on model validations for data integrity
- Don't use `validates_associated` without understanding cascading
- Don't skip validations in production code
- Don't put complex logic in validation methods

## Validation Order

Validations run in the order they're defined:

```ruby
class User < ApplicationRecord
  validates :email, presence: true
  validates :email, format: { with: URI::MailTo::EMAIL_REGEXP }
  validates :email, uniqueness: true

  # Better: combine into one
  validates :email,
            presence: true,
            format: { with: URI::MailTo::EMAIL_REGEXP },
            uniqueness: { case_sensitive: false }
end
```
