# Value Objects in Rails

Value objects are immutable objects that represent a descriptive aspect of the domain with no conceptual identity. They are defined only by their attributes.

## Struct for Value Objects

Use `Struct` to create lightweight value objects with named attributes.

### Basic Struct Pattern

```ruby
Color = Struct.new(:name, :value) do
  class << self
    def for_value(value)
      COLORS.find { |it| it.value == value }
    end
  end

  def to_s
    value
  end

  COLORS = [
    new("Blue", "var(--color-card-default)"),
    new("Gray", "var(--color-card-1)"),
    new("Green", "var(--color-card-2)"),
    new("Red", "var(--color-card-3)"),
    new("Yellow", "var(--color-card-4)"),
  ].freeze
end

# Usage
color = Color::COLORS.first
color.name  # => "Blue"
color.value # => "var(--color-card-default)"

Color.for_value("var(--color-card-2)")  # => #<struct Color name="Green", ...>
```

### Struct with Custom Methods

```ruby
Point = Struct.new(:x, :y) do
  def distance_to(other)
    Math.sqrt((x - other.x)**2 + (y - other.y)**2)
  end

  def +(other)
    Point.new(x + other.x, y + other.y)
  end

  def to_s
    "(#{x}, #{y})"
  end
end

# Usage
p1 = Point.new(0, 0)
p2 = Point.new(3, 4)
p1.distance_to(p2)  # => 5.0
```

### Enum-like Structs

```ruby
Status = Struct.new(:name, :value, :color) do
  class << self
    def all
      STATUSES
    end

    def find_by_value(value)
      STATUSES.find { |s| s.value == value }
    end
  end

  def active?
    value == "active"
  end

  STATUSES = [
    new("Active", "active", "green"),
    new("Pending", "pending", "yellow"),
    new("Archived", "archived", "gray"),
  ].freeze
end

# Usage in models
class Project < ApplicationRecord
  def status
    Status.find_by_value(read_attribute(:status))
  end
end
```

## Data Class for Immutable Objects

**Context Awareness:** Requires Ruby 3.2+. Check Ruby version with `ruby -v`.

### Basic Data Pattern (Ruby 3.2+)

```ruby
Coordinate = Data.define(:lat, :lng) do
  def to_s
    "#{lat}, #{lng}"
  end

  def distance_to(other)
    # Haversine formula or simple Euclidean distance
    Math.sqrt((lat - other.lat)**2 + (lng - other.lng)**2)
  end
end

# Usage
coord = Coordinate.new(lat: 40.7128, lng: -74.0060)
coord.lat = 50.0  # => FrozenError (Data objects are immutable)

# Create new modified instance
new_coord = coord.with(lat: 41.0)
```

### Fallback for Ruby < 3.2

```ruby
# Frozen Struct pattern for older Ruby versions
Coordinate = Struct.new(:lat, :lng) do
  def initialize(lat:, lng:)
    super(lat, lng)
    freeze
  end

  def to_s
    "#{lat}, #{lng}"
  end

  def with(**changes)
    self.class.new(
      lat: changes.fetch(:lat, self.lat),
      lng: changes.fetch(:lng, self.lng)
    )
  end
end
```

### Money Value Object

```ruby
Money = Data.define(:amount, :currency) do
  def initialize(amount:, currency: "USD")
    super(amount: amount.to_d, currency: currency)
  end

  def to_s
    "#{currency}#{format('%.2f', amount)}"
  end

  def +(other)
    raise ArgumentError, "Currency mismatch" unless currency == other.currency
    Money.new(amount: amount + other.amount, currency: currency)
  end

  def -(other)
    raise ArgumentError, "Currency mismatch" unless currency == other.currency
    Money.new(amount: amount - other.amount, currency: currency)
  end

  def *(multiplier)
    Money.new(amount: amount * multiplier, currency: currency)
  end

  def zero?
    amount.zero?
  end
end

# Usage
price = Money.new(amount: 99.99, currency: "USD")
tax = price * 0.08
total = price + tax
```

### Email Value Object with Validation

```ruby
Email = Data.define(:address) do
  def initialize(address:)
    normalized = address.to_s.downcase.strip
    raise ArgumentError, "Invalid email format" unless valid_format?(normalized)
    super(address: normalized)
  end

  def domain
    address.split("@").last
  end

  def username
    address.split("@").first
  end

  def to_s
    address
  end

  private

  def valid_format?(addr)
    addr.match?(URI::MailTo::EMAIL_REGEXP)
  end
end

# Usage
email = Email.new(address: "  User@Example.COM  ")
email.address  # => "user@example.com"
email.domain   # => "example.com"
```

### DateRange Value Object

```ruby
DateRange = Data.define(:start_date, :end_date) do
  def initialize(start_date:, end_date:)
    raise ArgumentError, "End date must be after start date" if end_date < start_date
    super
  end

  def include?(date)
    date >= start_date && date <= end_date
  end

  def days
    (end_date - start_date).to_i + 1
  end

  def to_range
    start_date..end_date
  end

  def overlap?(other)
    start_date <= other.end_date && end_date >= other.start_date
  end
end

# Usage
range = DateRange.new(start_date: Date.today, end_date: 7.days.from_now.to_date)
range.include?(3.days.from_now.to_date)  # => true
range.days  # => 8
```

## Value Objects with ActiveModel

For value objects that need validation, serialization, or form integration:

```ruby
class Address
  include ActiveModel::Model
  include ActiveModel::Attributes

  attribute :street, :string
  attribute :city, :string
  attribute :state, :string
  attribute :zip, :string

  validates :street, :city, :state, :zip, presence: true
  validates :zip, format: { with: /\A\d{5}(-\d{4})?\z/ }

  def to_s
    "#{street}, #{city}, #{state} #{zip}"
  end

  def ==(other)
    other.is_a?(Address) &&
      street == other.street &&
      city == other.city &&
      state == other.state &&
      zip == other.zip
  end

  alias eql? ==

  def hash
    [street, city, state, zip].hash
  end
end

# Usage in models with composed_of (deprecated but still works)
class User < ApplicationRecord
  # Store as JSON or in separate columns
  def address
    @address ||= Address.new(address_attributes)
  end

  def address=(addr)
    @address = addr
    self.address_attributes = addr.attributes
  end
end
```

## Serialization to Database

### Option 1: JSON Column (PostgreSQL, MySQL 5.7+)

```ruby
class Event < ApplicationRecord
  # Migration: add_column :events, :location, :jsonb

  def location
    @location ||= begin
      data = read_attribute(:location)
      data ? Coordinate.new(**data.symbolize_keys) : nil
    end
  end

  def location=(coord)
    @location = coord
    write_attribute(:location, coord&.to_h)
  end
end
```

### Option 2: Serialized Attribute

```ruby
class Event < ApplicationRecord
  serialize :location, coder: JSON

  def location
    @location ||= begin
      data = super
      data ? Coordinate.new(**data.symbolize_keys) : nil
    end
  end

  def location=(coord)
    @location = coord
    super(coord&.to_h)
  end
end
```

### Option 3: Custom Type (Rails 5+)

```ruby
class CoordinateType < ActiveRecord::Type::Value
  def cast(value)
    case value
    when Coordinate then value
    when Hash then Coordinate.new(**value.symbolize_keys)
    when String then Coordinate.new(**JSON.parse(value).symbolize_keys)
    else nil
    end
  end

  def serialize(value)
    value&.to_h&.to_json
  end

  def deserialize(value)
    return nil unless value
    data = JSON.parse(value)
    Coordinate.new(**data.symbolize_keys)
  end
end

# Register type
ActiveRecord::Type.register(:coordinate, CoordinateType)

# Use in model
class Event < ApplicationRecord
  attribute :location, :coordinate
end
```

## When to Upgrade: Decision Matrix

| Start With | Upgrade When | Upgrade To |
|------------|--------------|------------|
| Hash | Same keys appear 3+ times | Struct |
| Struct | Need custom methods | Struct with block |
| Struct | Need immutability | Data (Ruby 3.2+) |
| Struct | Need database persistence | Class < ApplicationRecord |
| Data | Need mutability | Struct |
| Value Object | Need validations | ActiveModel::Model |
| ActiveModel | Need persistence | ApplicationRecord |

## Best Practices

### Do
- Use value objects for domain concepts (Money, Address, DateRange)
- Make value objects immutable (use Data or frozen Struct)
- Implement equality methods (==, eql?, hash)
- Add meaningful custom methods (distance_to, overlap?, etc.)
- Use Data for Ruby 3.2+, frozen Struct for older versions
- Validate in initialize for Data/Struct objects
- Store as JSON in database when appropriate
- Use custom ActiveRecord types for clean serialization

### Don't
- Don't use value objects for entities with identity
- Don't mutate value objects after creation
- Don't add behavior unrelated to the value
- Don't use value objects when a simple string/number suffices
- Don't forget to implement to_s for debugging
- Don't mix domain logic with persistence logic
- Don't create value objects for single attributes

## Common Value Object Examples

### Priority

```ruby
Priority = Struct.new(:level, :label, :color) do
  class << self
    def find(level)
      ALL.find { |p| p.level == level }
    end
  end

  def high?
    level >= 3
  end

  ALL = [
    new(1, "Low", "gray"),
    new(2, "Medium", "yellow"),
    new(3, "High", "orange"),
    new(4, "Critical", "red"),
  ].freeze
end
```

### PhoneNumber

```ruby
PhoneNumber = Data.define(:number) do
  def initialize(number:)
    cleaned = number.to_s.gsub(/\D/, '')
    raise ArgumentError, "Invalid phone number" unless cleaned.length == 10
    super(number: cleaned)
  end

  def formatted
    "(#{area_code}) #{exchange}-#{line}"
  end

  def area_code
    number[0..2]
  end

  def exchange
    number[3..5]
  end

  def line
    number[6..9]
  end

  def to_s
    formatted
  end
end
```

### Percentage

```ruby
Percentage = Data.define(:value) do
  def initialize(value:)
    raise ArgumentError, "Value must be between 0 and 100" unless (0..100).cover?(value)
    super
  end

  def decimal
    value / 100.0
  end

  def to_s
    "#{value}%"
  end

  def of(amount)
    amount * decimal
  end
end

# Usage
discount = Percentage.new(value: 15)
discount.of(100)  # => 15.0
```
