# -*- coding: utf-8 -*-
# =============================================================================
# Agent-Skills/powerpoint-automation - AI-powered PPTX generation pipeline
# https://github.com/aktsmm/Agent-Skills/tree/main/powerpoint-automation
# 
# Copyright (c) aktsmm. Licensed under CC BY-NC-SA 4.0.
# DO NOT MODIFY THIS HEADER BLOCK.
# =============================================================================
"""
Create PPTX from template using slide master layouts.
Inherits design (colors, fonts, backgrounds) from template.

Usage:
    python scripts/create_from_template.py <template.pptx> <content.json> <output.pptx> [--config <layouts.json>]
    python scripts/create_from_template.py <template.pptx> --list-layouts

Options:
    --config <layouts.json>   Use pre-analyzed layout mapping file
                              (Generated by: python scripts/analyze_template.py <template.pptx>)
    --no-signature            Disable auto-signature in speaker notes

Supports image embedding:
    "image": {"path": "images/foo.png", "position": "right", "width_percent": 45}
    "image": {"url": "https://...", "position": "bottom", "height_percent": 50}

Auto-fixes applied:
    - Section slides: title/subtitle overlap detection and dynamic repositioning
    - Section slides: title position preserved unless out of range (20%-60%)
    - Section slides: subtitle font size 24pt (larger for readability)
    - Section slides: title width extended to prevent text wrapping
    - Two Column layouts: empty placeholders removed automatically
    - Images: size capped to 150% of natural size to prevent blurry enlargement
    - Images: height limited to prevent overflow on center-positioned images

Examples:
    # Analyze template first (recommended):
    python scripts/analyze_template.py templates/sample-ppf.pptx

    # Then use the generated config:
    python scripts/create_from_template.py templates/sample-ppf.pptx content.json output.pptx --config output_manifest/sample-ppf_layouts.json

    # Or let the script auto-detect layouts:
    python scripts/create_from_template.py templates/sample.pptx content.json output.pptx
"""

from pptx import Presentation
from pptx.util import Inches, Pt
from pptx.enum.text import PP_ALIGN
from pptx.dml.color import RGBColor
from pptx.oxml.ns import qn
from lxml import etree
import json
import sys
import argparse
import urllib.request
import tempfile
import subprocess
from pathlib import Path
from typing import Optional

try:
    from PIL import Image
    PIL_AVAILABLE = True
except ImportError:
    PIL_AVAILABLE = False

# Base directory for resolving relative image paths (set in create_pptx_from_template)
BASE_DIR: Optional[Path] = None

# Repository signature
REPO_URL = "https://github.com/aktsmm/Agent-Skills/tree/main/powerpoint-automation"
SIGNATURE_FIRST = f"📌 Generated by: {REPO_URL}"
SIGNATURE_LAST = f"---\n🔧 This presentation was created using Agent-Skills/powerpoint-automation\n{REPO_URL}"

# Text colors for light/dark backgrounds
TEXT_COLOR_LIGHT_BG = RGBColor(51, 51, 51)  # Dark gray for light backgrounds
TEXT_COLOR_DARK_BG = RGBColor(255, 255, 255)  # White for dark backgrounds


def is_dark_color(rgb_color) -> bool:
    """
    Determine if a color is dark (needs white text).
    Uses luminance formula: 0.299*R + 0.587*G + 0.114*B
    
    Args:
        rgb_color: Can be RGBColor, tuple (r,g,b), or hex string
    
    Returns:
        True if color is dark (luminance < 128)
    """
    try:
        if rgb_color is None:
            return False
        
        if hasattr(rgb_color, '__iter__') and len(rgb_color) == 3:
            r, g, b = rgb_color
        elif isinstance(rgb_color, str):
            hex_color = rgb_color.replace('#', '')
            r = int(hex_color[0:2], 16)
            g = int(hex_color[2:4], 16)
            b = int(hex_color[4:6], 16)
        elif hasattr(rgb_color, '__str__'):
            # RGBColor has format like 'RRGGBB'
            hex_str = str(rgb_color)
            if len(hex_str) == 6:
                r = int(hex_str[0:2], 16)
                g = int(hex_str[2:4], 16)
                b = int(hex_str[4:6], 16)
            else:
                return False
        else:
            return False
        
        luminance = 0.299 * r + 0.587 * g + 0.114 * b
        return luminance < 128
    except Exception:
        return False


def get_layout_background_color(slide):
    """
    Get the effective background color of a slide's layout.
    Checks slide, layout, and master backgrounds.
    
    Returns:
        RGBColor or None
    """
    try:
        # Check slide background
        if hasattr(slide, 'background') and slide.background.fill:
            fill = slide.background.fill
            if fill.type is not None and hasattr(fill, 'fore_color') and fill.fore_color:
                try:
                    return fill.fore_color.rgb
                except Exception:
                    pass
        
        # Check layout background
        layout = slide.slide_layout
        if hasattr(layout, 'background') and layout.background.fill:
            fill = layout.background.fill
            if fill.type is not None and hasattr(fill, 'fore_color') and fill.fore_color:
                try:
                    return fill.fore_color.rgb
                except Exception:
                    pass
        
        # Check master background
        master = layout.slide_master
        if hasattr(master, 'background') and master.background.fill:
            fill = master.background.fill
            if fill.type is not None and hasattr(fill, 'fore_color') and fill.fore_color:
                try:
                    return fill.fore_color.rgb
                except Exception:
                    pass
    except Exception:
        pass
    
    return None


def has_dark_background(slide, slide_data: dict = None) -> bool:
    """
    Determine if a slide has a dark background.
    Checks: explicit flag, layout background color, layout name hints.
    
    Args:
        slide: The PPTX slide object
        slide_data: Optional slide data dict with dark_background flag
    
    Returns:
        True if slide has dark background
    """
    # Check explicit flag in content.json
    if slide_data:
        if slide_data.get('dark_background') or slide_data.get('darkBackground'):
            return True
    
    # Check layout background color
    bg_color = get_layout_background_color(slide)
    if bg_color and is_dark_color(bg_color):
        return True
    
    # Check layout name for hints (common dark layout names)
    try:
        layout_name = slide.slide_layout.name.lower()
        dark_hints = ['dark', 'black', 'navy', 'section', 'セクション']
        for hint in dark_hints:
            if hint in layout_name:
                return True
    except Exception:
        pass
    
    return False


def get_text_color_for_slide(slide, slide_data: dict = None) -> RGBColor:
    """
    Get appropriate text color based on slide background.
    
    Returns:
        White for dark backgrounds, dark gray for light backgrounds
    """
    if has_dark_background(slide, slide_data):
        return TEXT_COLOR_DARK_BG
    return TEXT_COLOR_LIGHT_BG


def get_repo_info() -> str:
    """Get repository URL from git remote (fallback to default)."""
    try:
        result = subprocess.run(
            ['git', 'remote', 'get-url', 'origin'],
            capture_output=True, text=True, timeout=5
        )
        if result.returncode == 0:
            return result.stdout.strip()
    except Exception:
        pass
    return REPO_URL


def _normalize_view_settings(pptx_path: str) -> None:
    """
    Normalize PPTX view settings to ensure it opens in Normal view.
    
    Fixes: Template saved with Slide Master view (lastView="sldMasterView")
    will cause generated PPTX to open in Master view.
    
    This function modifies viewProps.xml inside the PPTX ZIP to set
    lastView="sldView" (Normal view).
    """
    import zipfile
    import shutil
    import os
    
    temp_path = pptx_path + '.tmp'
    modified = False
    
    try:
        with zipfile.ZipFile(pptx_path, 'r') as z_in:
            with zipfile.ZipFile(temp_path, 'w', zipfile.ZIP_DEFLATED) as z_out:
                for item in z_in.namelist():
                    content = z_in.read(item)
                    if item == 'ppt/viewProps.xml':
                        # Replace any non-normal view setting with sldView
                        original = content
                        content = content.replace(b'lastView="sldMasterView"', b'lastView="sldView"')
                        content = content.replace(b'lastView="sldSorterView"', b'lastView="sldView"')
                        content = content.replace(b'lastView="notesView"', b'lastView="sldView"')
                        content = content.replace(b'lastView="outlineView"', b'lastView="sldView"')
                        if content != original:
                            modified = True
                    z_out.writestr(item, content)
        
        # Replace original with modified file
        os.remove(pptx_path)
        shutil.move(temp_path, pptx_path)
        
        if modified:
            print("   🔧 View settings normalized to Normal view")
    except Exception as e:
        # Clean up temp file if it exists
        if os.path.exists(temp_path):
            os.remove(temp_path)
        # Silently continue - view settings are not critical


def list_layouts(template_path: str) -> None:
    """List all slide layouts in the template with their placeholders."""
    prs = Presentation(template_path)
    
    print(f"\n{'='*60}")
    print(f"Template: {template_path}")
    print(f"Slide Size: {prs.slide_width.inches:.2f}\" x {prs.slide_height.inches:.2f}\"")
    print(f"{'='*60}\n")
    
    print(f"Found {len(prs.slide_masters)} slide master(s)\n")
    
    for master_idx, master in enumerate(prs.slide_masters):
        print(f"━━━ Slide Master {master_idx} ━━━")
        if master.name:
            print(f"    Name: {master.name}")
        print(f"    Layouts: {len(master.slide_layouts)}")
        print()
        
        for layout_idx, layout in enumerate(master.slide_layouts):
            print(f"  [{layout_idx}] {layout.name or '(unnamed)'}")
            
            # Show placeholders
            placeholders = []
            for shape in layout.placeholders:
                ph_type = str(shape.placeholder_format.type).replace('PLACEHOLDER_TYPE.', '')
                placeholders.append(f"{shape.placeholder_format.idx}:{ph_type}")
            
            if placeholders:
                print(f"      Placeholders: {', '.join(placeholders)}")
            else:
                print(f"      Placeholders: (none - blank layout)")
            print()
    
    # Show existing slides
    if prs.slides:
        print(f"━━━ Existing Slides ({len(prs.slides)}) ━━━")
        for slide_idx, slide in enumerate(prs.slides):
            layout_name = slide.slide_layout.name or '(unnamed)'
            print(f"  Slide {slide_idx + 1}: Layout = {layout_name}")
        print()
    
    print("Usage tip:")
    print('  In content.json, specify "layout": <index> to use a specific layout.')
    print('  Use "layout": 0 for title slides, "layout": 1 for content slides (typical).')
    print()


def find_placeholder(slide, ph_type_names: list):
    """Find a placeholder by type name(s)."""
    for shape in slide.placeholders:
        ph_type = str(shape.placeholder_format.type)
        for name in ph_type_names:
            if name in ph_type:
                return shape
    return None


def resolve_image_path(image_config: dict, content_path: str) -> Optional[str]:
    """
    Resolve image path from config. Supports local path or URL.
    Returns the resolved local file path (downloads URL if needed).
    Also tries alternate extensions (.png, .jpg, .jpeg) if exact match not found.
    """
    if 'url' in image_config:
        # Download from URL to temp file
        url = image_config['url']
        try:
            suffix = Path(url.split('?')[0]).suffix or '.png'
            with tempfile.NamedTemporaryFile(delete=False, suffix=suffix) as tmp:
                print(f"    📥 Downloading image: {url[:50]}...")
                urllib.request.urlretrieve(url, tmp.name)
                return tmp.name
        except Exception as e:
            print(f"    [!] Failed to download image: {e}")
            return None
    elif 'path' in image_config:
        # Resolve local path
        path = image_config['path']
        base_dir = Path(content_path).parent.parent if content_path else Path('.')
        
        # Try exact path first
        candidates = [
            base_dir / path,
            Path(path),
            Path(content_path).parent / path if content_path else None,
        ]
        for candidate in candidates:
            if candidate and candidate.exists():
                return str(candidate)
        
        # Try alternate extensions (.png, .jpg, .jpeg)
        orig_path = Path(path)
        alt_extensions = ['.png', '.jpg', '.jpeg', '.gif', '.webp']
        for ext in alt_extensions:
            if ext == orig_path.suffix:
                continue  # Skip same extension
            alt_path = orig_path.with_suffix(ext)
            alt_candidates = [
                base_dir / alt_path,
                Path(alt_path),
            ]
            for candidate in alt_candidates:
                if candidate and candidate.exists():
                    return str(candidate)
        
        print(f"    [!] Image not found: {path}")
        return None
    return None


def get_image_size(image_path: str) -> tuple:
    """
    Get image dimensions in pixels.
    Returns (width, height) or (None, None) if unable to read.
    """
    if not PIL_AVAILABLE:
        return (None, None)
    try:
        with Image.open(image_path) as img:
            return img.size  # (width, height)
    except Exception:
        return (None, None)


def calculate_max_width_percent(image_path: str, slide_width_inches: float = 13.333) -> int:
    """
    Calculate maximum width_percent based on original image size.
    Prevents small images from being stretched too large.
    
    Returns adjusted max width_percent (10-100).
    """
    img_width, img_height = get_image_size(image_path)
    if img_width is None:
        return 100  # No limit if can't read
    
    # Assume 96 DPI for typical screen images
    dpi = 96
    img_width_inches = img_width / dpi
    
    # Calculate what percentage of slide width the image naturally fills
    natural_pct = (img_width_inches / slide_width_inches) * 100
    
    # Allow up to 1.5x natural size (150% scale), but cap at 100%
    max_pct = min(int(natural_pct * 1.5), 100)
    
    # Minimum 10% to avoid tiny images
    return max(max_pct, 10)


def remove_empty_picture_placeholders(slide) -> int:
    """
    Remove empty Picture Placeholders from slide.
    These show "image cannot be displayed" if left empty after adding a separate image.
    
    Returns:
        Number of placeholders removed
    """
    removed = 0
    shapes_to_remove = []
    
    for shape in slide.shapes:
        # Check if it's a Picture Placeholder without actual image content
        # Use shape.is_placeholder to safely check
        if hasattr(shape, 'is_placeholder') and shape.is_placeholder:
            try:
                ph_type = str(shape.placeholder_format.type)
                if 'PICTURE' in ph_type:
                    # Check if placeholder has no image (blip)
                    from lxml import etree
                    xml = etree.tostring(shape._element, encoding='unicode')
                    if 'blip' not in xml or 'r:embed' not in xml:
                        shapes_to_remove.append(shape)
            except (ValueError, AttributeError):
                # Not a valid placeholder, skip
                pass
            except Exception:
                pass
    
    for shape in shapes_to_remove:
        try:
            sp = shape._element
            sp.getparent().remove(sp)
            removed += 1
        except Exception as e:
            print(f"    [!] Could not remove placeholder: {e}")
    
    return removed


def remove_empty_body_placeholders(slide) -> int:
    """
    Remove empty Body/Content Placeholders from slide.
    These cause excess whitespace when layout has more placeholders than content.
    
    Returns:
        Number of placeholders removed
    """
    removed = 0
    shapes_to_remove = []
    
    for shape in slide.shapes:
        if hasattr(shape, 'is_placeholder') and shape.is_placeholder:
            try:
                ph_type = str(shape.placeholder_format.type)
                # Target BODY, OBJECT, CONTENT placeholders
                if any(t in ph_type for t in ['BODY', 'OBJECT', 'CONTENT']):
                    # Check if text frame is empty or has only whitespace
                    if hasattr(shape, 'has_text_frame') and shape.has_text_frame:
                        tf = shape.text_frame
                        text = ''.join(p.text for p in tf.paragraphs).strip()
                        if not text:
                            shapes_to_remove.append(shape)
                    else:
                        # No text frame at all - remove
                        shapes_to_remove.append(shape)
            except (ValueError, AttributeError):
                pass
            except Exception:
                pass
    
    for shape in shapes_to_remove:
        try:
            sp = shape._element
            sp.getparent().remove(sp)
            removed += 1
        except Exception as e:
            print(f"    [!] Could not remove body placeholder: {e}")
    
    return removed


def remove_empty_textboxes(slide) -> int:
    """
    Remove empty TextBox shapes from slide.
    Textboxes added programmatically may end up empty if no content is assigned.
    These are unnecessary and should be removed.
    
    Returns:
        Number of textboxes removed
    """
    from pptx.enum.shapes import MSO_SHAPE_TYPE
    
    removed = 0
    shapes_to_remove = []
    
    for shape in slide.shapes:
        # Check if it's a TextBox (not a placeholder)
        try:
            if shape.shape_type == MSO_SHAPE_TYPE.TEXT_BOX:
                # Check if text frame is empty or has only whitespace
                if hasattr(shape, 'has_text_frame') and shape.has_text_frame:
                    tf = shape.text_frame
                    text = ''.join(p.text for p in tf.paragraphs).strip()
                    if not text:
                        shapes_to_remove.append(shape)
                else:
                    # No text frame at all - remove
                    shapes_to_remove.append(shape)
        except Exception:
            pass
    
    for shape in shapes_to_remove:
        try:
            sp = shape._element
            sp.getparent().remove(sp)
            removed += 1
        except Exception as e:
            print(f"    [!] Could not remove empty textbox: {e}")
    
    return removed


def is_icon_or_logo(image_path: str, min_content_size: int = 400) -> tuple:
    """
    Detect if image is likely an icon/logo.
    
    Criteria:
    - Square aspect ratio (0.9-1.1) AND small size (<= 800px)
    - Very small size (< min_content_size on any dimension)
    
    Returns:
        Tuple of (is_icon_logo: bool, suggested_width_percent: int)
        If is_icon_logo is True, suggested_width_percent is a small appropriate size.
    """
    img_width, img_height = get_image_size(image_path)
    if img_width is None or img_height is None:
        return False, 45
    
    # Very small images (< 400px) are likely icons - use small size
    if img_width < min_content_size or img_height < min_content_size:
        # Suggest 15-20% for icons
        suggested = min(20, max(10, int(img_width / 13.333 / 96 * 100 * 1.2)))
        return True, suggested
    
    # Square images under 800px are likely logos - use small size
    aspect_ratio = img_width / img_height if img_height > 0 else 1
    if 0.9 <= aspect_ratio <= 1.1 and max(img_width, img_height) <= 800:
        # Suggest 15-25% for logos
        suggested = min(25, max(15, int(img_width / 13.333 / 96 * 100 * 1.2)))
        return True, suggested
    
    return False, 45


def add_image_to_slide(slide, prs, image_config: dict, content_path: str) -> dict:
    """
    Add image to slide based on configuration.
    
    Args:
        slide: pptx slide object
        prs: Presentation object (for slide dimensions)
        image_config: {"path": ..., "position": "right|bottom|full|center", "width_percent": 45}
        content_path: Path to content.json for resolving relative paths
    
    Returns:
        Content area dict with adjusted dimensions, or None for full image
    """
    image_path = resolve_image_path(image_config, content_path)
    # Get slide dimensions early for default return
    default_width = prs.slide_width.inches - 1.0  # 0.5 margin on each side
    default_height = prs.slide_height.inches - 2.0  # top/bottom margins
    
    if not image_path:
        return {'left': Inches(0.5), 'top': Inches(1.5), 'width': Inches(default_width), 'height': Inches(default_height)}
    
    position = image_config.get('position', 'right')
    width_pct = image_config.get('width_percent', 45)
    
    # For icons/logos, use appropriate small size instead of skipping
    is_icon, suggested_pct = is_icon_or_logo(image_path)
    if is_icon:
        img_w, img_h = get_image_size(image_path)
        print(f"    [i] Icon/logo detected ({img_w}x{img_h}px) - using size: {suggested_pct}%")
        width_pct = min(width_pct, suggested_pct)
    height_pct = image_config.get('height_percent', 50)
    
    # Get actual slide dimensions from presentation
    slide_width_inches = prs.slide_width.inches
    slide_height_inches = prs.slide_height.inches
    content_width = slide_width_inches - 1.0  # margin on both sides
    
    # Limit width_percent based on original image size to prevent over-stretching
    max_width_pct = calculate_max_width_percent(image_path, slide_width_inches)
    if width_pct > max_width_pct:
        print(f"    [i] Image size limited: {width_pct}% → {max_width_pct}% (original image is small)")
        width_pct = max_width_pct
    
    try:
        if position == 'full' or position == 'center':
            # Full slide image (centered below title bar)
            title_height = 1.5  # Reserved for title
            available_height = slide_height_inches - title_height - 0.3  # 0.3 bottom margin
            
            img_left = Inches(0.5)
            img_top = Inches(title_height)
            
            if position == 'center':
                # Calculate width first
                img_width_inches = content_width * width_pct / 100
                
                # Get image aspect ratio to calculate height
                img_w, img_h = get_image_size(image_path)
                if img_w and img_h:
                    aspect_ratio = img_w / img_h
                    calculated_height = img_width_inches / aspect_ratio
                    
                    # If height exceeds available space, limit by height instead
                    max_height = available_height * 0.95  # 95% of available
                    if calculated_height > max_height:
                        print(f"    [i] Image height limited: {calculated_height:.1f}\" → {max_height:.1f}\" (fits slide)")
                        # Recalculate width based on limited height
                        img_width_inches = max_height * aspect_ratio
                
                img_width = Inches(img_width_inches)
                img_left = Inches((slide_width_inches - img_width_inches) / 2)
            else:
                img_width = Inches(content_width)
            
            slide.shapes.add_picture(image_path, img_left, img_top, width=img_width)
            return None  # No text area
            
        elif position == 'right':
            # Image on right, text on left
            # Ensure image fits within slide with proper margin
            margin = 0.5  # inches
            
            img_width = Inches(slide_width_inches * width_pct / 100)
            # Position image with right margin
            img_left = Inches(slide_width_inches - margin) - img_width
            img_top = Inches(1.5)
            
            # Ensure img_left doesn't go negative or overlap with left content
            min_text_width = slide_width_inches * 0.35  # Minimum 35% for text
            if img_left < Inches(min_text_width + margin):
                # Reduce image width to fit
                available_width = slide_width_inches - min_text_width - margin * 2
                img_width = Inches(available_width)
                img_left = Inches(slide_width_inches - margin) - img_width
                print(f"    [i] Image width adjusted to fit slide")
            
            slide.shapes.add_picture(image_path, img_left, img_top, width=img_width)
            return {
                'left': Inches(margin),
                'top': Inches(1.5),
                'width': img_left - Inches(margin * 2),  # Text area width
                'height': Inches(slide_height_inches - 2.0)  # Dynamic height
            }
            
        elif position == 'bottom':
            # Image at bottom, text on top
            text_height = Inches(slide_height_inches * (100 - height_pct - 10) / 100)
            img_height = Inches(slide_height_inches * height_pct / 100)
            img_top = Inches(slide_height_inches - 0.5) - img_height
            img_left = Inches(0.5)
            slide.shapes.add_picture(image_path, img_left, img_top, height=img_height)
            return {
                'left': Inches(0.5),
                'top': Inches(1.5),
                'width': Inches(12.333),
                'height': text_height
            }
            
    except Exception as e:
        print(f"    [!] Failed to add image: {e}")
    
    # Default content area (using slide dimensions)
    return {'left': Inches(0.5), 'top': Inches(1.5), 'width': Inches(default_width), 'height': Inches(default_height)}


def add_slide_from_layout(prs, layout_idx: int, title: str, items: Optional[list] = None, 
                          subtitle: Optional[str] = None, image_config: Optional[dict] = None,
                          content_path: Optional[str] = None, code: Optional[str] = None,
                          slide_type: Optional[str] = None) -> None:
    """Add a slide using the specified layout from the template, with optional image.
    
    Features:
    - Section slides: auto-adjusts title width, height, and position for proper display
    - Two Column layouts: removes empty placeholders after content is set
    - Images: supports right/center/bottom positions with auto-sizing
    - Code blocks: styled with dark background and Consolas font
    
    Args:
        prs: Presentation object
        layout_idx: Layout index to use
        title: Slide title
        items: List of bullet items (strings)
        subtitle: Optional subtitle (for section/title slides)
        image_config: Image configuration dict with path, position, width_percent
        content_path: Base path for resolving relative image paths
        code: Code block content
        slide_type: Slide type ('section', 'content', etc.) for type-specific handling
    """
    # Get layout (with fallback)
    try:
        layout = prs.slide_layouts[layout_idx]
    except IndexError:
        print(f"  Warning: Layout {layout_idx} not found, using layout 0")
        layout = prs.slide_layouts[0]
    
    slide = prs.slides.add_slide(layout)
    
    # Get slide dimensions
    slide_width = prs.slide_width.inches
    slide_height = prs.slide_height.inches
    
    # Try to set title
    title_placeholder = find_placeholder(slide, ['TITLE', 'CENTER_TITLE'])
    if title_placeholder and title_placeholder.has_text_frame:
        title_placeholder.text_frame.paragraphs[0].text = title
        
        # For section slides, adjust title for better display
        if slide_type == 'section':
            # Extend title placeholder width to almost full slide width
            title_placeholder.width = Inches(slide_width - 1.5)
            # Disable word wrap to keep title on one line
            title_placeholder.text_frame.word_wrap = False
            # Disable autofit to prevent character spacing changes
            title_placeholder.text_frame.auto_size = None  # MSO_AUTO_SIZE.NONE
            
            # Fix height if it's reported as 0 (causes display issues)
            if title_placeholder.height.inches < 0.5:
                title_placeholder.height = Inches(1.0)
            
            # For section slides, always position title at 35% from top
            # This leaves room for subtitle below and looks balanced
            target_top = slide_height * 0.35
            original_top = title_placeholder.top.inches
            if abs(original_top - target_top) > 0.3:  # Only adjust if difference > 0.3"
                title_placeholder.top = Inches(target_top)
                print(f"    [i] Section title repositioned to 35% (was {original_top:.2f}\")")
    
    # Try to set subtitle (for title slides and section slides)
    if subtitle:
        subtitle_placeholder = find_placeholder(slide, ['SUBTITLE'])
        if subtitle_placeholder and subtitle_placeholder.has_text_frame:
            subtitle_placeholder.text_frame.paragraphs[0].text = subtitle
            
            # Dynamic positioning: check subtitle distance from title
            if title_placeholder:
                title_bottom = title_placeholder.top.inches + max(title_placeholder.height.inches, 0.8)
                subtitle_top = subtitle_placeholder.top.inches
                gap = subtitle_top - title_bottom
                
                # Maximum gap: 25% of slide height (or 1.5 inches, whichever is smaller)
                max_gap = min(slide_height * 0.25, 1.5)
                # Ideal gap: 0.3-0.5 inches
                ideal_gap = 0.4
                
                if gap < 0.2:
                    # Overlapping: move subtitle below title
                    new_top = title_bottom + ideal_gap
                    # IMPORTANT: python-pptx doesn't preserve width/height when only 'top' is set.
                    # This causes XML <a:xfrm> to have <a:off> without <a:ext>, making text render
                    # as single-column vertical. See TROUBLESHOOTING.md #48 for details.
                    original_left = subtitle_placeholder.left
                    original_width = subtitle_placeholder.width
                    original_height = subtitle_placeholder.height
                    subtitle_placeholder.top = Inches(new_top)
                    subtitle_placeholder.left = original_left  # Restore left position
                    subtitle_placeholder.width = original_width  # Restore width
                    subtitle_placeholder.height = original_height  # Restore height
                    print(f"    [i] Subtitle repositioned to avoid overlap (was {subtitle_top:.2f}, now {new_top:.2f})")
                elif gap > max_gap:
                    # Too far apart: bring subtitle closer to title
                    new_top = title_bottom + ideal_gap
                    # IMPORTANT: Same fix as above - preserve all dimensions. See #48.
                    original_left = subtitle_placeholder.left
                    original_width = subtitle_placeholder.width
                    original_height = subtitle_placeholder.height
                    subtitle_placeholder.top = Inches(new_top)
                    subtitle_placeholder.left = original_left  # Restore left position
                    subtitle_placeholder.width = original_width  # Restore width
                    subtitle_placeholder.height = original_height  # Restore height
                    print(f"    [i] Subtitle repositioned to reduce gap (was {subtitle_top:.2f}, now {new_top:.2f}, gap was {gap:.2f})")
            
            # Ensure subtitle font is readable (24pt for section slides)
            if slide_type == 'section':
                for para in subtitle_placeholder.text_frame.paragraphs:
                    para.font.size = Pt(24)
        else:
            # Fallback: add textbox for subtitle if no placeholder exists
            # Calculate subtitle position dynamically based on title position
            
            if title_placeholder:
                title_top = title_placeholder.top.inches
                title_height = title_placeholder.height.inches
                
                # Some placeholders report height=0, use minimum of 0.8 inches
                if title_height < 0.5:
                    title_height = 0.8
                
                # Position subtitle below title with small margin (0.15" for tighter grouping)
                subtitle_top = title_top + title_height + 0.15
            else:
                # No title placeholder: use 45% of slide height as default
                subtitle_top = slide_height * 0.45
            
            subtitle_box = slide.shapes.add_textbox(
                Inches(0.64),  # Match title left margin
                Inches(subtitle_top),  # Position below title
                Inches(slide_width - 1.5),  # Almost full width
                Inches(0.8)  # Fixed height
            )
            tf = subtitle_box.text_frame
            tf.word_wrap = False  # Keep on one line
            p = tf.paragraphs[0]
            p.text = subtitle
            p.font.size = Pt(24)  # Larger subtitle for section slides
            p.font.color.rgb = RGBColor(100, 100, 100)  # Gray color
            p.alignment = PP_ALIGN.LEFT
    
    # Add image if configured (before content, so image appears behind text if overlapping)
    content_area = None
    if image_config:
        content_area = add_image_to_slide(slide, prs, image_config, content_path or "")
        # Remove empty Picture Placeholders to avoid "image cannot be displayed" issue
        removed = remove_empty_picture_placeholders(slide)
        if removed > 0:
            print(f"    [i] Removed {removed} empty picture placeholder(s)")
        if content_area is None:
            # Full image slide - skip content
            return slide
    
    # Try to set content/body
    if items:
        body_placeholder = find_placeholder(slide, ['BODY', 'OBJECT', 'CONTENT'])
        if body_placeholder and body_placeholder.has_text_frame:
            tf = body_placeholder.text_frame
            tf.clear()  # Clear existing content
            
            for i, item in enumerate(items):
                if i > 0:
                    p = tf.add_paragraph()
                else:
                    p = tf.paragraphs[0] if tf.paragraphs else tf.add_paragraph()
                
                # Handle different item formats
                if isinstance(item, dict):
                    p.text = item.get('text', str(item))
                    p.level = item.get('level', 0)
                else:
                    p.text = str(item)
                    p.level = 0
        else:
            # Fallback: add textbox if no body placeholder
            # Use adjusted content_area if image was added
            if content_area:
                content_box = slide.shapes.add_textbox(
                    content_area['left'], content_area['top'],
                    content_area['width'], content_area['height']
                )
            else:
                content_box = slide.shapes.add_textbox(
                    Inches(0.5), Inches(1.5), 
                    prs.slide_width - Inches(1), Inches(5.5)
                )
            tf = content_box.text_frame
            tf.word_wrap = True
            
            for i, item in enumerate(items):
                if i > 0:
                    p = tf.add_paragraph()
                else:
                    p = tf.paragraphs[0]
                
                if isinstance(item, dict):
                    p.text = f"• {item.get('text', str(item))}"
                else:
                    p.text = f"• {str(item)}"
                p.font.size = Pt(18)
    
    # Add code block if provided
    if code:
        # Get slide dimensions for dynamic positioning
        slide_height = prs.slide_height.inches
        slide_width = prs.slide_width.inches
        
        # Determine code block position (below content or standalone)
        if items:
            # Position code below the content area
            # For smaller slides, adjust proportionally
            code_top = Inches(slide_height * 0.6)  # 60% down from top
            code_height = Inches(slide_height * 0.32)  # 32% of slide height
        else:
            # Code-only slide - larger code area
            code_top = Inches(1.5)
            code_height = Inches(slide_height - 2.0)  # Leave margin at bottom
        
        # Calculate code block width (consider image if present)
        if image_config and image_config.get('position') == 'right':
            width_pct = image_config.get('width_percent', 40)
            code_width = prs.slide_width * (1 - width_pct / 100) - Inches(0.7)
        else:
            code_width = prs.slide_width - Inches(1.0)
        
        code_box = slide.shapes.add_textbox(
            Inches(0.5), code_top,
            code_width, code_height
        )
        tf = code_box.text_frame
        tf.word_wrap = True
        
        # Style as code block
        p = tf.paragraphs[0]
        p.text = code
        p.font.name = 'Consolas'
        p.font.size = Pt(11)
        
        # Add dark background to code box
        code_box.fill.solid()
        code_box.fill.fore_color.rgb = RGBColor(40, 44, 52)  # Dark background
        p.font.color.rgb = RGBColor(171, 178, 191)  # Light gray text
    
    # Final cleanup: remove empty placeholders and textboxes
    # This catches empty body/content placeholders in Two Column layouts
    # where only one column is used (e.g., content + image)
    removed_body = remove_empty_body_placeholders(slide)
    if removed_body > 0:
        print(f"    [i] Removed {removed_body} empty body placeholder(s)")
    
    removed_textboxes = remove_empty_textboxes(slide)
    if removed_textboxes > 0:
        print(f"    [i] Removed {removed_textboxes} empty textbox(es)")
    
    return slide


def validate_and_fix_content(slides_data: list, strict: bool = False) -> tuple[list, bool]:
    """
    Validate content and auto-fix common issues.
    - closing type with items → convert to content type
    - title type with items → keep items (subtitle might also exist)
    - content/agenda slides without items → ERROR (strict mode) or WARNING
    
    Args:
        slides_data: List of slide data dictionaries
        strict: If True, raise error on critical issues (empty content slides)
    
    Returns:
        tuple: (fixed slides_data, has_errors)
    """
    fixed_count = 0
    errors = []
    warnings = []
    
    for i, slide in enumerate(slides_data):
        slide_type = slide.get('type', 'content')
        # Support multiple content key names: items, content, bullets
        items = slide.get('items', slide.get('content', slide.get('bullets', [])))
        title = slide.get('title', f'Slide {i + 1}')
        
        # Check: closing type should not have multiple items
        if slide_type == 'closing' and items and len(items) > 1:
            warnings.append(f"  [!] Warning: Slide {i + 1} '{title[:30]}...' has type='closing' with {len(items)} items")
            warnings.append(f"      → Auto-converting to type='content' (closing is for short endings only)")
            slide['type'] = 'content'
            fixed_count += 1
        
        # Check: agenda type should have items
        if slide_type == 'agenda' and not items:
            warnings.append(f"  [!] Warning: Slide {i + 1} '{title[:30]}...' has type='agenda' but no items")
        
        # CRITICAL: content slides must have items
        if slide_type == 'content' and not items:
            errors.append(f"  [X] ERROR: Slide {i + 1} '{title[:30]}...' has type='content' but NO CONTENT!")
            errors.append(f"      Check if 'items', 'content', or 'bullets' key exists in JSON")
        
        # Check: title type with many items might be better as content
        if slide_type == 'title' and items and len(items) > 2:
            warnings.append(f"  ℹ️  Note: Slide {i + 1} '{title[:30]}...' has type='title' with {len(items)} items")
            warnings.append(f"      Consider using type='content' for better display")
    
    # Print warnings
    for w in warnings:
        print(w)
    
    # Print errors
    for e in errors:
        print(e)
    
    if fixed_count > 0:
        print(f"\n  ✅ Auto-fixed {fixed_count} slide(s)")
    
    has_errors = len(errors) > 0
    if has_errors:
        print(f"\n  [!] Found {len(errors) // 2} critical error(s)!")
        if strict:
            print(f"  [?] Hint: content.json may use wrong key name (try 'items' or 'bullets')")
    
    print()
    return slides_data, has_errors


def create_pptx_from_template(template_path: str, content_path: str, output_path: str, config_path: Optional[str] = None, force: bool = False, add_signature: bool = True) -> None:
    """Create a new PPTX from template and content JSON.
    
    Args:
        add_signature: Add repository signature to first/last slide notes (default: True)
    """
    
    # Load template
    print(f"\nLoading template: {template_path}")
    prs = Presentation(template_path)
    
    # Show layout info
    print(f"\n{'─'*50}")
    print("Available Layouts:")
    for idx, layout in enumerate(prs.slide_layouts):
        print(f"  [{idx}] {layout.name or '(unnamed)'}")
    print(f"{'─'*50}\n")
    
    # Remove existing slides (start fresh with just the master)
    while len(prs.slides) > 0:
        rId = prs.slides._sldIdLst[0].rId
        prs.part.drop_rel(rId)
        del prs.slides._sldIdLst[0]
    print("Cleared existing slides from template.\n")
    
    # Load content
    print(f"Loading content: {content_path}")
    with open(content_path, 'r', encoding='utf-8') as f:
        data = json.load(f)
    
    slides_data = data.get('slides', data) if isinstance(data, dict) else data
    if isinstance(slides_data, dict):
        slides_data = [slides_data]
    
    # Validate and auto-fix content issues
    print("[*] Validating content...")
    slides_data, has_critical_errors = validate_and_fix_content(slides_data)
    
    if has_critical_errors and not force:
        print("\n[!] CRITICAL: Content validation failed!")
        print("   Found slides with type='content' but no items/bullets.")
        print("   Please fix the content.json before generating PPTX.")
        print("   (Use --force to ignore this check)")
        sys.exit(1)
    elif has_critical_errors and force:
        print("\n[!] WARNING: Content validation found issues but continuing due to --force flag.")
    
    print(f"Creating {len(slides_data)} slides...\n")
    
    # Load layout mapping from config file if provided
    layout_map = None
    if config_path and Path(config_path).exists():
        print(f"[*] Using layout config: {config_path}")
        with open(config_path, 'r', encoding='utf-8') as f:
            config = json.load(f)
        layout_map = config.get('layout_mapping', {})
        print(f"   Loaded mappings: {list(layout_map.keys())}\n")
    
    # Auto-detect layouts if no config provided
    if not layout_map:
        print("[*] Auto-detecting layouts from template names...")
        
        # Smart layout detection - find best matching layout by name
        def find_layout_by_keywords(layouts, keywords: list, fallback: int = 0) -> int:
            """Find layout index by matching keywords in layout name."""
            for keyword in keywords:
                for idx, layout in enumerate(layouts):
                    name = (layout.name or '').lower()
                    if keyword.lower() in name:
                        return idx
            return fallback
        
        # Detect layout indices based on template
        title_layout = find_layout_by_keywords(prs.slide_layouts, 
            ['title slide', 'タイトル スライド', 'タイトルスライド'], 0)
        content_layout = find_layout_by_keywords(prs.slide_layouts, 
            ['title and content', 'タイトルとコンテンツ', 'content'], 1)
        section_layout = find_layout_by_keywords(prs.slide_layouts, 
            ['section title', 'section', 'セクション'], content_layout)
        agenda_layout = find_layout_by_keywords(prs.slide_layouts, 
            ['agenda', 'アジェンダ'], content_layout)
        closing_layout = find_layout_by_keywords(prs.slide_layouts, 
            ['closing', 'クロージング', 'end'], section_layout)
        title_only_layout = find_layout_by_keywords(prs.slide_layouts, 
            ['title only', 'タイトルのみ', 'blank'], section_layout)
        blank_layout = find_layout_by_keywords(prs.slide_layouts, 
            ['blank', '白紙', 'empty'], title_only_layout)
        photo_layout = find_layout_by_keywords(prs.slide_layouts, 
            ['picture', 'photo', '50/50', 'image'], content_layout)
        
        layout_map = {
            'title': title_layout,
            'content': content_layout,
            'section': section_layout,
            'section_title': section_layout,
            'two_content': content_layout,
            'two_column': content_layout,
            'comparison': content_layout,
            'blank': blank_layout,
            'agenda': agenda_layout,
            'summary': content_layout,      # まとめ → コンテンツスタイル
            'next_steps': content_layout,   # 今後のステップ
            'feature': content_layout,
            'closing': closing_layout,      # 最終スライド → クロージング
            'code': content_layout,
            'photo': photo_layout,
            'title_only': title_only_layout,  # コンテンツなしスライド用
        }
        
        print(f"   Auto-detected: title={title_layout}, content={content_layout}, section={section_layout}, title_only={title_only_layout}\n")
        print("[?] Tip: Run 'python scripts/analyze_template.py <template.pptx>' for accurate layout mapping.\n")
    
    # Create slides
    first_slide = None
    last_slide = None
    for i, slide_data in enumerate(slides_data):
        slide_type = slide_data.get('type', 'content')
        title = slide_data.get('title', f'Slide {i + 1}')
        
        # Support multiple content key names: items, content, bullets
        # Also merge left_items + right_items for two_column slides
        items = slide_data.get('items', slide_data.get('content', slide_data.get('bullets', [])))
        if not items:
            left_items = slide_data.get('left_items', [])
            right_items = slide_data.get('right_items', [])
            if left_items or right_items:
                items = left_items + right_items
                print(f"    [i] Merged left_items + right_items → {len(items)} items")
        
        subtitle = slide_data.get('subtitle', '')
        has_image = slide_data.get('image') is not None
        
        # Determine layout index (priority: slide.layout > optimal selection > layout_map > fallback)
        layout_idx = slide_data.get('layout')
        if layout_idx is None:
            # Smart layout selection based on content amount
            base_layout = layout_map.get(slide_type, layout_map.get('content', 1))
            
            # If content-type slide has no items and no image, use simpler layout
            if slide_type == 'content' and not items and not has_image:
                # Prefer title_only or blank layout to avoid empty body placeholders
                simpler_layout = layout_map.get('title_only') or layout_map.get('blank')
                if simpler_layout is not None:
                    layout_idx = simpler_layout
                    print(f"    [i] No content → using simpler layout [{simpler_layout}]")
                else:
                    layout_idx = base_layout
            # Content slides with image should use two-column or content_with_image layout
            elif slide_type == 'content' and has_image:
                # Prefer content_with_image > two_column > content
                image_layout = layout_map.get('content_with_image') or layout_map.get('two_column')
                if image_layout is not None:
                    layout_idx = image_layout
                    print(f"    [i] Content + image → using layout [{layout_idx}]")
                else:
                    layout_idx = base_layout
            # Photo slides use photo layout for proper image placement
            elif slide_type == 'photo' and has_image:
                photo_layout = layout_map.get('photo')
                if photo_layout is not None:
                    layout_idx = photo_layout
                else:
                    layout_idx = base_layout
            else:
                layout_idx = base_layout
        
        # Ensure layout_idx is valid
        if layout_idx is None:
            layout_idx = 1
        layout_idx = min(layout_idx, len(prs.slide_layouts) - 1)
        
        layout_name = prs.slide_layouts[layout_idx].name or '(unnamed)'
        
        # Get optional image configuration
        image_config = slide_data.get('image')
        notes = slide_data.get('notes', '')
        code = slide_data.get('code', '')
        
        if image_config:
            print(f"  Slide {i + 1}: {title[:40]}... [Layout {layout_idx}: {layout_name}] 📷")
        elif code:
            print(f"  Slide {i + 1}: {title[:40]}... [Layout {layout_idx}: {layout_name}] 💻")
        else:
            print(f"  Slide {i + 1}: {title[:40]}... [Layout {layout_idx}: {layout_name}]")
        
        slide = add_slide_from_layout(prs, layout_idx, title, items, subtitle, image_config, content_path, code, slide_type)
        
        # Add speaker notes if provided
        if notes and slide:
            notes_slide = slide.notes_slide
            notes_tf = notes_slide.notes_text_frame
            notes_tf.text = notes
        
        # Store first and last slides for signature
        if i == 0:
            first_slide = slide
        last_slide = slide
    
    # Add signature to first and last slides (unless --no-signature)
    if add_signature and len(prs.slides) > 0:
        repo_url = get_repo_info()
        sig_first = f"📌 Generated by: {repo_url}"
        sig_last = f"---\n🔧 This presentation was created using Ag-ppt-create\n{repo_url}"
        
        # First slide
        if first_slide:
            try:
                notes_slide = first_slide.notes_slide
                notes_tf = notes_slide.notes_text_frame
                existing = notes_tf.text or ""
                if sig_first not in existing:
                    notes_tf.text = f"{sig_first}\n\n{existing}" if existing else sig_first
            except Exception:
                pass
        
        # Last slide
        if last_slide and (len(prs.slides) == 1 or last_slide != first_slide):
            try:
                notes_slide = last_slide.notes_slide
                notes_tf = notes_slide.notes_text_frame
                existing = notes_tf.text or ""
                if sig_last not in existing:
                    notes_tf.text = f"{existing}\n\n{sig_last}" if existing else sig_last
            except Exception:
                pass
        
        print(f"   📝 Signature added to speaker notes")
    
    # Set view to Normal (slide view) instead of Slide Master view
    # This prevents PowerPoint from opening in Master view
    # Note: python-pptx doesn't fully support viewProps modification,
    # so we'll fix it after saving using ZIP manipulation
    
    # Save first
    Path(output_path).parent.mkdir(parents=True, exist_ok=True)
    prs.save(output_path)
    
    # Post-save: Fix viewProps.xml to ensure Normal view
    _normalize_view_settings(output_path)
    
    print(f"\n✅ Created: {output_path}")
    print(f"   Total slides: {len(prs.slides)}")


def _auto_clean_template(template_path: str) -> str:
    """
    Automatically clean template if it has issues (dark backgrounds, decorations, etc.).
    Creates a cleaned version in output_manifest/ and returns the path.
    
    Args:
        template_path: Path to original template
    
    Returns:
        Path to cleaned template (or original if no cleaning needed)
    """
    import subprocess
    
    template_stem = Path(template_path).stem
    clean_path = f"output_manifest/{template_stem}_auto_clean.pptx"
    
    # Run diagnose to check if cleaning is needed
    try:
        result = subprocess.run(
            ["python", "scripts/diagnose_template.py", template_path, "--json"],
            capture_output=True, text=True, timeout=60
        )
        
        if result.returncode == 0:
            import json
            diag = json.loads(result.stdout)
            issues = diag.get("total_issues", 0)
            
            if issues == 0:
                print(f"[i] Template is clean, no auto-cleaning needed")
                return template_path
            
            print(f"[!] Template has {issues} issues, auto-cleaning...")
    except Exception as e:
        print(f"[!] Diagnose failed ({e}), proceeding with auto-clean")
    
    # Run clean_template
    try:
        result = subprocess.run(
            ["python", "scripts/create_clean_template.py", template_path, clean_path, "--all"],
            capture_output=True, text=True, timeout=120
        )
        
        if result.returncode == 0 and Path(clean_path).exists():
            print(f"[+] Auto-cleaned template: {clean_path}")
            
            # Also run analyze_template for the clean version
            subprocess.run(
                ["python", "scripts/analyze_template.py", clean_path],
                capture_output=True, text=True, timeout=60
            )
            
            return clean_path
        else:
            print(f"[!] Auto-clean failed, using original template")
            if result.stderr:
                print(f"    Error: {result.stderr[:200]}")
            return template_path
    except Exception as e:
        print(f"[!] Auto-clean failed ({e}), using original template")
        return template_path


def main():
    parser = argparse.ArgumentParser(
        description='Create PPTX from template using slide master layouts.',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog='''
Examples:
  # List available layouts in template:
  python scripts/create_from_template.py templates/sample.pptx --list-layouts

  # Analyze template and generate config (recommended):
  python scripts/analyze_template.py templates/sample-ppf.pptx

  # Create PPTX with layout config:
  python scripts/create_from_template.py templates/sample-ppf.pptx content.json output.pptx \\
      --config output_manifest/sample-ppf_layouts.json

  # Auto-clean template before generation (removes dark backgrounds, decorations):
  python scripts/create_from_template.py input/source.pptx content.json output.pptx --auto-clean

  # Auto-detect layouts (no config):
  python scripts/create_from_template.py templates/sample.pptx content.json output.pptx
'''
    )
    
    parser.add_argument('template', help='Path to template PPTX file')
    parser.add_argument('content', nargs='?', help='Path to content JSON file')
    parser.add_argument('output', nargs='?', help='Path to output PPTX file')
    parser.add_argument('--config', '-c', help='Path to layout config JSON (from analyze_template.py)')
    parser.add_argument('--list-layouts', '-l', action='store_true', help='List available layouts and exit')
    parser.add_argument('--force', '-f', action='store_true', help='Force generation even if validation fails')
    parser.add_argument('--no-signature', action='store_true', help='Disable repository signature in speaker notes')
    parser.add_argument('--auto-clean', action='store_true', 
                        help='Auto-clean template (remove dark backgrounds, decorations, fix positions)')
    
    args = parser.parse_args()
    
    # List layouts mode
    if args.list_layouts or not args.content:
        list_layouts(args.template)
        return
    
    # Validate arguments
    if not args.output:
        parser.error("Output path is required")
    
    # Auto-clean template if requested
    template_path = args.template
    if args.auto_clean:
        template_path = _auto_clean_template(args.template)
    
    # Auto-find config if not specified
    config_path = args.config
    if not config_path:
        # Try to find matching config file (check cleaned template first)
        template_stem = Path(template_path).stem
        auto_config = Path(f"output_manifest/{template_stem}_layouts.json")
        if auto_config.exists():
            config_path = str(auto_config)
            print(f"[+] Auto-found config: {config_path}")
        else:
            # Try original template name
            orig_stem = Path(args.template).stem
            auto_config = Path(f"output_manifest/{orig_stem}_layouts.json")
            if auto_config.exists():
                config_path = str(auto_config)
                print(f"[+] Auto-found config: {config_path}")
    
    create_pptx_from_template(
        template_path, args.content, args.output, config_path,
        force=args.force, add_signature=not args.no_signature
    )


if __name__ == '__main__':
    main()
