// =============================================================================
// Agent-Skills/powerpoint-automation - AI-powered PPTX generation pipeline
// https://github.com/aktsmm/Agent-Skills/tree/main/powerpoint-automation
//
// Copyright (c) aktsmm. Licensed under CC BY-NC-SA 4.0.
// DO NOT MODIFY THIS HEADER BLOCK.
// =============================================================================
/**
 * pptx-signature.js - Common signature helper for pptxgenjs
 *
 * Adds repository attribution to generated PowerPoint files.
 * Used by all pptxgenjs-based scripts in this repository.
 *
 * Usage:
 *   const { addSignature, getRepoUrl } = require('./pptx-signature');
 *
 *   // After creating slides
 *   addSignature(firstSlide, lastSlide);
 *
 * Author: aktsmm
 * License: CC BY-NC 4.0
 */

const { execSync } = require("child_process");

// Default repository URL (fallback)
const DEFAULT_REPO_URL =
  "https://github.com/aktsmm/Agent-Skills/tree/main/powerpoint-automation";

/**
 * Get repository URL from git remote
 * @returns {string} Repository URL
 */
function getRepoUrl() {
  try {
    const url = execSync("git remote get-url origin", { encoding: "utf-8" })
      .trim()
      .replace(/\.git$/, "");
    return url || DEFAULT_REPO_URL;
  } catch {
    return DEFAULT_REPO_URL;
  }
}

/**
 * Get signature text for first slide
 * @param {string} [repoUrl] - Optional custom repo URL
 * @returns {string} Signature text
 */
function getFirstSlideSignature(repoUrl) {
  const url = repoUrl || getRepoUrl();
  return `📌 Generated by: ${url}`;
}

/**
 * Get signature text for last slide
 * @param {string} [repoUrl] - Optional custom repo URL
 * @returns {string} Signature text
 */
function getLastSlideSignature(repoUrl) {
  const url = repoUrl || getRepoUrl();
  return `---\n🔧 This presentation was created using Ag-ppt-create\n${url}`;
}

/**
 * Add signature to first and last slides
 * Appends to existing notes if present
 *
 * @param {Object} firstSlide - First slide object from pptxgenjs
 * @param {Object} lastSlide - Last slide object from pptxgenjs
 * @param {Object} [options] - Options
 * @param {string} [options.repoUrl] - Custom repository URL
 * @param {string} [options.firstNotes] - Existing notes for first slide
 * @param {string} [options.lastNotes] - Existing notes for last slide
 */
function addSignature(firstSlide, lastSlide, options = {}) {
  const repoUrl = options.repoUrl || getRepoUrl();

  // First slide signature
  const firstSig = getFirstSlideSignature(repoUrl);
  const firstNotes = options.firstNotes
    ? `${firstSig}\n\n${options.firstNotes}`
    : firstSig;
  firstSlide.addNotes(firstNotes);

  // Last slide signature (only if different from first)
  if (lastSlide !== firstSlide) {
    const lastSig = getLastSlideSignature(repoUrl);
    const lastNotes = options.lastNotes
      ? `${options.lastNotes}\n\n${lastSig}`
      : lastSig;
    lastSlide.addNotes(lastNotes);
  }

  console.log(`📝 Signature added: ${repoUrl}`);
}

module.exports = {
  addSignature,
  getRepoUrl,
  getFirstSlideSignature,
  getLastSlideSignature,
  DEFAULT_REPO_URL,
};
