# vim-ai Quick Reference Guide

This guide provides quick patterns for common vim-ai configuration tasks.

## Common Configuration Tasks

### 1. Adding a Custom Role

**File to modify**: `~/.vim/vim-ai-roles.ini` (check [file-paths.md](file-paths.md) for exact location)

**Pattern**:
```ini
[role-name]
prompt=Your instruction here
```

**Steps**:
1. Read the current roles file
2. Use Edit tool to append new role
3. Verify role name is unique
4. Test with `:AI /role-name`

### 2. Changing AI Model

**File to modify**: `~/.vimrc` (check [file-paths.md](file-paths.md))

**Configuration section**: Look for `g:vim_ai_chat`, `g:vim_ai_edit`, or `g:vim_ai_complete`

**Steps**:
1. Fetch current model options from README
2. Read existing vimrc
3. Edit the `"model"` value in the appropriate section
4. Restart Vim to apply changes

### 3. Updating API Key

**File to modify**: `~/.config/openai.token` (or custom path in vimrc)

**Steps**:
1. Check vimrc for `g:vim_ai_token_file_path` (if set)
2. Write new API key to the token file
3. Ensure file has 600 permissions: `chmod 600 ~/.config/openai.token`
4. No Vim restart needed for key updates

### 4. Adding Keyboard Shortcuts

**File to modify**: `~/.vimrc`

**Pattern for normal mode**:
```vim
nnoremap <leader>x :AICommand<CR>
```

**Pattern for visual mode**:
```vim
xnoremap <leader>x :AICommand<CR>
```

**Steps**:
1. Read existing vimrc
2. Find keybindings section (or create one)
3. Add new mapping without conflicting with existing keys
4. Restart Vim or source config: `:source ~/.vimrc`

### 5. Configuring Custom Endpoint

**File to modify**: `~/.vimrc`

**Section to edit**: `g:vim_ai_chat["options"]["endpoint_url"]`

**Steps**:
1. Fetch endpoint documentation for the provider
2. Read current vimrc configuration
3. Update `endpoint_url` value
4. May need to adjust authentication settings
5. Test connection

### 6. Adjusting Temperature/Parameters

**File to modify**: `~/.vimrc`

**Common parameters**:
- `temperature`: 0.0-2.0 (creativity level)
- `max_tokens`: Response length limit
- `request_timeout`: Seconds before timeout

**Steps**:
1. Read current parameter values
2. Edit specific parameter in `options` section
3. Restart Vim to apply

## Configuration Sections in .vimrc

### Main Configuration Variables

```vim
" API Key location
let g:vim_ai_token_file_path = '~/.config/openai.token'

" Chat configuration
let g:vim_ai_chat = { ... }

" Edit configuration
let g:vim_ai_edit = { ... }

" Completion configuration
let g:vim_ai_complete = { ... }

" Custom roles file
let g:vim_ai_roles_config_file = expand('~/.vim/vim-ai-roles.ini')
```

### Structure of Configuration Dictionaries

Each configuration (chat, edit, complete) follows this structure:
```vim
let g:vim_ai_[type] = {
\  "engine": "chat" or "complete",
\  "options": {
\    "model": "gpt-4-turbo-preview",
\    "endpoint_url": "https://api.openai.com/v1/...",
\    "max_tokens": 0,
\    "temperature": 1,
\    "request_timeout": 20,
\    "enable_auth": 1,
\  },
\  "ui": {
\    "paste_mode": 1,
\  },
\}
```

## Troubleshooting Patterns

### Issue: vim-ai commands not found
1. Check plugin installation path exists
2. Verify Python 3 support in Vim: `vim --version | grep python3`
3. Check for error messages: `:messages`

### Issue: API authentication errors
1. Verify token file exists and has correct permissions
2. Check token file path in vimrc
3. Ensure API key is valid and not expired

### Issue: Role not working
1. Verify role syntax in .ini file
2. Check role name doesn't have spaces or special characters
3. Test with `:AI /role-name` (note the `/` prefix)

## Best Practices

1. **Always backup before editing**: Copy current config before making changes
2. **Edit, don't replace**: Use targeted edits rather than rewriting entire sections
3. **Test incrementally**: Make one change at a time and test
4. **Check documentation**: Fetch latest docs for new features or changes
5. **Verify file paths**: Confirm files exist before attempting modifications

## Workflow Template

For any configuration task:

1. **Identify** which file needs modification (use [file-paths.md](file-paths.md))
2. **Fetch** current documentation (use [documentation-sources.md](documentation-sources.md))
3. **Read** existing configuration file
4. **Edit** specific section using Edit tool
5. **Verify** syntax is correct
6. **Test** the change in Vim
7. **Document** what was changed for user reference
