---
name: senior-qa
description: This skill should be used when the user asks to "generate tests", "write unit tests", "analyze test coverage", "scaffold E2E tests", "set up Playwright", "configure Jest", "implement testing patterns", or "improve test quality". Use for React/Next.js testing with Jest, React Testing Library, and Playwright.
---

# Senior QA Engineer

Test automation, coverage analysis, and quality assurance patterns for React and Next.js applications.

## Table of Contents

- [Quick Start](#quick-start)
- [Tools Overview](#tools-overview)
  - [Test Suite Generator](#1-test-suite-generator)
  - [Coverage Analyzer](#2-coverage-analyzer)
  - [E2E Test Scaffolder](#3-e2e-test-scaffolder)
- [QA Workflows](#qa-workflows)
  - [Unit Test Generation Workflow](#unit-test-generation-workflow)
  - [Coverage Analysis Workflow](#coverage-analysis-workflow)
  - [E2E Test Setup Workflow](#e2e-test-setup-workflow)
- [Reference Documentation](#reference-documentation)
- [Common Patterns Quick Reference](#common-patterns-quick-reference)

---

## Quick Start

```bash
# Generate Jest test stubs for React components
python scripts/test_suite_generator.py src/components/ --output __tests__/

# Analyze test coverage from Jest/Istanbul reports
python scripts/coverage_analyzer.py coverage/coverage-final.json --threshold 80

# Scaffold Playwright E2E tests for Next.js routes
python scripts/e2e_test_scaffolder.py src/app/ --output e2e/
```

---

## Tools Overview

### 1. Test Suite Generator

Scans React/TypeScript components and generates Jest + React Testing Library test stubs with proper structure.

**Input:** Source directory containing React components
**Output:** Test files with describe blocks, render tests, interaction tests

**Usage:**
```bash
# Basic usage - scan components and generate tests
python scripts/test_suite_generator.py src/components/ --output __tests__/

# Output:
# Scanning: src/components/
# Found 24 React components
#
# Generated tests:
#   __tests__/Button.test.tsx (render, click handler, disabled state)
#   __tests__/Modal.test.tsx (render, open/close, keyboard events)
#   __tests__/Form.test.tsx (render, validation, submission)
#   ...
#
# Summary: 24 test files, 87 test cases

# Include accessibility tests
python scripts/test_suite_generator.py src/ --output __tests__/ --include-a11y

# Generate with custom template
python scripts/test_suite_generator.py src/ --template custom-template.tsx
```

**Supported Patterns:**
- Functional components with hooks
- Components with Context providers
- Components with data fetching
- Form components with validation

---

### 2. Coverage Analyzer

Parses Jest/Istanbul coverage reports and identifies gaps, uncovered branches, and provides actionable recommendations.

**Input:** Coverage report (JSON or LCOV format)
**Output:** Coverage analysis with recommendations

**Usage:**
```bash
# Analyze coverage report
python scripts/coverage_analyzer.py coverage/coverage-final.json

# Output:
# === Coverage Analysis Report ===
# Overall: 72.4% (target: 80%)
#
# BY TYPE:
#   Statements: 74.2%
#   Branches: 68.1%
#   Functions: 71.8%
#   Lines: 73.5%
#
# CRITICAL GAPS (uncovered business logic):
#   src/services/payment.ts:45-67 - Payment processing
#   src/hooks/useAuth.ts:23-41 - Authentication flow
#
# RECOMMENDATIONS:
#   1. Add tests for payment service error handling
#   2. Cover authentication edge cases
#   3. Test form validation branches
#
# Files below threshold (80%):
#   src/components/Checkout.tsx: 45%
#   src/services/api.ts: 62%

# Enforce threshold (exit 1 if below)
python scripts/coverage_analyzer.py coverage/ --threshold 80 --strict

# Generate HTML report
python scripts/coverage_analyzer.py coverage/ --format html --output report.html
```

---

### 3. E2E Test Scaffolder

Scans Next.js pages/app directory and generates Playwright test files with common interactions.

**Input:** Next.js pages or app directory
**Output:** Playwright test files organized by route

**Usage:**
```bash
# Scaffold E2E tests for Next.js App Router
python scripts/e2e_test_scaffolder.py src/app/ --output e2e/

# Output:
# Scanning: src/app/
# Found 12 routes
#
# Generated E2E tests:
#   e2e/home.spec.ts (navigation, hero section)
#   e2e/auth/login.spec.ts (form submission, validation)
#   e2e/auth/register.spec.ts (registration flow)
#   e2e/dashboard.spec.ts (authenticated routes)
#   e2e/products/[id].spec.ts (dynamic routes)
#   ...
#
# Generated: playwright.config.ts
# Generated: e2e/fixtures/auth.ts

# Include Page Object Model classes
python scripts/e2e_test_scaffolder.py src/app/ --output e2e/ --include-pom

# Generate for specific routes
python scripts/e2e_test_scaffolder.py src/app/ --routes "/login,/dashboard,/checkout"
```

---

## QA Workflows

### Unit Test Generation Workflow

Use when setting up tests for new or existing React components.

**Step 1: Scan project for untested components**
```bash
python scripts/test_suite_generator.py src/components/ --scan-only
```

**Step 2: Generate test stubs**
```bash
python scripts/test_suite_generator.py src/components/ --output __tests__/
```

**Step 3: Review and customize generated tests**
```typescript
// __tests__/Button.test.tsx (generated)
import { render, screen, fireEvent } from '@testing-library/react';
import { Button } from '../src/components/Button';

describe('Button', () => {
  it('renders with label', () => {
    render(<Button>Click me</Button>);
    expect(screen.getByRole('button', { name: /click me/i })).toBeInTheDocument();
  });

  it('calls onClick when clicked', () => {
    const handleClick = jest.fn();
    render(<Button onClick={handleClick}>Click</Button>);
    fireEvent.click(screen.getByRole('button'));
    expect(handleClick).toHaveBeenCalledTimes(1);
  });

  // TODO: Add your specific test cases
});
```

**Step 4: Run tests and check coverage**
```bash
npm test -- --coverage
python scripts/coverage_analyzer.py coverage/coverage-final.json
```

---

### Coverage Analysis Workflow

Use when improving test coverage or preparing for release.

**Step 1: Generate coverage report**
```bash
npm test -- --coverage --coverageReporters=json
```

**Step 2: Analyze coverage gaps**
```bash
python scripts/coverage_analyzer.py coverage/coverage-final.json --threshold 80
```

**Step 3: Identify critical paths**
```bash
python scripts/coverage_analyzer.py coverage/ --critical-paths
```

**Step 4: Generate missing test stubs**
```bash
python scripts/test_suite_generator.py src/ --uncovered-only --output __tests__/
```

**Step 5: Verify improvement**
```bash
npm test -- --coverage
python scripts/coverage_analyzer.py coverage/ --compare previous-coverage.json
```

---

### E2E Test Setup Workflow

Use when setting up Playwright for a Next.js project.

**Step 1: Initialize Playwright (if not installed)**
```bash
npm init playwright@latest
```

**Step 2: Scaffold E2E tests from routes**
```bash
python scripts/e2e_test_scaffolder.py src/app/ --output e2e/
```

**Step 3: Configure authentication fixtures**
```typescript
// e2e/fixtures/auth.ts (generated)
import { test as base } from '@playwright/test';

export const test = base.extend({
  authenticatedPage: async ({ page }, use) => {
    await page.goto('/login');
    await page.fill('[name="email"]', 'test@example.com');
    await page.fill('[name="password"]', 'password');
    await page.click('button[type="submit"]');
    await page.waitForURL('/dashboard');
    await use(page);
  },
});
```

**Step 4: Run E2E tests**
```bash
npx playwright test
npx playwright show-report
```

**Step 5: Add to CI pipeline**
```yaml
# .github/workflows/e2e.yml
- name: Run E2E tests
  run: npx playwright test
- name: Upload report
  uses: actions/upload-artifact@v3
  with:
    name: playwright-report
    path: playwright-report/
```

---

## Reference Documentation

| File | Contains | Use When |
|------|----------|----------|
| `references/testing_strategies.md` | Test pyramid, testing types, coverage targets, CI/CD integration | Designing test strategy |
| `references/test_automation_patterns.md` | Page Object Model, mocking (MSW), fixtures, async patterns | Writing test code |
| `references/qa_best_practices.md` | Testable code, flaky tests, debugging, quality metrics | Improving test quality |

---

## Common Patterns Quick Reference

### React Testing Library Queries

```typescript
// Preferred (accessible)
screen.getByRole('button', { name: /submit/i })
screen.getByLabelText(/email/i)
screen.getByPlaceholderText(/search/i)

// Fallback
screen.getByTestId('custom-element')
```

### Async Testing

```typescript
// Wait for element
await screen.findByText(/loaded/i);

// Wait for removal
await waitForElementToBeRemoved(() => screen.queryByText(/loading/i));

// Wait for condition
await waitFor(() => {
  expect(mockFn).toHaveBeenCalled();
});
```

### Mocking with MSW

```typescript
import { rest } from 'msw';
import { setupServer } from 'msw/node';

const server = setupServer(
  rest.get('/api/users', (req, res, ctx) => {
    return res(ctx.json([{ id: 1, name: 'John' }]));
  })
);

beforeAll(() => server.listen());
afterEach(() => server.resetHandlers());
afterAll(() => server.close());
```

### Playwright Locators

```typescript
// Preferred
page.getByRole('button', { name: 'Submit' })
page.getByLabel('Email')
page.getByText('Welcome')

// Chaining
page.getByRole('listitem').filter({ hasText: 'Product' })
```

### Coverage Thresholds (jest.config.js)

```javascript
module.exports = {
  coverageThreshold: {
    global: {
      branches: 80,
      functions: 80,
      lines: 80,
      statements: 80,
    },
  },
};
```

---

## Common Commands

```bash
# Jest
npm test                           # Run all tests
npm test -- --watch                # Watch mode
npm test -- --coverage             # With coverage
npm test -- Button.test.tsx        # Single file

# Playwright
npx playwright test                # Run all E2E tests
npx playwright test --ui           # UI mode
npx playwright test --debug        # Debug mode
npx playwright codegen             # Generate tests

# Coverage
npm test -- --coverage --coverageReporters=lcov,json
python scripts/coverage_analyzer.py coverage/coverage-final.json
```
