# Conventional Folder Structure for Testing Artifacts

To maintain organization and reproducibility in playwright-cli testing, use this conventional folder structure for test sessions. This structure separates concerns, makes artifacts easy to find, and supports both manual and automated workflows. **Artifacts are relative to the current working directory (cwd) or repository root at runtime—run scripts from the test-session folder to keep paths simple and ephemeral.**

## Recommended Structure

```txt
[current-working-directory]/     # Root folder for a test session - set as cwd
├── scripts/                     # Bash scripts (.sh files) for test execution
│   ├── test.sh                  # Main test script
│   └── setup.sh                 # Optional: Environment setup script
├── screenshots/                 # Screenshots captured during tests
│   ├── 01-initial.png           # Step-by-step screenshots
│   ├── 02-after-interaction.png
│   └── ...
├── logs/                        # Log files from servers, clients, or test output
│   ├── client.log               # Client application logs
│   ├── server.log               # Server application logs
│   └── test.log                 # Optional: Test script output
└── .playwright-cli/             # Auto-generated by playwright-cli (do not modify)
    ├── page-*.png               # Temporary screenshots
    ├── profiles/                # Browser profiles (if configured)
    └── ...
```

## Folder Purposes

- **`scripts/`**: Store executable bash scripts. Use this for hybrid approach templates or custom test runners. Keep scripts version-controlled for reproducibility.
- **`screenshots/`**: Collect visual evidence of test states. Number files sequentially (e.g., `01-`, `02-`) for easy chronological review. Copy from `.playwright-cli/` to here for persistence.
- **`logs/`**: Capture output from background processes (servers, clients) and test scripts. Essential for debugging failures or verifying no errors occurred.
- **`.playwright-cli/`**: Managed by the CLI tool. Contains temporary files like screenshots and profiles. Do not commit this folder—it's auto-generated and can be large.

## Best Practices

- **Session-Based Organization**: Create one folder per test session (e.g., `quiz-test-2026-01-30/`). This isolates artifacts and prevents conflicts. **Run scripts from within this folder to make all paths relative to cwd.**
- **Version Control**: Commit `scripts/`, `screenshots/` (if meaningful), and `logs/` (if needed for debugging). Ignore `.playwright-cli/` in `.gitignore`.
- **Cleanup**: Use scripts to clean up artifacts between runs. For example, add `rm -rf screenshots/* logs/*` in setup.
- **Naming Conventions**: Use descriptive, date-stamped folder names (e.g., `login-flow-20260130`). Number screenshots and logs for sequence.
- **Path Variables**: In scripts, use relative paths like `./screenshots` (relative to cwd) to reference folders dynamically.
- **Integration with Hybrid Template**: The [HYBRID_TEMPLATE.md](HYBRID_TEMPLATE.md) assumes this structure—update paths if customizing.

## Example Usage in Script

```bash
# In test.sh (run from test-session/ folder)
SESSION_DIR="$(pwd)"  # Current working directory
SCREENSHOTS_DIR="$SESSION_DIR/screenshots"
LOGS_DIR="$SESSION_DIR/logs"

# Create directories if needed
mkdir -p "$SCREENSHOTS_DIR" "$LOGS_DIR"

# Capture logs
dotnet run > "$LOGS_DIR/server.log" 2>&1 &

# Copy screenshots
cp .playwright-cli/page-*.png "$SCREENSHOTS_DIR/"
```

This structure promotes clean, maintainable test artifacts and aligns with the hybrid approach for CLI-based E2E testing.