# Integration Guide: Task Grounding Validation
**Version**: 1.0.0
**Purpose**: How to integrate task grounding validation into your SpecKit workflow

---

## Quick Start (5 minutes)

### For Immediate Use (Manual Process)

1. **After tasks.md is generated**, run this checklist:

   ```bash
   # 1. Check artifacts exist
   ls specs/007-*/spec.md specs/007-*/plan.md specs/007-*/tasks.md

   # 2. Read the grounding validation checklist
   cat .claude/skills/speckit-task-grounding/VALIDATION_CHECKLIST.md

   # 3. Generate TASK_GROUNDING_ANALYSIS.md (manual review process)
   # [Use the template from SPECIFICATION.md]

   # 4. Review against checklist
   # [Check each item in VALIDATION_CHECKLIST.md]

   # 5. Approve or block
   # [Mark decision gate: Approved / Needs Clarification / Blocked]
   ```

2. **Review checklist** (10-15 min):
   - Run through "Report Validation Checklist"
   - Mark each item ✅
   - Document any ⚠️ or 🔴 findings

3. **Make decision**:
   - If ≥95% items checked → APPROVED
   - If ≥70% items checked → NEEDS CLARIFICATION
   - If <70% items checked → BLOCKED

---

## Step 1: Define Your Artifact Schema

**Create**: `.specify/config/artifact-schema.yaml`

```yaml
# Define which documents are required for task grounding
Features:
  - spec.md:
      Type: specification
      Required: true
      Weight: 1.0

  - plan.md:
      Type: plan
      Required: true
      Weight: 0.9

  - data-model.md:
      Type: design
      Required: false
      Weight: 0.7

  - api-contracts.md:
      Type: contract
      Required: false
      Weight: 0.7

  - research.md:
      Type: research
      Required: false
      Weight: 0.6

  - quickstart.md:
      Type: implementation
      Required: false
      Weight: 0.5

# Minimum coverage thresholds
MinimumCoverage:
  RequiredArtifacts: 2  # Must have spec.md + plan.md
  TotalArtifacts: 4     # Minimum 4 artifacts total
  CoveragePercentage: 71%

# Grounding thresholds by phase
GroundingThresholds:
  Phase1:
    MinLevel: 80%
    ApprovalPercentage: 90%  # ≥90% of tasks at 80%+

  Phase2:
    MinLevel: 70%
    ApprovalPercentage: 80%  # ≥80% of tasks at 70%+

  Phase3Plus:
    MinLevel: 60%
    ApprovalPercentage: 70%  # ≥70% of tasks at 60%+
```

---

## Step 2: Create Report Template

**Create**: `.specify/templates/task-grounding-template.md`

```markdown
# Task Grounding Analysis: [FEATURE_NAME]
**Feature**: [FEATURE_ID] | **Date**: [DATE]
**Status**: 🔄 In Review

---

## Executive Summary

[Generated by: Extract from tasks.md, count tasks per phase]

| Phase | Status | Count | Avg Grounding | Risk |
|-------|--------|-------|---------------|------|
| Phase 1 | ⏳ | [N] | [N]% | 🟢 |
| Phase 2 | ⏳ | [N] | [N]% | 🟡 |

---

## Phase [N]: [PHASE_NAME]

[For each task in tasks.md]

### Task T[NNN]: [Task Title]

**Artifact Mapping**:
- **Primary**: [artifact.md - Section]
- **Secondary**: [artifact2.md, artifact3.md]
- **Grounding Level**: [0-100%]

**Evidence**:
[Copy quote from artifact with context]

**Assessment**: [Fully/Partially/Weakly Grounded]
- ✅ [Positive finding]
- ⚠️ [Concern if any]

---

## Cross-Artifact Consistency

| Item | Artifact A | Artifact B | Match |
|------|-----------|-----------|-------|
| [Item] | [Value] | [Value] | ✅ |

---

## Critical Gaps

[List any gaps identified]

---

## Risk Assessment

[For each HIGH-risk task, explain why]

---

## Decision Gate

**Status**: ⏳ [APPROVED / NEEDS CLARIFICATION / BLOCKED]

[Reasoning]
```

---

## Step 3: Create Validation Script

**Create**: `.specify/scripts/powershell/validate-task-grounding.ps1`

```powershell
<#
.SYNOPSIS
    Validates that tasks.md is grounded in planning artifacts

.PARAMETER FeaturePath
    Path to feature directory (e.g., specs/007-lifeline-invitation-auto-role-mvp)

.PARAMETER OutputPath
    Path to output TASK_GROUNDING_ANALYSIS.md (optional)

.EXAMPLE
    ./validate-task-grounding.ps1 -FeaturePath "specs/007-lifeline-invitation-auto-role-mvp"
#>

param(
    [Parameter(Mandatory = $true)]
    [string]$FeaturePath,

    [Parameter(Mandatory = $false)]
    [string]$OutputPath = "$FeaturePath/TASK_GROUNDING_ANALYSIS.md"
)

# Validate artifacts exist
$artifacts = @(
    "spec.md",
    "plan.md",
    "data-model.md",
    "api-contracts.md",
    "research.md",
    "quickstart.md",
    "tasks.md"
)

$foundArtifacts = @()
$missingArtifacts = @()

foreach ($artifact in $artifacts) {
    $artifactPath = Join-Path $FeaturePath $artifact
    if (Test-Path $artifactPath) {
        $foundArtifacts += $artifact
        Write-Host "✅ Found: $artifact" -ForegroundColor Green
    } else {
        $missingArtifacts += $artifact
        Write-Host "⚠️  Missing: $artifact" -ForegroundColor Yellow
    }
}

# Parse tasks.md
$tasksPath = Join-Path $FeaturePath "tasks.md"
$tasksContent = Get-Content $tasksPath -Raw

# Extract all tasks using regex pattern: - [ ] T\d+ ... file path
$taskPattern = '- \[.\] T\d+.*?(?=\n-|\Z)'
$tasks = [regex]::Matches($tasksContent, $taskPattern, 'Multiline')

Write-Host "`n📋 Found $($tasks.Count) tasks in tasks.md"

# Output summary
Write-Host "`n📊 Artifact Coverage:"
Write-Host "  Found: $($foundArtifacts.Count)/7"
Write-Host "  Missing: $($missingArtifacts.Count)/7"
Write-Host "  Coverage: $([Math]::Round($foundArtifacts.Count/7*100))%"

Write-Host "`n⚙️  Next Steps:"
Write-Host "  1. Review VALIDATION_CHECKLIST.md"
Write-Host "  2. Generate TASK_GROUNDING_ANALYSIS.md using template"
Write-Host "  3. Cross-check each task against artifacts"
Write-Host "  4. Document gaps and risk assessment"
Write-Host "  5. Mark decision gate: Approved/Needs Clarification/Blocked"

if ($missingArtifacts.Count -gt 2) {
    Write-Host "`n⚠️  WARNING: More than 2 artifacts missing. Consider regenerating planning phase." -ForegroundColor Yellow
}
```

**Usage**:
```powershell
.specify/scripts/powershell/validate-task-grounding.ps1 -FeaturePath "specs/007-lifeline-invitation-auto-role-mvp"
```

---

## Step 4: Integrate into SpecKit Workflow

### Option A: Manual Decision Gate (Current Recommended)

Add this step to your existing workflow:

```
[speckit.tasks] Generate tasks.md
         ↓
[HUMAN REVIEW] Check tasks against VALIDATION_CHECKLIST.md
         ↓
    Decision: Approved?
    ├─ YES → Proceed to implementation
    └─ NO → Return to planning, regenerate
```

**Implementation**:
1. After tasks.md generated, run validation script
2. Use VALIDATION_CHECKLIST.md to review
3. Generate TASK_GROUNDING_ANALYSIS.md
4. Seek approval from project lead

---

### Option B: Automated Agent-Based (Future)

Create a new SpecKit agent:

```yaml
# In .specify/config/agents.yaml
Agents:
  - Name: "speckit.task-grounding"
    Description: "Validates task grounding in planning artifacts"
    Inputs:
      - tasks.md
      - spec.md
      - plan.md
      - data-model.md (optional)
      - api-contracts.md (optional)
    Outputs:
      - TASK_GROUNDING_ANALYSIS.md
    Config:
      Template: "task-grounding-template.md"
      Schema: "artifact-schema.yaml"
      ChecklistRef: "VALIDATION_CHECKLIST.md"
    Decision:
      Gate: "Block if Blocked, warn if Needs Clarification"
      ApprovalRequired: "Project Lead"
```

---

## Step 5: Create Decision Framework

**Create**: `.specify/config/task-grounding-decisions.yaml`

```yaml
DecisionFramework:
  APPROVED:
    When:
      - Phase1: "≥90% tasks at 80%+ grounding"
      - Phase2: "≥80% tasks at 70%+ grounding"
      - Gaps: "All high-risk gaps have mitigation"
      - Consistency: "No critical artifact inconsistencies"
    ConfidenceLevel: "🟢 80%+"
    NextStep: "Proceed to implementation"
    Approval: "Automatic (if all thresholds met)"

  NEEDS_CLARIFICATION:
    When:
      - Any: "Phase has <60% grounding in any task"
      - Or: "Moderate gaps with resolvable items"
      - Or: "Inconsistencies requiring clarification"
    ConfidenceLevel: "🟡 50-79%"
    NextStep: "Update artifacts, regenerate tasks.md"
    Approval: "Manual review + update needed"
    Rework: "Return to planning phase"

  BLOCKED:
    When:
      - Any: ">50% of tasks have <50% grounding"
      - Or: "Critical gaps preventing implementation"
      - Or: "Major artifact inconsistencies"
      - Or: "Unresolved planning questions"
    ConfidenceLevel: "🔴 <50%"
    NextStep: "Return to planning phase, resolve gaps, regenerate"
    Approval: "Must unblock manually"
    ReworkScope: "Major (spec/plan revision)"
```

---

## Step 6: Review Checklist Integration

**For Project Leads**:

```markdown
# Pre-Implementation Sign-Off Checklist

- [ ] I've read the spec.md for this feature
- [ ] I've read the plan.md for this feature
- [ ] I've reviewed TASK_GROUNDING_ANALYSIS.md
- [ ] I understand the identified gaps
- [ ] I'm comfortable with the risk assessment
- [ ] I approve the decision gate status
- [ ] Team is ready to implement

**Status**: ☐ Approved | ☐ Needs Clarification | ☐ Blocked

**Lead**: _______________ **Date**: _______ **Notes**: ______________
```

---

## Implementation Timeline

### Week 1: Setup
- [ ] Define artifact schema (artifact-schema.yaml)
- [ ] Create report template (task-grounding-template.md)
- [ ] Set up validation script (validate-task-grounding.ps1)

### Week 2: Pilot
- [ ] Apply to current feature (007-lifeline-invitation-auto-role-mvp)
- [ ] Generate analysis report
- [ ] Gather feedback from project lead

### Week 3: Refine
- [ ] Update checklist based on pilot findings
- [ ] Adjust grounding thresholds if needed
- [ ] Document lessons learned

### Week 4: Integrate
- [ ] Add to standard workflow documentation
- [ ] Train team on process
- [ ] Make it part of PR checklist

---

## Common Integration Points

### GitHub Actions (CI/CD)
```yaml
# .github/workflows/validate-tasks.yml
name: Validate Task Grounding
on: [pull_request]
jobs:
  validate:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      - name: Validate tasks grounding
        run: |
          pwsh .specify/scripts/powershell/validate-task-grounding.ps1 \
            -FeaturePath "specs/${{ github.head_ref }}"
```

### VS Code Task
```json
{
  "label": "Validate Task Grounding",
  "type": "shell",
  "command": "pwsh",
  "args": [
    ".specify/scripts/powershell/validate-task-grounding.ps1",
    "-FeaturePath",
    "${fileDirname}"
  ],
  "group": {
    "kind": "test",
    "isDefault": true
  }
}
```

### Pre-Commit Hook
```bash
#!/bin/bash
# .git/hooks/pre-commit
if [[ $(git diff --cached --name-only) == *"tasks.md"* ]]; then
  pwsh .specify/scripts/powershell/validate-task-grounding.ps1 \
    -FeaturePath "$(dirname $(git diff --cached --name-only | grep tasks.md))"
fi
```

---

## Troubleshooting

### Issue: Script can't find artifacts

**Cause**: Path not absolute or relative to correct directory

**Solution**:
```powershell
# Use absolute path
$FeaturePath = Resolve-Path $FeaturePath
```

### Issue: Too many tasks marked as "Weakly Grounded"

**Cause**: Artifacts incomplete or vague

**Solution**:
1. Verify all artifacts exist
2. Check if research.md has additional details
3. Consider: Should these tasks be in Phase 2 (foundational)?
4. If <60% grounding: Mark as NEEDS_CLARIFICATION

### Issue: High-risk gaps with no mitigation

**Cause**: Tasks invented without planning artifact grounding

**Solution**:
1. Remove task from tasks.md, or
2. Add explanation to plan.md justifying the task, or
3. Mark as BLOCKED and return to planning

---

## Maintenance & Updates

### When to Update This Process

- [ ] After first 3 features (gather feedback)
- [ ] When artifact types change
- [ ] When thresholds prove unrealistic
- [ ] Quarterly review of effectiveness

### Metrics to Track

- Average grounding level by phase
- % of features approved on first try
- % blocked due to planning gaps
- Time spent in validation (should be <30 min)

---

## Next Steps

1. **Copy SPECIFICATION.md and VALIDATION_CHECKLIST.md** to your `.claude/skills/speckit-task-grounding/` directory
2. **Customize artifact-schema.yaml** for your project
3. **Run validation script** on your current feature (007-lifeline-invitation-auto-role-mvp)
4. **Generate TASK_GROUNDING_ANALYSIS.md** using the template
5. **Get feedback** from project lead on process
6. **Refine** based on feedback
7. **Document** final process in your team wiki
