#!/usr/bin/env python3
"""
OnchainKit Environment Setup

Helps set up environment variables and configuration for OnchainKit projects.
"""

import os
import sys
from pathlib import Path
import json

def check_current_directory():
    """Check if we're in a valid OnchainKit project."""
    package_json = Path("package.json")
    
    if not package_json.exists():
        print("❌ No package.json found. Are you in a project directory?")
        return False
    
    try:
        with open(package_json) as f:
            data = json.load(f)
        
        # Check if OnchainKit is in dependencies
        deps = data.get("dependencies", {})
        dev_deps = data.get("devDependencies", {})
        
        if "@coinbase/onchainkit" not in deps and "@coinbase/onchainkit" not in dev_deps:
            print("⚠️  OnchainKit not found in dependencies. This might not be an OnchainKit project.")
            return False
        
        return True
    except json.JSONDecodeError:
        print("❌ Invalid package.json file")
        return False

def create_env_file():
    """Create or update .env.local file with OnchainKit configuration."""
    env_path = Path(".env.local")
    env_example_path = Path(".env.local.example")
    
    env_template = {
        "NEXT_PUBLIC_CDP_API_KEY": {
            "description": "Coinbase Developer Platform API Key",
            "required": True,
            "url": "https://portal.cdp.coinbase.com/",
            "default": "your-api-key-here"
        },
        "NEXT_PUBLIC_WC_PROJECT_ID": {
            "description": "WalletConnect Project ID",
            "required": True,
            "url": "https://cloud.walletconnect.com/",
            "default": "your-walletconnect-project-id"
        },
        "NEXT_PUBLIC_CHAIN_ID": {
            "description": "Chain ID (8453 = Base mainnet, 84532 = Base sepolia)",
            "required": False,
            "default": "8453"
        },
        "NEXT_PUBLIC_ONCHAINKIT_ANALYTICS": {
            "description": "Enable OnchainKit analytics",
            "required": False,
            "default": "true"
        }
    }
    
    # Read existing env file if it exists
    existing_env = {}
    if env_path.exists():
        with open(env_path, 'r') as f:
            for line in f:
                line = line.strip()
                if line and not line.startswith('#'):
                    if '=' in line:
                        key, value = line.split('=', 1)
                        existing_env[key] = value
    
    # Create/update env file
    with open(env_path, 'w') as f:
        f.write("# OnchainKit Environment Configuration\n")
        f.write("# Auto-generated by setup-environment.py\n\n")
        
        for key, config in env_template.items():
            f.write(f"# {config['description']}\n")
            if 'url' in config:
                f.write(f"# Get from: {config['url']}\n")
            
            # Use existing value if available, otherwise use default
            value = existing_env.get(key, config['default'])
            f.write(f"{key}={value}\n\n")
    
    # Also create example file
    with open(env_example_path, 'w') as f:
        f.write("# OnchainKit Environment Configuration Example\n")
        f.write("# Copy this to .env.local and fill in your actual values\n\n")
        
        for key, config in env_template.items():
            f.write(f"# {config['description']}\n")
            if 'url' in config:
                f.write(f"# Get from: {config['url']}\n")
            f.write(f"{key}={config['default']}\n\n")
    
    return env_path, env_example_path

def install_onchainkit():
    """Install OnchainKit if not already installed."""
    try:
        import subprocess
        result = subprocess.run(["npm", "ls", "@coinbase/onchainkit"], 
                              capture_output=True, text=True)
        
        if result.returncode != 0:
            print("📦 Installing @coinbase/onchainkit...")
            install_result = subprocess.run(["npm", "install", "@coinbase/onchainkit"], 
                                          capture_output=True, text=True)
            if install_result.returncode == 0:
                print("✅ OnchainKit installed successfully")
            else:
                print(f"❌ Failed to install OnchainKit: {install_result.stderr}")
                return False
        else:
            print("✅ OnchainKit is already installed")
        
        return True
    except Exception as e:
        print(f"❌ Error checking/installing OnchainKit: {e}")
        return False

def setup_typescript_config():
    """Add OnchainKit path aliases to tsconfig.json if it exists."""
    tsconfig_path = Path("tsconfig.json")
    
    if not tsconfig_path.exists():
        return
    
    try:
        with open(tsconfig_path, 'r') as f:
            content = f.read()
        
        # Simple check - don't modify if OnchainKit paths already exist
        if '"@coinbase/onchainkit' in content:
            return
        
        print("📝 TypeScript config looks good")
        
    except Exception as e:
        print(f"⚠️  Could not read tsconfig.json: {e}")

def print_next_steps():
    """Print helpful next steps for the user."""
    print("""
🎉 OnchainKit environment setup complete!

Next steps:
1. Edit .env.local and add your actual API keys:
   - CDP API Key: https://portal.cdp.coinbase.com/
   - WalletConnect Project ID: https://cloud.walletconnect.com/

2. Run your development server:
   npm run dev

3. Validate your setup:
   python validate-setup.py

4. Start building with OnchainKit components:
   import { Wallet, ConnectWallet } from '@coinbase/onchainkit/wallet';

Need help? Check the OnchainKit docs at https://onchainkit.xyz
""")

def main():
    print("🔧 Setting up OnchainKit environment...\n")
    
    # Check if we're in a valid project
    if not check_current_directory():
        sys.exit(1)
    
    # Install OnchainKit if needed
    if not install_onchainkit():
        print("⚠️  Continuing despite installation issues...")
    
    # Create environment files
    env_path, example_path = create_env_file()
    print(f"✅ Created {env_path}")
    print(f"✅ Created {example_path}")
    
    # Check TypeScript config
    setup_typescript_config()
    
    # Print next steps
    print_next_steps()

if __name__ == "__main__":
    main()