"""
Data Cleaning Skill - Comprehensive, Reusable Data Cleaning Framework

This module provides a modular, strategy-based approach to data cleaning for
tabular datasets. It supports deduplication, missing value imputation, text
processing, outlier handling, and pipeline orchestration.

Usage:
    from data_cleaning import CleaningStrategies, DataCleaningPipeline

    pipeline = DataCleaningPipeline(name="MyData")
    pipeline.add_step(CleaningStrategies.remove_duplicates, subset=['id'])
    pipeline.add_step(CleaningStrategies.drop_missing, columns=['required_col'])
    cleaned_df = pipeline.execute(raw_df)
"""

import pandas as pd
import numpy as np
from typing import Callable, Optional, List, Dict, Any
from sklearn.preprocessing import LabelEncoder
from sklearn.impute import SimpleImputer, KNNImputer
import re
import warnings
warnings.filterwarnings('ignore')


class CleaningStrategies:
    """Collection of modular data cleaning strategies"""

    # ==================== DEDUPLICATION STRATEGIES ====================

    @staticmethod
    def remove_duplicates(df: pd.DataFrame, subset: Optional[List[str]] = None) -> pd.DataFrame:
        """
        Remove duplicate rows based on all columns or specified subset.

        Args:
            df: Input dataframe
            subset: List of columns to consider for duplicates (None = all columns)

        Returns:
            DataFrame with duplicates removed
        """
        if subset:
            return df.drop_duplicates(subset=subset, keep='first')
        return df.drop_duplicates(keep='first')

    # ==================== MISSING VALUE STRATEGIES ====================

    @staticmethod
    def drop_missing(df: pd.DataFrame, columns: List[str]) -> pd.DataFrame:
        """
        Drop rows with missing values in specified columns.

        Args:
            df: Input dataframe
            columns: List of columns where missing values should trigger row removal

        Returns:
            DataFrame with rows containing missing values in specified columns removed
        """
        return df.dropna(subset=columns)

    @staticmethod
    def impute_median(df: pd.DataFrame, columns: List[str]) -> pd.DataFrame:
        """
        Impute missing values with median for numerical columns.

        Args:
            df: Input dataframe
            columns: List of numerical columns to impute

        Returns:
            DataFrame with median-imputed values
        """
        df = df.copy()
        imputer = SimpleImputer(strategy='median')
        for col in columns:
            if col in df.columns:
                df[col] = pd.to_numeric(df[col], errors='coerce')
                if pd.api.types.is_numeric_dtype(df[col]):
                    df[col] = imputer.fit_transform(df[[col]]).ravel()
        return df

    @staticmethod
    def impute_mode(df: pd.DataFrame, columns: List[str]) -> pd.DataFrame:
        """
        Impute missing values with mode (most frequent value) for categorical columns.

        Args:
            df: Input dataframe
            columns: List of categorical columns to impute

        Returns:
            DataFrame with mode-imputed values
        """
        df = df.copy()
        for col in columns:
            if col in df.columns and not df[col].dropna().empty:
                mode_val = df[col].mode()
                if len(mode_val) > 0:
                    df[col] = df[col].fillna(mode_val[0])
        return df

    @staticmethod
    def impute_knn(df: pd.DataFrame,
                   target_features: Dict[str, Dict[str, Any]],
                   n_neighbors: int = 5) -> pd.DataFrame:
        """
        KNN imputation with flexible feature selection.

        Args:
            df: Input dataframe
            target_features: Dictionary mapping target columns to their configuration
                Format: {
                    'target_col': {
                        'features': ['feature1', 'feature2'],
                        'type': 'numeric' | 'categorical' | 'binary'
                    }
                }
            n_neighbors: Number of neighbors for KNN

        Returns:
            DataFrame with KNN-imputed values

        Example:
            target_features = {
                'income': {'features': ['age', 'education'], 'type': 'numeric'},
                'has_children': {'features': ['age', 'marital_status'], 'type': 'binary'}
            }
        """
        df = df.copy()

        for target_col, config in target_features.items():
            if target_col not in df.columns:
                continue

            feature_cols = config.get('features', [])
            col_type = config.get('type', 'numeric')

            feature_cols = [c for c in feature_cols if c in df.columns]
            if not feature_cols:
                continue

            if not df[target_col].isna().any():
                continue

            work_cols = feature_cols + [target_col]
            df_work = df[work_cols].copy()

            # Encode feature columns
            feature_encoders = {}
            for col in feature_cols:
                numeric_vals = pd.to_numeric(df_work[col], errors='coerce')
                original_na_count = df_work[col].isna().sum()
                new_na_count = numeric_vals.isna().sum()

                if new_na_count > original_na_count:
                    le = LabelEncoder()
                    non_null_mask = df_work[col].notna()
                    if non_null_mask.sum() > 0:
                        df_work.loc[non_null_mask, col] = le.fit_transform(
                            df_work.loc[non_null_mask, col].astype(str)
                        )
                        feature_encoders[col] = le
                else:
                    df_work[col] = numeric_vals

            # Convert target column
            target_encoder = None
            if col_type == 'numeric':
                df_work[target_col] = pd.to_numeric(df_work[target_col], errors='coerce')
            elif col_type in ['categorical', 'binary']:
                le = LabelEncoder()
                non_null_mask = df_work[target_col].notna()
                if non_null_mask.sum() > 0:
                    df_work.loc[non_null_mask, target_col] = le.fit_transform(
                        df_work.loc[non_null_mask, target_col].astype(str)
                    )
                    target_encoder = le

            # Apply KNN imputation
            knn_imputer = KNNImputer(n_neighbors=n_neighbors)
            df_imputed = pd.DataFrame(
                knn_imputer.fit_transform(df_work),
                columns=df_work.columns,
                index=df_work.index
            )

            # Decode target column if needed
            if target_encoder is not None:
                df_imputed[target_col] = df_imputed[target_col].round().astype(int)
                df_imputed[target_col] = df_imputed[target_col].clip(
                    lower=0, upper=len(target_encoder.classes_) - 1
                )
                df_imputed[target_col] = target_encoder.inverse_transform(df_imputed[target_col])

            df[target_col] = df_imputed[target_col]

        return df

    # ==================== TEXT PROCESSING STRATEGIES ====================

    @staticmethod
    def process_text(df: pd.DataFrame, columns: List[str],
                     operation: str = 'extract_numbers') -> pd.DataFrame:
        """
        Apply text processing operations to specified columns.

        Args:
            df: Input dataframe
            columns: Columns to process
            operation: Type of text processing
                - 'extract_numbers': Extract first numeric value from text
                - 'clean_whitespace': Remove leading/trailing whitespace
                - 'extract_email': Extract email address from text
                - 'lowercase': Convert to lowercase
                - 'remove_special': Remove special characters

        Returns:
            DataFrame with processed text columns
        """
        df = df.copy()

        for col in columns:
            if col in df.columns:
                if operation == 'extract_numbers':
                    df[col] = df[col].astype(str).apply(
                        lambda x: re.search(r'\d+', x).group() if re.search(r'\d+', x) else None
                    )
                elif operation == 'clean_whitespace':
                    df[col] = df[col].astype(str).str.strip()
                elif operation == 'extract_email':
                    email_pattern = r'\b[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Z|a-z]{2,}\b'
                    df[col] = df[col].astype(str).apply(
                        lambda x: re.search(email_pattern, x).group() if re.search(email_pattern, x) else None
                    )
                elif operation == 'lowercase':
                    df[col] = df[col].astype(str).str.lower()
                elif operation == 'remove_special':
                    df[col] = df[col].astype(str).apply(lambda x: re.sub(r'[^a-zA-Z0-9\s]', '', x))
                else:
                    raise ValueError(f"Unknown operation: {operation}")

        return df

    # ==================== OUTLIER HANDLING STRATEGIES ====================

    @staticmethod
    def cap_outliers_iqr(df: pd.DataFrame, columns: List[str],
                         multiplier: float = 1.5) -> pd.DataFrame:
        """
        Cap outliers using IQR method (winsorization).

        Args:
            df: Input dataframe
            columns: Numerical columns to cap
            multiplier: IQR multiplier for bounds (1.5 = mild, 3.0 = extreme)

        Returns:
            DataFrame with capped outliers
        """
        df = df.copy()
        for col in columns:
            if col in df.columns:
                df[col] = pd.to_numeric(df[col], errors='coerce')
                if pd.api.types.is_numeric_dtype(df[col]):
                    Q1 = df[col].quantile(0.25)
                    Q3 = df[col].quantile(0.75)
                    IQR = Q3 - Q1
                    lower_bound = Q1 - multiplier * IQR
                    upper_bound = Q3 + multiplier * IQR
                    df[col] = df[col].clip(lower=lower_bound, upper=upper_bound)
        return df

    @staticmethod
    def remove_outliers_iqr(df: pd.DataFrame, columns: List[str],
                            multiplier: float = 1.5) -> pd.DataFrame:
        """
        Remove rows with outliers using IQR method.

        Args:
            df: Input dataframe
            columns: Numerical columns to check
            multiplier: IQR multiplier for bounds

        Returns:
            DataFrame with outlier rows removed
        """
        df = df.copy()
        for col in columns:
            if col in df.columns:
                df[col] = pd.to_numeric(df[col], errors='coerce')
                if pd.api.types.is_numeric_dtype(df[col]):
                    Q1 = df[col].quantile(0.25)
                    Q3 = df[col].quantile(0.75)
                    IQR = Q3 - Q1
                    lower_bound = Q1 - multiplier * IQR
                    upper_bound = Q3 + multiplier * IQR
                    df = df[(df[col] >= lower_bound) & (df[col] <= upper_bound)]
        return df

    @staticmethod
    def remove_outliers_zscore(df: pd.DataFrame, columns: List[str],
                               threshold: float = 3.0) -> pd.DataFrame:
        """
        Remove rows with outliers using Z-score method.

        Args:
            df: Input dataframe
            columns: Numerical columns to check
            threshold: Z-score threshold (3.0 = ~99.7% of data)

        Returns:
            DataFrame with outlier rows removed
        """
        df = df.copy()
        for col in columns:
            if col in df.columns:
                df[col] = pd.to_numeric(df[col], errors='coerce')
                if pd.api.types.is_numeric_dtype(df[col]):
                    z_scores = np.abs((df[col] - df[col].mean()) / df[col].std())
                    df = df[z_scores < threshold]
        return df


class DataCleaningPipeline:
    """
    Modular data cleaning pipeline with configurable strategies.

    Usage:
        pipeline = DataCleaningPipeline(name="Transactions")
        pipeline.add_step(
            strategy=CleaningStrategies.remove_duplicates,
            subset=['customer_id', 'order_date'],
            description="Remove duplicate orders"
        ).add_step(
            strategy=CleaningStrategies.drop_missing,
            columns=['customer_id'],
            description="Drop rows with missing customer_id"
        )
        df_clean = pipeline.execute(df)
    """

    def __init__(self, name: str = "Data"):
        self.name = name
        self.steps = []
        self.execution_log = []

    def add_step(self, strategy: Callable, description: str = "", **kwargs) -> 'DataCleaningPipeline':
        """
        Add a cleaning step to the pipeline.

        Args:
            strategy: Cleaning function from CleaningStrategies
            description: Human-readable description of the step
            **kwargs: Parameters to pass to the strategy function

        Returns:
            Self for method chaining
        """
        self.steps.append({
            'strategy': strategy,
            'description': description or strategy.__name__,
            'kwargs': kwargs
        })
        return self

    def execute(self, df: pd.DataFrame, verbose: bool = True) -> pd.DataFrame:
        """
        Execute all cleaning steps in the pipeline.

        Args:
            df: Input dataframe
            verbose: Print progress information

        Returns:
            Cleaned dataframe
        """
        df_clean = df.copy()
        self.execution_log = []

        if verbose:
            print("=" * 80)
            print(f"EXECUTING {self.name.upper()} CLEANING PIPELINE")
            print("=" * 80)

        for i, step in enumerate(self.steps, 1):
            strategy = step['strategy']
            description = step['description']
            kwargs = step['kwargs']

            rows_before = len(df_clean)
            missing_before = df_clean.isnull().sum().sum()

            try:
                df_clean = strategy(df_clean, **kwargs)

                rows_after = len(df_clean)
                missing_after = df_clean.isnull().sum().sum()
                rows_removed = rows_before - rows_after

                log_entry = {
                    'step': i,
                    'description': description,
                    'rows_before': rows_before,
                    'rows_after': rows_after,
                    'rows_removed': rows_removed,
                    'status': 'success'
                }
                self.execution_log.append(log_entry)

                if verbose:
                    print(f"\n[Step {i}/{len(self.steps)}] {description}")
                    print(f"  Rows: {rows_before:,} -> {rows_after:,} (removed: {rows_removed:,})")

            except Exception as e:
                log_entry = {
                    'step': i,
                    'description': description,
                    'error': str(e),
                    'status': 'failed'
                }
                self.execution_log.append(log_entry)

                if verbose:
                    print(f"\n[Step {i}/{len(self.steps)}] {description}")
                    print(f"  ERROR: {str(e)}")

        if verbose:
            print("\n" + "=" * 80)
            print(f"Pipeline completed: {len(df_clean):,} rows, {len(df_clean.columns)} columns")
            print("=" * 80)

        return df_clean

    def get_log(self) -> pd.DataFrame:
        """Get execution log as DataFrame"""
        return pd.DataFrame(self.execution_log)
