---
name: ffmpeg-video-editing
description: "Video editing with ffmpeg including cutting, trimming, concatenating segments, and re-encoding. Use when working with video files (.mp4, .mkv, .avi) for: removing segments, joining clips, extracting portions, or any video manipulation task."
---

# FFmpeg Video Editing

## Cutting Video Segments

### Extract a portion (keep segment)

```bash
# Extract from start_time to end_time
ffmpeg -i input.mp4 -ss START -to END -c copy output.mp4

# With re-encoding for frame-accurate cuts
ffmpeg -i input.mp4 -ss START -to END -c:v libx264 -c:a aac output.mp4
```

### Remove a segment (cut out middle)

To remove a segment, split into parts and concatenate:

```bash
# 1. Extract before the cut
ffmpeg -i input.mp4 -to CUT_START -c copy part1.mp4

# 2. Extract after the cut  
ffmpeg -i input.mp4 -ss CUT_END -c copy part2.mp4

# 3. Concatenate
ffmpeg -f concat -safe 0 -i filelist.txt -c copy output.mp4
```

## Concatenating Multiple Segments

### Using concat demuxer (recommended for same-codec files)

Create a file list (`segments.txt`):
```
file 'segment1.mp4'
file 'segment2.mp4'
file 'segment3.mp4'
```

Then concatenate:
```bash
ffmpeg -f concat -safe 0 -i segments.txt -c copy output.mp4
```

### Using filter_complex (for re-encoding)

```bash
ffmpeg -i seg1.mp4 -i seg2.mp4 -i seg3.mp4 \
  -filter_complex "[0:v][0:a][1:v][1:a][2:v][2:a]concat=n=3:v=1:a=1[v][a]" \
  -map "[v]" -map "[a]" output.mp4
```

## Removing Multiple Segments (Batch)

For removing many short segments (like filler words), the efficient approach:

1. Calculate the "keep" segments (inverse of remove segments)
2. Extract each keep segment
3. Concatenate all keep segments

```python
import subprocess
import os

def remove_segments(input_file, segments_to_remove, output_file):
    """
    segments_to_remove: list of (start, end) tuples in seconds
    """
    # Get video duration
    result = subprocess.run([
        'ffprobe', '-v', 'error', '-show_entries', 'format=duration',
        '-of', 'default=noprint_wrappers=1:nokey=1', input_file
    ], capture_output=True, text=True)
    duration = float(result.stdout.strip())

    # Sort segments and merge overlapping
    segments = sorted(segments_to_remove)

    # Calculate keep segments (gaps between remove segments)
    keep_segments = []
    current_pos = 0.0

    for start, end in segments:
        if start > current_pos:
            keep_segments.append((current_pos, start))
        current_pos = max(current_pos, end)

    if current_pos < duration:
        keep_segments.append((current_pos, duration))

    # Extract each keep segment
    temp_files = []
    for i, (start, end) in enumerate(keep_segments):
        temp_file = f'/tmp/seg_{i:04d}.mp4'
        subprocess.run([
            'ffmpeg', '-y', '-i', input_file,
            '-ss', str(start), '-to', str(end),
            '-c', 'copy', temp_file
        ], check=True)
        temp_files.append(temp_file)

    # Create concat list
    list_file = '/tmp/concat_list.txt'
    with open(list_file, 'w') as f:
        for temp_file in temp_files:
            f.write(f"file '{temp_file}'\n")

    # Concatenate
    subprocess.run([
        'ffmpeg', '-y', '-f', 'concat', '-safe', '0',
        '-i', list_file, '-c', 'copy', output_file
    ], check=True)

    # Cleanup
    for f in temp_files:
        os.remove(f)
    os.remove(list_file)
```

## Common Issues

### Audio/Video sync problems
- Use `-c copy` only when cutting at keyframes
- For precise cuts, re-encode: `-c:v libx264 -c:a aac`

### Gaps or glitches at cut points
- Ensure segments don't overlap
- Small buffer (0.01s) between segments can help

### Getting video duration

```bash
ffprobe -v error -show_entries format=duration -of default=noprint_wrappers=1:nokey=1 input.mp4
```
