#!/usr/bin/env node

import { google } from 'googleapis';
import { readFileSync, writeFileSync, existsSync } from 'fs';
import { fileURLToPath } from 'url';
import { dirname, join } from 'path';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

const TOKEN_PATH = join(__dirname, 'tokens.json');
const CREDENTIALS_PATH = join(__dirname, 'credentials.json');

/**
 * Load the multi-account token configuration
 * @returns {Object} Token configuration with accounts
 */
export function loadTokenConfig() {
  if (!existsSync(TOKEN_PATH)) {
    throw new Error(`Token file not found. Run: npm run setup`);
  }

  const content = readFileSync(TOKEN_PATH, 'utf8');
  const data = JSON.parse(content);

  // Check if it's the old single-account format
  if (data.access_token && !data.accounts) {
    // Old format detected, migrate automatically
    console.error('⚠️  Migrating token.json to multi-account format...');
    const migratedConfig = {
      defaultAccount: 'default',
      accounts: {
        default: data
      }
    };
    saveTokenConfig(migratedConfig);
    return migratedConfig;
  }

  return data;
}

/**
 * Save the multi-account token configuration
 * @param {Object} config - Token configuration to save
 */
export function saveTokenConfig(config) {
  writeFileSync(TOKEN_PATH, JSON.stringify(config, null, 2), { mode: 0o600 });
}

/**
 * Get the default account ID
 * @returns {string} Default account ID
 */
export function getDefaultAccount() {
  const config = loadTokenConfig();
  return config.defaultAccount || Object.keys(config.accounts)[0];
}

/**
 * Set the default account ID
 * @param {string} accountId - Account ID to set as default
 */
export function setDefaultAccount(accountId) {
  const config = loadTokenConfig();
  if (!config.accounts[accountId]) {
    throw new Error(`Account '${accountId}' not found`);
  }
  config.defaultAccount = accountId;
  saveTokenConfig(config);
}

/**
 * List all configured accounts
 * @returns {Array} Array of account objects with id, email, and scopes
 */
export function listAccounts() {
  const config = loadTokenConfig();
  return Object.entries(config.accounts).map(([id, data]) => ({
    id,
    email: data.email || 'unknown',
    scope: data.scope || '',
    isDefault: id === config.defaultAccount
  }));
}

/**
 * Load tokens for a specific account
 * @param {string} accountId - Account ID (defaults to default account)
 * @returns {Object} Token object for the account
 */
export function loadTokens(accountId = null) {
  const config = loadTokenConfig();
  const targetAccount = accountId || config.defaultAccount;

  if (!config.accounts[targetAccount]) {
    const available = Object.keys(config.accounts).join(', ');
    throw new Error(
      `Account '${targetAccount}' not found. Available accounts: ${available}`
    );
  }

  return config.accounts[targetAccount];
}

/**
 * Save tokens for a specific account
 * @param {string} accountId - Account ID
 * @param {Object} tokens - Token object to save
 * @param {string} email - Email address for the account (optional)
 */
export function saveTokens(accountId, tokens, email = null) {
  const config = existsSync(TOKEN_PATH) ? loadTokenConfig() : { accounts: {} };

  if (!config.accounts) {
    config.accounts = {};
  }

  // Preserve existing email if not provided
  const existingEmail = config.accounts[accountId]?.email;

  config.accounts[accountId] = {
    ...tokens,
    ...(email && { email }),
    ...(!email && existingEmail && { email: existingEmail })
  };

  // Set as default if it's the first account
  if (!config.defaultAccount || Object.keys(config.accounts).length === 1) {
    config.defaultAccount = accountId;
  }

  saveTokenConfig(config);
}

/**
 * Remove an account
 * @param {string} accountId - Account ID to remove
 */
export function removeAccount(accountId) {
  const config = loadTokenConfig();

  if (!config.accounts[accountId]) {
    throw new Error(`Account '${accountId}' not found`);
  }

  delete config.accounts[accountId];

  // Update default if we removed it
  if (config.defaultAccount === accountId) {
    const remaining = Object.keys(config.accounts);
    config.defaultAccount = remaining.length > 0 ? remaining[0] : null;
  }

  saveTokenConfig(config);
}

/**
 * Check if an account's token needs refresh (within 5 minutes of expiry)
 * @param {Object} tokens - Token object
 * @returns {boolean} True if token needs refresh
 */
function needsRefresh(tokens) {
  if (!tokens.expiry_date) return false;
  const expiryTime = new Date(tokens.expiry_date);
  const now = new Date();
  const fiveMinutes = 5 * 60 * 1000;
  return (expiryTime - now) < fiveMinutes;
}

/**
 * Refresh access token if needed
 * @param {string} accountId - Account ID (defaults to default account)
 * @returns {Object} Current tokens (refreshed if needed)
 */
export async function refreshTokenIfNeeded(accountId = null) {
  const tokens = loadTokens(accountId);
  const targetAccount = accountId || getDefaultAccount();

  if (!needsRefresh(tokens)) {
    return tokens;
  }

  // Token needs refresh
  if (!tokens.refresh_token) {
    throw new Error(`No refresh token available for account '${targetAccount}'. Please re-authenticate.`);
  }

  // Load credentials
  if (!existsSync(CREDENTIALS_PATH)) {
    throw new Error(`Credentials file not found at ${CREDENTIALS_PATH}`);
  }

  const credentials = JSON.parse(readFileSync(CREDENTIALS_PATH, 'utf8'));
  const { client_id, client_secret } = credentials.installed || credentials.web;

  // Create OAuth client and refresh
  const oauth2Client = new google.auth.OAuth2(client_id, client_secret);
  oauth2Client.setCredentials(tokens);

  try {
    const { credentials: newTokens } = await oauth2Client.refreshAccessToken();
    const refreshedTokens = {
      access_token: newTokens.access_token,
      refresh_token: newTokens.refresh_token || tokens.refresh_token, // Keep old refresh token if not provided
      scope: newTokens.scope || tokens.scope,
      token_type: newTokens.token_type || 'Bearer',
      expiry_date: newTokens.expiry_date
    };

    // Save refreshed tokens
    saveTokens(targetAccount, refreshedTokens);

    return refreshedTokens;
  } catch (error) {
    throw new Error(`Failed to refresh token for account '${targetAccount}': ${error.message}`);
  }
}

/**
 * Get authenticated OAuth2 client for an account
 * @param {string} accountId - Account ID (defaults to default account)
 * @returns {Object} Authenticated OAuth2 client
 */
export async function getAuthClient(accountId = null) {
  const tokens = await refreshTokenIfNeeded(accountId);

  // Load credentials
  if (!existsSync(CREDENTIALS_PATH)) {
    throw new Error(`Credentials file not found at ${CREDENTIALS_PATH}`);
  }

  const credentials = JSON.parse(readFileSync(CREDENTIALS_PATH, 'utf8'));
  const { client_id, client_secret } = credentials.installed || credentials.web;

  const oauth2Client = new google.auth.OAuth2(client_id, client_secret);
  oauth2Client.setCredentials(tokens);

  return oauth2Client;
}

/**
 * Parse account argument from command line args
 * @param {Object} args - Parsed command line arguments
 * @returns {string|null} Account ID or null for default
 */
export function parseAccountArg(args) {
  return args.account || null;
}
