#!/usr/bin/env node

import { google } from 'googleapis';
import minimist from 'minimist';
import { getAuthClient, parseAccountArg } from './auth/auth-utils.js';
import { logAction } from './action-logger.js';

async function forwardEmail(args) {
  if (!args.id || !args.to) {
    throw new Error('Missing required arguments: --id and --to are required');
  }

  const accountId = parseAccountArg(args);
  const oauth2Client = await getAuthClient(accountId);
  const gmail = google.gmail({ version: 'v1', auth: oauth2Client });

  // Get the original message in raw format
  const message = await gmail.users.messages.get({
    userId: 'me',
    id: args.id,
    format: 'raw'
  });

  // Decode the message
  const rawMessage = Buffer.from(message.data.raw, 'base64url').toString();

  // Add forwarding headers
  const forwardedMessage = `To: ${args.to}\r\n${rawMessage}`;

  // Encode for sending
  const encodedMessage = Buffer.from(forwardedMessage)
    .toString('base64')
    .replace(/\+/g, '-')
    .replace(/\//g, '_')
    .replace(/=+$/, '');

  // Send the forwarded message
  const result = await gmail.users.messages.send({
    userId: 'me',
    requestBody: {
      raw: encodedMessage
    }
  });

  return {
    success: true,
    originalId: args.id,
    forwardedId: result.data.id,
    to: args.to,
    account: accountId || 'default'
  };
}

const args = minimist(process.argv.slice(2));
const startTime = Date.now();

forwardEmail(args)
  .then(result => {
    // Log the action
    logAction('forward', args, result, {
      account: result.account,
      duration: Date.now() - startTime
    });

    console.log(JSON.stringify(result, null, 2));
  })
  .catch(error => {
    const errorResult = {
      success: false,
      error: error.message
    };

    // Log the failed action
    logAction('forward', args, errorResult, {
      account: args.account || 'default',
      duration: Date.now() - startTime,
      failed: true
    });

    console.error(JSON.stringify(errorResult, null, 2));
    process.exit(1);
  });
