#!/usr/bin/env node

import { google } from 'googleapis';
import minimist from 'minimist';
import { getAuthClient, parseAccountArg } from './auth/auth-utils.js';
import { logAction } from './action-logger.js';

async function manageLabels(args) {
  const action = args.action || args.a;

  if (!action) {
    throw new Error('Missing required argument: --action (list|create|add|remove)');
  }

  // Get authenticated client for specified account (or default)
  const accountId = parseAccountArg(args);
  const oauth2Client = await getAuthClient(accountId);

  const gmail = google.gmail({ version: 'v1', auth: oauth2Client });

  switch (action) {
    case 'list': {
      const response = await gmail.users.labels.list({ userId: 'me' });
      return {
        success: true,
        action: 'list',
        count: response.data.labels.length,
        labels: response.data.labels.map(label => ({
          id: label.id,
          name: label.name,
          type: label.type,
          messageListVisibility: label.messageListVisibility,
          labelListVisibility: label.labelListVisibility
        }))
      };
    }

    case 'create': {
      if (!args.name) {
        throw new Error('--name is required for create action');
      }

      const response = await gmail.users.labels.create({
        userId: 'me',
        requestBody: {
          name: args.name,
          labelListVisibility: args.visibility || 'labelShow',
          messageListVisibility: 'show'
        }
      });

      return {
        success: true,
        action: 'create',
        label: {
          id: response.data.id,
          name: response.data.name
        }
      };
    }

    case 'add': {
      if (!args.id || !args.label) {
        throw new Error('--id (message ID) and --label (label name or ID) are required for add action');
      }

      // If label is a name, find its ID
      let labelId = args.label;
      if (!labelId.startsWith('Label_')) {
        const labelsResponse = await gmail.users.labels.list({ userId: 'me' });
        const label = labelsResponse.data.labels.find(l => l.name === args.label);
        if (!label) {
          throw new Error(`Label "${args.label}" not found`);
        }
        labelId = label.id;
      }

      await gmail.users.messages.modify({
        userId: 'me',
        id: args.id,
        requestBody: {
          addLabelIds: [labelId]
        }
      });

      return {
        success: true,
        action: 'add',
        messageId: args.id,
        labelId: labelId,
        label: args.label
      };
    }

    case 'remove': {
      if (!args.id || !args.label) {
        throw new Error('--id (message ID) and --label (label name or ID) are required for remove action');
      }

      // If label is a name, find its ID
      let labelId = args.label;
      if (!labelId.startsWith('Label_')) {
        const labelsResponse = await gmail.users.labels.list({ userId: 'me' });
        const label = labelsResponse.data.labels.find(l => l.name === args.label);
        if (!label) {
          throw new Error(`Label "${args.label}" not found`);
        }
        labelId = label.id;
      }

      await gmail.users.messages.modify({
        userId: 'me',
        id: args.id,
        requestBody: {
          removeLabelIds: [labelId]
        }
      });

      return {
        success: true,
        action: 'remove',
        messageId: args.id,
        labelId: labelId,
        label: args.label
      };
    }

    default:
      throw new Error(`Unknown action: ${action}. Use: list, create, add, or remove`);
  }
}

// Main execution
const args = minimist(process.argv.slice(2));
const startTime = Date.now();

manageLabels(args)
  .then(result => {
    // Log the action
    logAction('labels', args, result, {
      account: args.account || 'default',
      duration: Date.now() - startTime,
      labelAction: result.action
    });

    console.log(JSON.stringify(result, null, 2));
  })
  .catch(error => {
    const errorResult = {
      success: false,
      error: error.message
    };

    // Log the failed action
    logAction('labels', args, errorResult, {
      account: args.account || 'default',
      duration: Date.now() - startTime,
      failed: true
    });

    console.error(JSON.stringify(errorResult, null, 2));
    process.exit(1);
  });
