---
name: jakarta-namespace
description: Migrate Java EE javax.* imports to Jakarta EE jakarta.* namespace. Use when upgrading to Spring Boot 3.x, migrating javax.persistence, javax.validation, javax.servlet imports, or fixing compilation errors after Jakarta EE transition. Covers package mappings, batch sed commands, and verification steps.
---

# Jakarta EE Namespace Migration Skill

## Overview

Spring Boot 3.0 has upgraded from Java EE to Jakarta EE APIs for all dependencies. This is one of the most significant breaking changes and requires updating all `javax.*` package imports to `jakarta.*`.

**Important:** Packages such as `javax.sql.*` and `javax.crypto.*` will NOT change to `jakarta.*` as they are part of the Java 17+ JDK itself, not part of Java EE.

## Required Package Mappings

| Before (Java EE) | After (Jakarta EE) |
|------------------|-------------------|
| `javax.persistence.*` | `jakarta.persistence.*` |
| `javax.validation.*` | `jakarta.validation.*` |
| `javax.servlet.*` | `jakarta.servlet.*` |
| `javax.annotation.*` | `jakarta.annotation.*` |
| `javax.transaction.*` | `jakarta.transaction.*` |
| `javax.ws.rs.*` | `jakarta.ws.rs.*` |
| `javax.mail.*` | `jakarta.mail.*` |
| `javax.jms.*` | `jakarta.jms.*` |
| `javax.xml.bind.*` | `jakarta.xml.bind.*` |

## Affected Annotations and Classes

### Persistence (JPA)
- `@Entity`, `@Table`, `@Column`
- `@Id`, `@GeneratedValue`
- `@ManyToOne`, `@OneToMany`, `@ManyToMany`, `@OneToOne`
- `@JoinColumn`, `@JoinTable`
- `@PrePersist`, `@PreUpdate`, `@PostLoad`
- `@Enumerated`, `@Temporal`
- `EntityManager`, `EntityManagerFactory`
- `EntityNotFoundException`

### Validation
- `@Valid`
- `@NotNull`, `@NotBlank`, `@NotEmpty`
- `@Size`, `@Min`, `@Max`
- `@Email`, `@Pattern`
- `@Positive`, `@Negative`
- `@Past`, `@Future`

### Servlet
- `HttpServletRequest`, `HttpServletResponse`
- `ServletException`
- `Filter`, `FilterChain`
- `HttpSession`, `Cookie`

## Migration Commands

### Step 1: Find All javax Imports That Need Migration

```bash
# List all files with javax imports that need migration (excludes JDK packages)
grep -r "import javax\." --include="*.java" . | grep -v "javax.sql" | grep -v "javax.crypto" | grep -v "javax.net"
```

### Step 2: Batch Replace All Namespaces

```bash
# Replace javax.persistence with jakarta.persistence
find . -name "*.java" -type f -exec sed -i 's/import javax\.persistence/import jakarta.persistence/g' {} +

# Replace javax.validation with jakarta.validation
find . -name "*.java" -type f -exec sed -i 's/import javax\.validation/import jakarta.validation/g' {} +

# Replace javax.servlet with jakarta.servlet
find . -name "*.java" -type f -exec sed -i 's/import javax\.servlet/import jakarta.servlet/g' {} +

# Replace javax.annotation (common ones used in Spring)
find . -name "*.java" -type f -exec sed -i 's/import javax\.annotation\.PostConstruct/import jakarta.annotation.PostConstruct/g' {} +
find . -name "*.java" -type f -exec sed -i 's/import javax\.annotation\.PreDestroy/import jakarta.annotation.PreDestroy/g' {} +
find . -name "*.java" -type f -exec sed -i 's/import javax\.annotation\.Resource/import jakarta.annotation.Resource/g' {} +

# Replace javax.transaction with jakarta.transaction
find . -name "*.java" -type f -exec sed -i 's/import javax\.transaction/import jakarta.transaction/g' {} +
```

### Step 3: Handle Wildcard Imports

```bash
# Replace wildcard imports
find . -name "*.java" -type f -exec sed -i 's/import javax\.persistence\.\*/import jakarta.persistence.*/g' {} +
find . -name "*.java" -type f -exec sed -i 's/import javax\.validation\.\*/import jakarta.validation.*/g' {} +
find . -name "*.java" -type f -exec sed -i 's/import javax\.servlet\.\*/import jakarta.servlet.*/g' {} +
```

## Critical: Entity Classes Must Use jakarta.persistence

After migration, **every JPA entity class MUST have jakarta.persistence imports**. This is a hard requirement for Spring Boot 3.

### Example Entity Migration

```java
// BEFORE (Spring Boot 2.x) - WILL NOT COMPILE in Spring Boot 3
import javax.persistence.Entity;
import javax.persistence.Table;
import javax.persistence.Id;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;

@Entity
@Table(name = "users")
public class User {
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;
}

// AFTER (Spring Boot 3.x) - REQUIRED
import jakarta.persistence.Entity;
import jakarta.persistence.Table;
import jakarta.persistence.Id;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.GenerationType;

@Entity
@Table(name = "users")
public class User {
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)
    private Long id;
}
```

### Example Validation Migration

```java
// BEFORE
import javax.validation.constraints.Email;
import javax.validation.constraints.NotBlank;
import javax.validation.Valid;

// AFTER
import jakarta.validation.constraints.Email;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.Valid;
```

### Example Servlet Migration

```java
// BEFORE
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

// AFTER
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
```

## Verification

### Verify No Old javax Imports Remain

```bash
# These commands should return NO results after migration
grep -r "import javax\.persistence" --include="*.java" .
grep -r "import javax\.validation" --include="*.java" .
grep -r "import javax\.servlet" --include="*.java" .

# Combined check for all Java EE packages
grep -r "import javax\." --include="*.java" . | grep -E "(persistence|validation|servlet|transaction|annotation\.(PostConstruct|PreDestroy|Resource))"
```

### Verify jakarta Imports Are Present

```bash
# These MUST return results showing your migrated classes
grep -r "import jakarta\.persistence" --include="*.java" .
grep -r "import jakarta\.validation" --include="*.java" .
```

**If the jakarta.persistence grep returns no results but you have JPA entities, the migration is incomplete and will fail.**

## Using OpenRewrite for Automated Migration

OpenRewrite can automate the entire namespace migration:

```xml
<!-- Add to pom.xml plugins section -->
<plugin>
    <groupId>org.openrewrite.maven</groupId>
    <artifactId>rewrite-maven-plugin</artifactId>
    <version>5.42.0</version>
    <configuration>
        <activeRecipes>
            <recipe>org.openrewrite.java.migrate.jakarta.JavaxMigrationToJakarta</recipe>
        </activeRecipes>
    </configuration>
    <dependencies>
        <dependency>
            <groupId>org.openrewrite.recipe</groupId>
            <artifactId>rewrite-migrate-java</artifactId>
            <version>2.26.0</version>
        </dependency>
    </dependencies>
</plugin>
```

Then run:
```bash
mvn rewrite:run
```

## Common Pitfalls

1. **Don't change javax.sql or javax.crypto** - These are JDK packages, not Java EE
2. **Check test files too** - Test classes also need migration
3. **Update XML configurations** - If using persistence.xml, update namespaces there too
4. **Third-party libraries** - Ensure all dependencies have Jakarta EE compatible versions
5. **Mixed namespaces cause runtime errors** - All must be migrated together

## Sources

- [Spring Boot 3.0 Migration Guide](https://github.com/spring-projects/spring-boot/wiki/Spring-Boot-3.0-Migration-Guide)
- [Baeldung - Migrate to Spring Boot 3](https://www.baeldung.com/spring-boot-3-migration)
- [OpenRewrite Jakarta Migration](https://docs.openrewrite.org/running-recipes/popular-recipe-guides/migrate-to-spring-3)
