# HyperConnections Module Implementation

## Full Implementation

```python
class HyperConnections(nn.Module):
    def __init__(
        self,
        num_residual_streams,
        dim,
        branch=None,
        layer_index=0,
        sinkhorn_iters=20,
        sinkhorn_tau=0.05,
    ):
        super().__init__()
        self.num_residual_streams = num_residual_streams
        self.branch = branch
        self.sinkhorn_iters = sinkhorn_iters
        self.sinkhorn_tau = sinkhorn_tau

        # H_res: initialized near identity (diagonal = 0, off-diagonal = -0.1)
        # Note: Use small negative values to ensure gradient flow through sinkhorn
        init_h_res = torch.full((num_residual_streams, num_residual_streams), -0.1)
        init_h_res.fill_diagonal_(0.0)
        self.H_res_logits = nn.Parameter(init_h_res)

        # H_pre: selects which stream(s) feed the branch
        init_h_pre = torch.full((1, num_residual_streams), -0.1)
        init_h_pre[0, layer_index % num_residual_streams] = 0.0
        self.H_pre_logits = nn.Parameter(init_h_pre)

        # H_post: distributes branch output back to streams
        self.H_post_logits = nn.Parameter(torch.zeros(1, num_residual_streams))

    def forward(self, residuals, *args, **kwargs):
        """
        Args:
            residuals: (batch * num_streams, seq, dim)
        """
        streams = self.num_residual_streams

        # Reshape to (batch, seq, num_streams, dim)
        residuals = rearrange(residuals, "(b s) t d -> b t s d", s=streams)

        # Width connection: mix residual streams
        h_res = sinkhorn_knopp(self.H_res_logits, self.sinkhorn_iters, self.sinkhorn_tau)
        residuals_mixed = einsum(h_res, residuals, "s t, b n s d -> b n t d")

        # Select branch input
        h_pre = self.H_pre_logits.softmax(dim=-1)
        branch_input = einsum(h_pre, residuals, "v s, b n s d -> b n v d")
        branch_input = branch_input.squeeze(-2)  # (batch, seq, dim)

        # Process through branch (attention or MLP)
        if self.branch is not None:
            branch_output = self.branch(branch_input, *args, **kwargs)
        else:
            branch_output = branch_input

        # Distribute to streams (depth connection)
        h_post = self.H_post_logits.softmax(dim=-1)
        depth_out = einsum(branch_output, h_post, "b t d, v s -> b t s d")

        # Combine with residual
        output = residuals_mixed + depth_out

        # Flatten back to (batch * num_streams, seq, dim)
        output = rearrange(output, "b t s d -> (b s) t d")

        return output
```

## Initialization Details

| Matrix | Initialization | Behavior |
|--------|---------------|----------|
| H_res | diagonal=0, off-diagonal=-0.1 | Starts near identity after Sinkhorn |
| H_pre | one-hot at layer_index % num_streams | Each layer reads from different stream |
| H_post | all zeros | Uniform distribution after softmax |

## Parameters

| Parameter | Type | Description |
|-----------|------|-------------|
| `num_residual_streams` | int | Number of parallel residual streams (typically 4) |
| `dim` | int | Hidden dimension (unused but kept for API consistency) |
| `branch` | nn.Module | The sublayer to wrap (attention or MLP) |
| `layer_index` | int | Index of this layer for H_pre initialization |
| `sinkhorn_iters` | int | Sinkhorn iterations (default 20) |
| `sinkhorn_tau` | float | Sinkhorn temperature (default 0.05) |
