# Common Pitfalls

## 1. Don't use expand/collapse linear layers

mHC uses reshape operations, not learned projections. Adding linear layers defeats the purpose and increases parameters.

```python
# WRONG - adds unnecessary parameters
self.expand = nn.Linear(dim, dim * num_streams)
self.collapse = nn.Linear(dim * num_streams, dim)

# CORRECT - use einops reshape
x = repeat(x, "b t d -> (b s) t d", s=num_streams)
x = reduce(x, "(b s) t d -> b t d", "sum", s=num_streams)
```

## 2. Initialize H_res near identity

Use diagonal=0, off-diagonal=-0.1 with softmax/sinkhorn. This starts close to identity mapping while allowing gradient flow.

```python
# CORRECT initialization
init_h_res = torch.full((num_streams, num_streams), -0.1)
init_h_res.fill_diagonal_(0.0)
self.H_res_logits = nn.Parameter(init_h_res)

# WRONG - random initialization breaks training
self.H_res_logits = nn.Parameter(torch.randn(num_streams, num_streams))

# WRONG - values too negative cause gradient vanishing
# With tau=0.05: -8.0/0.05 = -160, exp(-160) ≈ 0, gradients vanish!
init_h_res = torch.full((num_streams, num_streams), -8.0)  # DON'T USE
```

**WARNING**: The off-diagonal value must be compatible with `tau` to allow gradient flow:
- `-8.0` with `tau=0.05` → `exp(-160) ≈ 0` (gradients vanish, H_res never updates)
- `-0.1` with `tau=0.05` → `exp(-2) ≈ 0.135` (proper gradient flow)

## 3. Use log-space Sinkhorn

Direct iterative normalization is numerically unstable. Always use the log-space version.

```python
# WRONG - numerically unstable
for _ in range(num_iters):
    alpha = alpha / alpha.sum(dim=-1, keepdim=True)
    alpha = alpha / alpha.sum(dim=-2, keepdim=True)

# CORRECT - log-space
log_alpha = logits / tau
for _ in range(num_iters):
    log_alpha = log_alpha - torch.logsumexp(log_alpha, dim=-1, keepdim=True)
    log_alpha = log_alpha - torch.logsumexp(log_alpha, dim=-2, keepdim=True)
return torch.exp(log_alpha)
```

## 4. Apply to both attention and MLP

Each sublayer in the transformer block should be wrapped. Wrapping only one breaks the design.

```python
# CORRECT - wrap both
self.hc_attn = HyperConnections(num_streams, dim, branch=attn, layer_index=idx*2)
self.hc_mlp = HyperConnections(num_streams, dim, branch=mlp, layer_index=idx*2+1)

# WRONG - only wrapping attention
self.hc_attn = HyperConnections(num_streams, dim, branch=attn)
# MLP uses standard residual
```

## 5. Track layer index

Pass `layer_index` to each HyperConnections instance for proper H_pre initialization. This ensures different streams are selected for different layers.

```python
# CORRECT
for i in range(num_layers):
    self.hc_attn = HyperConnections(..., layer_index=i*2)
    self.hc_mlp = HyperConnections(..., layer_index=i*2+1)

# WRONG - all layers use same H_pre initialization
for i in range(num_layers):
    self.hc_attn = HyperConnections(..., layer_index=0)
```

## 6. Adjust batch size

With `num_streams=4`, effective batch size multiplies by 4, so reduce nominal batch size accordingly.

```python
# If baseline uses batch_size=32
# mHC should use batch_size=8 to 16 for similar memory usage
```

## Expected Benefits

When implemented correctly:

- **Lower gradient norm variance** - More stable training
- **Fewer loss spikes** - Especially in deep networks (>12 layers)
- **Preserved signal magnitude** - Doubly stochastic constraint maintains scale
- **Comparable final performance** - Similar loss to baseline with more stability
