"""Utility for estimating distances and costs between cities."""

from __future__ import annotations

import argparse
from collections.abc import Callable
from pathlib import Path

import pandas as pd


# Default to the bundled distance matrix CSV.
def _find_data_path() -> Path:
    """Find data file, checking container path first, then relative to script."""
    relative = "googleDistanceMatrix/distance.csv"
    container_path = Path("/app/data") / relative
    if container_path.exists():
        return container_path
    return Path(__file__).resolve().parent.parent.parent / "data" / relative


DEFAULT_DATA_PATH = _find_data_path()

LOCAL_COLUMNS = ["origin", "destination", "duration", "distance"]


def _extract_before_parenthesis(value: str) -> str:
    """Return the substring before the first opening parenthesis."""
    if value is None:
        return ""
    return value.split("(", 1)[0].strip()


def _parse_distance_km(distance_text: str) -> float | None:
    """Convert a distance string like '1,234 km' to a float."""
    if not isinstance(distance_text, str):
        return None
    cleaned = distance_text.lower().replace("km", "").replace(",", "").strip()
    try:
        return float(cleaned)
    except (TypeError, ValueError):
        return None


def _compute_cost(distance_km: float | None, mode: str) -> int | None:
    """Compute cost based on mode."""
    if distance_km is None:
        return None
    if "driving" in mode:
        return int(distance_km * 0.05)
    if mode == "taxi":
        return int(distance_km)
    return None


class GoogleDistanceMatrix:
    """Distance/cost helper backed by a local CSV or Google API."""

    def __init__(
        self,
        subscription_key: str = "",
        path: str | Path = DEFAULT_DATA_PATH,
        city_normalizer: Callable[[str], str] | None = None,
    ) -> None:
        self.gplaces_api_key: str = subscription_key
        self.path = Path(path)
        self.city_normalizer = city_normalizer or (lambda value: value)
        self.data: pd.DataFrame = pd.DataFrame()
        self.load_db()
        print("GoogleDistanceMatrix loaded.")

    def load_db(self) -> None:
        """Load local distance matrix data from CSV."""
        if not self.path.exists():
            # Keep empty data frame; online mode can still work.
            return

        df = pd.read_csv(self.path)
        existing_columns = [col for col in LOCAL_COLUMNS if col in df.columns]
        df = df[existing_columns].dropna()
        df["origin"] = df["origin"].astype(str).str.strip()
        df["destination"] = df["destination"].astype(str).str.strip()
        self.data = df

    def _lookup_local(self, origin: str, destination: str, mode: str) -> dict[str, str | int | None]:
        origin_norm = self.city_normalizer(_extract_before_parenthesis(origin)).strip()
        dest_norm = self.city_normalizer(_extract_before_parenthesis(destination)).strip()

        info: dict[str, str | int | None] = {
            "origin": origin_norm,
            "destination": dest_norm,
            "duration": None,
            "distance": None,
            "cost": None,
        }

        if self.data.empty:
            return info

        matches = self.data[(self.data["origin"] == origin_norm) & (self.data["destination"] == dest_norm)]
        if matches.empty:
            return info

        duration = matches["duration"].values[0]
        distance_text = matches["distance"].values[0]
        if duration is None or distance_text is None or pd.isna(duration) or pd.isna(distance_text):
            return info

        if isinstance(duration, str) and "day" in duration.lower():
            return info

        distance_km = _parse_distance_km(distance_text)
        cost = _compute_cost(distance_km, mode)

        info["duration"] = duration
        info["distance"] = distance_text
        info["cost"] = cost
        return info

    def run(self, origin: str, destination: str, mode: str = "driving") -> str:
        """Return a human-readable distance/cost summary using local data."""
        info = self._lookup_local(origin, destination, mode)

        if info["duration"] and info["distance"]:
            cost_text = f", cost: {info['cost']}" if info["cost"] is not None else ""
            return (
                f"{mode}, from {info['origin']} to {info['destination']}, "
                f"duration: {info['duration']}, distance: {info['distance']}{cost_text}"
            )

        return f"{mode}, from {info['origin']} to {info['destination']}, no valid information."


def _build_arg_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(description="Lookup distance, duration, and cost between cities.")
    parser.add_argument("--origin", "-o", help="Origin city.")
    parser.add_argument("--destination", "-d", help="Destination city.")
    parser.add_argument(
        "--mode",
        "-m",
        default="driving",
        help="Mode: driving (default) or taxi; online uses driving for both.",
    )
    parser.add_argument(
        "--path",
        "-p",
        default=str(DEFAULT_DATA_PATH),
        help="Path to local distance CSV (defaults to bundled dataset).",
    )
    parser.add_argument(
        "--api-key",
        "-k",
        default="",
        help="Optional Google Distance Matrix API key; if omitted, uses local CSV only.",
    )

    return parser


def main() -> None:
    args = _build_arg_parser().parse_args()
    matrix = GoogleDistanceMatrix(subscription_key=args.api_key, path=args.path)

    if not (args.origin and args.destination):
        print("Please provide --origin and --destination.")
        return

    print(matrix.run(args.origin, args.destination, args.mode))


if __name__ == "__main__":
    main()
