import fs from 'fs';
import path from 'path';
import { pathToFileURL } from 'url';

const args = process.argv.slice(2);
const getArg = (flag, fallback = null) => {
    const idx = args.indexOf(flag);
    if (idx === -1 || idx + 1 >= args.length) return fallback;
    return args[idx + 1];
};

const input = getArg('--input');
const output = getArg('--output');
const meshDir = getArg('--mesh-dir');
const robotName = getArg('--robot-name', 'object');
const jointMapPath = getArg('--joint-map');
const jointDefault = getArg('--joint-default', 'fixed');
const useNameHints = args.includes('--name-hints');
const includeRoot = args.includes('--include-root');

if (!input || !output || !meshDir) {
    console.error(
        'Usage: node build_urdf_from_scene.mjs --input <file.js> --output <file.urdf> ' +
        '--mesh-dir <dir> [--robot-name <name>] [--joint-map <file.json>] ' +
        '[--joint-default <fixed|revolute|prismatic>] [--name-hints] [--include-root]'
    );
    process.exit(2);
}

const normalizeJointType = (value) => {
    const lower = String(value || '').toLowerCase();
    if (lower === 'revolute' || lower === 'prismatic' || lower === 'fixed') {
        return lower;
    }
    return 'fixed';
};

const jointDefaultType = normalizeJointType(jointDefault);

let jointMap = null;
if (jointMapPath) {
    try {
        const raw = fs.readFileSync(jointMapPath, 'utf8');
        jointMap = JSON.parse(raw);
    } catch (error) {
        console.error(`Failed to read joint map ${jointMapPath}: ${error.message}`);
        process.exit(2);
    }
}

const sceneModuleURL = pathToFileURL(input).href;
const sceneModule = await import(sceneModuleURL);
const root = sceneModule.createScene();
root.updateMatrixWorld(true);

const linkMap = new Map();
root.traverse((obj) => {
    if (!obj.isGroup || !obj.name) return;
    if (!includeRoot && obj === root) return;
    linkMap.set(obj.name, obj);
});

const linkNames = Array.from(linkMap.keys()).sort();
const linkNameSet = new Set(linkNames);

const inferJointType = (name) => {
    if (jointMap && typeof jointMap[name] === 'string') {
        return normalizeJointType(jointMap[name]);
    }
    if (useNameHints) {
        const lower = name.toLowerCase();
        if (/(slide|rail|linear|track|slider|prismatic)/.test(lower)) {
            return 'prismatic';
        }
        if (/(hinge|rotate|spin|wheel|pivot|revolute)/.test(lower)) {
            return 'revolute';
        }
    }
    return jointDefaultType;
};

const findParentLink = (obj) => {
    let current = obj.parent;
    while (current) {
        if (current.isGroup && current.name && linkNameSet.has(current.name)) {
            return current.name;
        }
        current = current.parent;
    }
    return null;
};

const joints = [];
for (const name of linkNames) {
    const linkObj = linkMap.get(name);
    const parent = findParentLink(linkObj);
    if (!parent) continue;
    joints.push({
        name: `joint_${name}`,
        parent,
        child: name,
        type: inferJointType(name),
    });
}

const lines = [];
lines.push('<?xml version="1.0"?>');
lines.push(`<robot name="${robotName}">`);
for (const name of linkNames) {
    lines.push(`  <link name="${name}">`);
    lines.push('    <visual>');
    lines.push('      <geometry>');
    lines.push(`        <mesh filename="${meshDir}/${name}.obj" />`);
    lines.push('      </geometry>');
    lines.push('    </visual>');
    lines.push('  </link>');
}
for (const joint of joints.sort((a, b) => a.name.localeCompare(b.name))) {
    lines.push(`  <joint name="${joint.name}" type="${joint.type}">`);
    lines.push(`    <parent link="${joint.parent}"/>`);
    lines.push(`    <child link="${joint.child}"/>`);
    lines.push('  </joint>');
}
lines.push('</robot>');

fs.mkdirSync(path.dirname(output), { recursive: true });
fs.writeFileSync(output, lines.join('\n'));
console.log(`Wrote ${output}`);
