# S3 Bucket Patterns

## Standard Private Bucket

```typescript
import * as s3 from 'aws-cdk-lib/aws-s3';
import { getResourceName } from './config';

const bucket = new s3.Bucket(this, 'UserFilesBucket', {
  // Include account ID for global uniqueness
  bucketName: getResourceName(config, 'user-files', config.awsAccount),

  encryption: s3.BucketEncryption.S3_MANAGED,
  blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
  enforceSSL: true,
  versioned: true,

  // Environment-based retention
  removalPolicy: config.environment === 'prod'
    ? cdk.RemovalPolicy.RETAIN
    : cdk.RemovalPolicy.DESTROY,
  autoDeleteObjects: config.environment !== 'prod',
});
```

## Bucket with CORS (Pre-signed URL Uploads)

```typescript
const corsOrigins = config.environment === 'prod'
  ? ['https://example.com']
  : ['http://localhost:4200', 'https://dev.example.com'];

const bucket = new s3.Bucket(this, 'UserFilesBucket', {
  bucketName: getResourceName(config, 'user-files', config.awsAccount),

  encryption: s3.BucketEncryption.S3_MANAGED,
  blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
  enforceSSL: true,

  // CORS for pre-signed URL uploads
  cors: [{
    allowedOrigins: corsOrigins,
    allowedMethods: [
      s3.HttpMethods.GET,
      s3.HttpMethods.PUT,
      s3.HttpMethods.HEAD,
    ],
    allowedHeaders: ['Content-Type', 'Content-Length', 'x-amz-*'],
    exposedHeaders: ['ETag', 'Content-Length', 'Content-Type'],
    maxAge: 3600,
  }],

  removalPolicy: config.environment === 'prod'
    ? cdk.RemovalPolicy.RETAIN
    : cdk.RemovalPolicy.DESTROY,
});
```

## Bucket with Lifecycle Rules

```typescript
const bucket = new s3.Bucket(this, 'UserFilesBucket', {
  bucketName: getResourceName(config, 'user-files', config.awsAccount),

  encryption: s3.BucketEncryption.S3_MANAGED,
  blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
  enforceSSL: true,
  versioned: true,

  lifecycleRules: [
    // Transition to Infrequent Access after 30 days
    {
      id: 'transition-to-ia',
      transitions: [{
        storageClass: s3.StorageClass.INFREQUENT_ACCESS,
        transitionAfter: cdk.Duration.days(30),
      }],
    },
    // Transition to Glacier after 90 days
    {
      id: 'transition-to-glacier',
      transitions: [{
        storageClass: s3.StorageClass.GLACIER,
        transitionAfter: cdk.Duration.days(90),
      }],
    },
    // Expire objects after retention period
    {
      id: 'expire-objects',
      expiration: cdk.Duration.days(config.fileUpload?.retentionDays || 365),
    },
    // Clean up old versions
    {
      id: 'delete-old-versions',
      noncurrentVersionExpiration: cdk.Duration.days(30),
    },
    // Abort incomplete multipart uploads
    {
      id: 'abort-incomplete-multipart',
      abortIncompleteMultipartUploadAfter: cdk.Duration.days(1),
    },
  ],

  removalPolicy: config.environment === 'prod'
    ? cdk.RemovalPolicy.RETAIN
    : cdk.RemovalPolicy.DESTROY,
});
```

## Frontend Bucket with CloudFront

```typescript
import * as cloudfront from 'aws-cdk-lib/aws-cloudfront';
import * as origins from 'aws-cdk-lib/aws-cloudfront-origins';

// S3 bucket for static files
const bucket = new s3.Bucket(this, 'FrontendBucket', {
  bucketName: getResourceName(config, 'frontend', config.awsAccount),

  blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
  versioned: true,
  encryption: s3.BucketEncryption.S3_MANAGED,

  lifecycleRules: [{
    id: 'DeleteOldVersions',
    noncurrentVersionExpiration: cdk.Duration.days(30),
  }],

  removalPolicy: cdk.RemovalPolicy.RETAIN,
  autoDeleteObjects: false,
});

// Origin Access Control for CloudFront
const oac = new cloudfront.CfnOriginAccessControl(this, 'OAC', {
  originAccessControlConfig: {
    name: getResourceName(config, 'frontend-oac'),
    originAccessControlOriginType: 's3',
    signingBehavior: 'always',
    signingProtocol: 'sigv4',
  },
});

// CloudFront distribution
const distribution = new cloudfront.Distribution(this, 'Distribution', {
  comment: `${config.projectPrefix} Frontend Distribution`,

  defaultBehavior: {
    origin: origins.S3BucketOrigin.withOriginAccessControl(bucket),
    viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
    allowedMethods: cloudfront.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
    cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS,
  },

  defaultRootObject: 'index.html',

  // SPA routing: serve index.html for 403/404
  errorResponses: [
    {
      httpStatus: 403,
      responseHttpStatus: 200,
      responsePagePath: '/index.html',
      ttl: cdk.Duration.minutes(5),
    },
    {
      httpStatus: 404,
      responseHttpStatus: 200,
      responsePagePath: '/index.html',
      ttl: cdk.Duration.minutes(5),
    },
  ],

  priceClass: cloudfront.PriceClass.PRICE_CLASS_100,  // US, Canada, Europe
  enabled: true,
  httpVersion: cloudfront.HttpVersion.HTTP2_AND_3,
});
```

## Bucket with Custom Domain

```typescript
import * as acm from 'aws-cdk-lib/aws-certificatemanager';
import * as route53 from 'aws-cdk-lib/aws-route53';
import * as route53Targets from 'aws-cdk-lib/aws-route53-targets';

// Import certificate (must be in us-east-1 for CloudFront)
const certificate = acm.Certificate.fromCertificateArn(
  this,
  'Certificate',
  config.frontend.certificateArn
);

// Import hosted zone
const hostedZone = route53.HostedZone.fromLookup(this, 'HostedZone', {
  domainName: config.frontend.hostedZoneDomain,
});

const distribution = new cloudfront.Distribution(this, 'Distribution', {
  // ... other config
  domainNames: [config.frontend.domainName],
  certificate: certificate,
});

// DNS record pointing to CloudFront
new route53.ARecord(this, 'AliasRecord', {
  zone: hostedZone,
  recordName: config.frontend.domainName,
  target: route53.RecordTarget.fromAlias(
    new route53Targets.CloudFrontTarget(distribution)
  ),
});
```

## Granting Access

```typescript
// Grant ECS task read/write access
bucket.grantReadWrite(taskDefinition.taskRole);

// Grant Lambda read access
bucket.grantRead(lambdaFunction);

// Grant specific actions
bucket.grantPut(uploadRole);
bucket.grantDelete(cleanupRole);

// Custom policy for pre-signed URLs
taskRole.addToPolicy(new iam.PolicyStatement({
  sid: 'S3PreSignedUrls',
  actions: [
    's3:PutObject',
    's3:GetObject',
  ],
  resources: [`${bucket.bucketArn}/*`],
  conditions: {
    StringEquals: {
      's3:x-amz-acl': 'bucket-owner-full-control',
    },
  },
}));
```

## Exporting Bucket to SSM

```typescript
new ssm.StringParameter(this, 'BucketNameParam', {
  parameterName: `/${config.projectPrefix}/file-upload/bucket-name`,
  stringValue: bucket.bucketName,
  description: 'User files S3 bucket name',
});

new ssm.StringParameter(this, 'BucketArnParam', {
  parameterName: `/${config.projectPrefix}/file-upload/bucket-arn`,
  stringValue: bucket.bucketArn,
  description: 'User files S3 bucket ARN',
});
```

## Event Notifications

```typescript
import * as s3n from 'aws-cdk-lib/aws-s3-notifications';

// Trigger Lambda on object creation
bucket.addEventNotification(
  s3.EventType.OBJECT_CREATED,
  new s3n.LambdaDestination(processingLambda),
  { prefix: 'uploads/', suffix: '.pdf' }
);

// Send to SQS on object deletion
bucket.addEventNotification(
  s3.EventType.OBJECT_REMOVED,
  new s3n.SqsDestination(cleanupQueue)
);
```

## Intelligent Tiering

```typescript
const bucket = new s3.Bucket(this, 'DataBucket', {
  bucketName: getResourceName(config, 'data', config.awsAccount),

  intelligentTieringConfigurations: [{
    name: 'AutoTiering',
    archiveAccessTierTime: cdk.Duration.days(90),
    deepArchiveAccessTierTime: cdk.Duration.days(180),
  }],
});
```
