---
name: camone-ralph-loop
description: "Run external Ralph Loop (Geoffrey Huntley method) with clean context per iteration. Supports Simple mode (single PROMPT.md) and Full Ecosystem mode (specs/, fix_plan.md, AGENT.md, plan/build prompts). Unlike the official ralph-loop plugin which uses Stop hooks (same session, context pollution), this spawns fresh `claude -p` sessions each iteration. Use when user mentions \"camone-ralph-loop\", \"external ralph\", \"clean ralph loop\", \"外部ラルフ\", \"ラルフループ実行\", or wants to run iterative AI development with isolated context windows."
---

# camone-ralph-loop

External Ralph Loop implementation faithful to Geoffrey Huntley's methodology: a bash `while true` loop that spawns fresh `claude -p` sessions, ensuring each iteration starts with a clean context window.

## Key Difference from Official Plugin

| | Official ralph-loop plugin | camone-ralph-loop |
|---|---|---|
| Loop mechanism | Stop hook (same session) | External bash loop (new session each) |
| Context | Accumulates across iterations | **Clean every iteration** |
| Self-reference | Claude sees previous work in files + conversation history | Claude sees previous work **only in files and git history** |

## Operating Modes

### Simple Mode (default, backward-compatible)

Single `PROMPT.md` file drives the loop. Best for focused, well-scoped tasks.

```bash
camone-ralph-loop.sh --max-iterations 10 PROMPT.md
```

### Full Ecosystem Mode

Geoffrey Huntley's document ecosystem with separate Plan and Build phases. Best for large, multi-task projects.

```bash
# Phase 1: Plan — analyzes specs, generates fix_plan.md
camone-ralph-loop.sh --mode plan --max-iterations 3

# Phase 2: Build — implements one task per iteration
camone-ralph-loop.sh --mode build --max-iterations 20 --completion-promise "ALL TESTS PASSING" --auto-commit
```

## Document Ecosystem — Geoffrey Huntley Method (Full Ecosystem Mode)

For detailed principles, see `references/geoffrey-principles.md`.

### specs/*.md — Specifications (Immutable)

One file per concern (JTBD — Job To Be Done). Claude reads but NEVER modifies these.

### fix_plan.md — Progress Tracker (Mutable)

Generated by Plan Loop. Tracks task status, priorities, and dependencies. Updated by Build Loop as tasks complete.

### AGENT.md — Self-Learning Document (Mutable)

Operational knowledge Claude discovers during iterations: build quirks, codebase conventions, environment notes. NOT progress notes.

### PROMPT_plan.md / PROMPT_build.md — Master Prompts

Phase-specific prompts. `--mode plan` auto-loads `PROMPT_plan.md`, `--mode build` auto-loads `PROMPT_build.md`. Templates available in `assets/templates/`.

## Workflow

### Phase 0: Interview & Setup

**PROMPT quality = Ralph Loop output quality.** Always interview before running.

#### Simple Mode

1. **Understand intent**: Grasp what the user wants to build
2. **Deep-dive with AskUserQuestion**: Requirements, tech stack, file structure, verification commands, edge cases, completion criteria
3. **Explore existing code**: Use Glob/Grep/Read to understand patterns
4. **Generate PROMPT.md**: High-quality prompt following the required structure (see below)
5. **User confirmation**: Show and get approval

#### Full Ecosystem Mode (7 steps)

1. **Interview**: Same as Simple Mode, plus ask about spec decomposition — how many distinct concerns exist?
2. **Generate specs/*.md**: One file per JTBD. Each spec is immutable once approved.
3. **Initialize AGENT.md**: Generate from template (`assets/templates/AGENT.md.template`), pre-fill build/test/lint commands
4. **Generate PROMPT_plan.md**: Use `assets/templates/PROMPT_plan.md.template` as reference, fill project-specific values
5. **Generate PROMPT_build.md**: Use `assets/templates/PROMPT_build.md.template` as reference, fill project-specific values
6. **User review**: Present all documents for approval
7. **Execution guidance**: Recommend "Run Plan Loop first (2-3 iterations), then Build Loop"

### Phase 1: Plan Loop (Full Ecosystem Mode only)

```bash
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" --mode plan --max-iterations 3
```

Each iteration reads specs/*.md and generates/refines `fix_plan.md`. Stops on `<promise>PLAN COMPLETE</promise>`.

### Phase 2: Build Loop

**Simple Mode**: Uses PROMPT.md directly.

```bash
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" --max-iterations 20 --completion-promise "ALL TESTS PASSING" PROMPT.md
```

**Full Ecosystem Mode**: Each iteration picks ONE task from fix_plan.md.

```bash
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" --mode build --max-iterations 20 --completion-promise "ALL TESTS PASSING" --auto-commit
```

Options:
- `--mode plan|build` — Full Ecosystem Mode: auto-select prompt file
- `--max-iterations N` — Stop after N iterations (default: unlimited)
- `--completion-promise TEXT` — Stop on `<promise>TEXT</promise>` detection
- `--auto-commit` — Git commit after each iteration
- `--budget-per-iteration N` — USD cap per iteration (passed as `--max-cost` to claude)
- `--timeout-per-iteration SECS` — Timeout per iteration in seconds (default: 300)
- `--dry-run` — Preview config without running
- `--log-dir DIR` — Log location (default: `.ralph-loop/logs/`)

**Important**: Always run `--dry-run` first to verify config before actual execution.

### Phase 3: Monitor

- Logs: `.ralph-loop/logs/iteration-NNN.log`
- State: `.ralph-loop/state.json`
- Completion: macOS notification sound + speech

### Phase 4: Cancel

Kill the process (Ctrl+C or `kill`). The script traps both SIGINT and SIGTERM and cleans up child `claude` processes.

## PROMPT.md Required Structure (Simple Mode)

```markdown
# Task
[Specific task description]

# Requirements
[Functional and non-functional requirements]

# Tech Stack
[Languages, libraries, frameworks]

# File Structure
[File paths to create or modify — use absolute paths]

# Verification
[Test commands and verification steps]
Example: cd /path/to/project && pnpm test

# Completion
[Completion criteria]
When ALL tests pass, output exactly: <promise>ALL TESTS PASSING</promise>
```

#### Quality checklist (verify before generating)

- [ ] Task is specific and unambiguous
- [ ] File paths are absolute
- [ ] Verification commands are executable as-is
- [ ] Completion criteria use `<promise>` tag
- [ ] Content is self-contained for `claude -p` (no conversation context dependency)

## Usage Examples

### Simple Mode

```bash
# Dry run
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" --dry-run --max-iterations 10 PROMPT.md

# Run with completion promise
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" \
  --max-iterations 20 \
  --completion-promise "ALL TESTS PASSING" \
  --auto-commit \
  PROMPT.md

# Quick inline prompt
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" \
  --max-iterations 5 \
  "Refactor auth module. Output <promise>DONE</promise> when complete."
```

### Full Ecosystem Mode

```bash
# Step 1: Plan phase dry run
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" --dry-run --mode plan --max-iterations 3

# Step 2: Run plan phase
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" --mode plan --max-iterations 3

# Step 3: Build phase dry run
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" --dry-run --mode build --max-iterations 20 --completion-promise "ALL TESTS PASSING"

# Step 4: Run build phase
"${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh" \
  --mode build \
  --max-iterations 20 \
  --completion-promise "ALL TESTS PASSING" \
  --auto-commit
```

## Generated Files

```
.ralph-loop/
  state.json              # Loop state (active/completed, iteration count, mode)
  logs/
    iteration-001.log     # Each iteration's claude output
    iteration-002.log
    ...
```

`.ralph-loop/` is auto-added to `.gitignore` on first run.

## Troubleshooting

- **Script not found**: Verify `${CLAUDE_SKILL_ROOT}/scripts/camone-ralph-loop.sh` exists and is executable
- **"Cannot launch inside another session"**: The script uses `env -u CLAUDECODE` to handle this automatically. If it still fails, run from a standalone terminal
- **Orphan claude processes**: The script tracks child PIDs and kills them on SIGINT/SIGTERM. If manually killed with SIGKILL (`kill -9`), child processes may survive — use `ps aux | grep 'claude.*-p' | grep -v grep` to find and kill them
- **PROMPT_plan.md not found**: When using `--mode plan`, ensure you've run Phase 0 to generate the prompt file, or specify an explicit prompt path
