# Data Format Reference

Supported dataset formats for Unsloth fine-tuning.

## Format Overview

| Format | Use Case | Key Fields |
|--------|----------|------------|
| **Raw Corpus** | Continued pretraining | `text` |
| **Alpaca** | Single-turn instruction | `instruction`, `output` |
| **ShareGPT** | Multi-turn conversation | `conversations[{from, value}]` |
| **ChatML** | Native chat format | `messages[{role, content}]` |

## Raw Corpus Format

For continued pretraining on domain text.

```json
{"text": "The mitochondria is the powerhouse of the cell..."}
{"text": "In quantum mechanics, superposition refers to..."}
```

### Usage
```python
# Just load and use directly
dataset = load_dataset("json", data_files="corpus.jsonl", split="train")

def format_fn(example):
    return {"text": example["text"]}
```

## Alpaca Format

Most common for instruction tuning.

```json
{
  "instruction": "Summarize the following text.",
  "input": "Long article text here...",
  "output": "Brief summary of the article."
}
```

### Variations
```json
// Without input
{
  "instruction": "What is the capital of France?",
  "output": "The capital of France is Paris."
}

// With system prompt
{
  "instruction": "Translate to Spanish",
  "input": "Hello, how are you?",
  "output": "Hola, ¿cómo estás?",
  "system": "You are a helpful translator."
}
```

### Conversion to Chat
```python
def alpaca_to_chat(example):
    messages = []

    if example.get("system"):
        messages.append({"role": "system", "content": example["system"]})

    user_content = example["instruction"]
    if example.get("input"):
        user_content += f"\n\n{example['input']}"

    messages.append({"role": "user", "content": user_content})
    messages.append({"role": "assistant", "content": example["output"]})

    return {"text": tokenizer.apply_chat_template(messages, tokenize=False)}
```

## ShareGPT Format

Multi-turn conversations, common from ChatGPT exports.

```json
{
  "conversations": [
    {"from": "human", "value": "What is Python?"},
    {"from": "gpt", "value": "Python is a programming language..."},
    {"from": "human", "value": "How do I install it?"},
    {"from": "gpt", "value": "You can download Python from..."}
  ]
}
```

### Role Mappings
```
"human" → user
"gpt" → assistant
"system" → system
```

### Important: Standardize First
```python
from unsloth.chat_templates import standardize_sharegpt

# This converts ShareGPT to ChatML internally
dataset = standardize_sharegpt(dataset)
```

### Variations
```json
// With system message
{
  "conversations": [
    {"from": "system", "value": "You are a helpful assistant."},
    {"from": "human", "value": "Hello!"},
    {"from": "gpt", "value": "Hi there! How can I help?"}
  ]
}

// Alternative field names (less common)
{
  "conversations": [
    {"role": "user", "content": "Hello!"},
    {"role": "assistant", "content": "Hi!"}
  ]
}
```

## ChatML Format

Native chat format, used by many models.

```json
{
  "messages": [
    {"role": "system", "content": "You are a helpful assistant."},
    {"role": "user", "content": "What's the weather like?"},
    {"role": "assistant", "content": "I don't have access to weather data..."}
  ]
}
```

### Valid Roles
- `system` - System instructions (optional, usually first)
- `user` - Human input
- `assistant` - Model response

### Usage
```python
def chatml_to_text(example):
    return {"text": tokenizer.apply_chat_template(
        example["messages"],
        tokenize=False
    )}
```

## DPO Format

For preference optimization (chosen vs rejected).

```json
{
  "prompt": "Write a haiku about coding.",
  "chosen": "Fingers on keyboard\nLogic flows like mountain streams\nBugs become features",
  "rejected": "coding is fun\ni like to code\nthe end"
}
```

### Alternative Structure
```json
{
  "prompt": "Explain quantum computing.",
  "chosen_messages": [
    {"role": "user", "content": "Explain quantum computing."},
    {"role": "assistant", "content": "Quantum computing harnesses..."}
  ],
  "rejected_messages": [
    {"role": "user", "content": "Explain quantum computing."},
    {"role": "assistant", "content": "its computers but quantum"}
  ]
}
```

## GRPO/RL Format

For reinforcement learning with rewards.

```json
{
  "prompt": "What is 15 + 27?",
  "ground_truth": "42"
}
```

The reward function compares model output to ground_truth.

## Format Detection

```python
def detect_format(sample):
    if "conversations" in sample:
        conv = sample["conversations"]
        if isinstance(conv, list) and len(conv) > 0:
            if "from" in conv[0]:
                return "sharegpt"

    if "messages" in sample:
        msgs = sample["messages"]
        if isinstance(msgs, list) and len(msgs) > 0:
            if "role" in msgs[0]:
                return "chatml"

    if "instruction" in sample and "output" in sample:
        return "alpaca"

    if "text" in sample:
        return "raw"

    return "unknown"
```

## Common Transformations

### ShareGPT → ChatML
```python
from unsloth.chat_templates import standardize_sharegpt
dataset = standardize_sharegpt(dataset)
```

### Alpaca → ChatML
```python
def alpaca_to_chatml(example):
    messages = []
    user = example["instruction"]
    if example.get("input"):
        user += f"\n\n{example['input']}"
    messages.append({"role": "user", "content": user})
    messages.append({"role": "assistant", "content": example["output"]})
    return {"messages": messages}

dataset = dataset.map(alpaca_to_chatml)
```

### Filter Empty Examples
```python
def is_valid(example):
    if "text" in example:
        return len(example["text"].strip()) > 10
    if "messages" in example:
        return len(example["messages"]) >= 2
    return True

dataset = dataset.filter(is_valid)
```

## Dataset Sources

### Hugging Face Hub
```python
from datasets import load_dataset

# Full dataset
dataset = load_dataset("mlabonne/FineTome-100k", split="train")

# Subset
dataset = load_dataset("mlabonne/FineTome-100k", split="train[:1000]")
```

### Local Files
```python
# JSONL
dataset = load_dataset("json", data_files="data.jsonl", split="train")

# JSON
dataset = load_dataset("json", data_files="data.json", split="train")

# CSV
dataset = load_dataset("csv", data_files="data.csv", split="train")

# Parquet
dataset = load_dataset("parquet", data_files="data.parquet", split="train")
```

### Multiple Files
```python
dataset = load_dataset("json", data_files={
    "train": ["train1.jsonl", "train2.jsonl"],
    "test": "test.jsonl"
})
```

## Quality Checks

### Verify Format
```python
sample = dataset[0]
print(f"Keys: {sample.keys()}")
print(f"Format: {detect_format(sample)}")
print(f"Sample:\n{json.dumps(sample, indent=2)[:500]}")
```

### Check for Issues
```python
issues = []

for i, ex in enumerate(dataset):
    # Empty content
    if not get_text_content(ex):
        issues.append(f"Empty at {i}")

    # Mismatched format
    if detect_format(ex) != expected_format:
        issues.append(f"Format mismatch at {i}")

print(f"Found {len(issues)} issues")
```
