# /// script
# requires-python = ">=3.10"
# dependencies = [
#     "datasets>=2.18",
#     "transformers>=4.45",
#     "huggingface_hub>=0.20",
# ]
# ///
"""
Dataset Validation Script for Fine-tuning

Validates dataset format, calculates statistics, and checks Chinchilla optimality.

Usage:
    python validate_dataset.py --dataset "mlabonne/FineTome-100k"
    python validate_dataset.py --file "./data/train.jsonl"
"""

import argparse
import json
import statistics
from datasets import load_dataset
from transformers import AutoTokenizer
from huggingface_hub import HfApi

# ============================================
# FORMAT DETECTION
# ============================================

def detect_format(sample: dict) -> str:
    """Detect dataset format from a sample."""
    if "conversations" in sample:
        conv = sample["conversations"]
        if isinstance(conv, list) and len(conv) > 0:
            if "from" in conv[0] and "value" in conv[0]:
                return "sharegpt"

    if "messages" in sample:
        msgs = sample["messages"]
        if isinstance(msgs, list) and len(msgs) > 0:
            if "role" in msgs[0] and "content" in msgs[0]:
                return "chatml"

    if "instruction" in sample and "output" in sample:
        return "alpaca"

    if "text" in sample and len(sample) <= 2:  # text + maybe id
        return "raw"

    return "unknown"


def validate_schema(sample: dict, format_type: str) -> list[str]:
    """Validate that sample has required fields."""
    issues = []

    if format_type == "alpaca":
        if "instruction" not in sample:
            issues.append("Missing 'instruction' field")
        if "output" not in sample:
            issues.append("Missing 'output' field")

    elif format_type == "sharegpt":
        if "conversations" not in sample:
            issues.append("Missing 'conversations' field")
        else:
            conv = sample["conversations"]
            if not isinstance(conv, list):
                issues.append("'conversations' should be a list")
            elif len(conv) == 0:
                issues.append("'conversations' is empty")
            else:
                for i, turn in enumerate(conv):
                    if "from" not in turn:
                        issues.append(f"Turn {i} missing 'from' key")
                    if "value" not in turn:
                        issues.append(f"Turn {i} missing 'value' key")

    elif format_type == "chatml":
        if "messages" not in sample:
            issues.append("Missing 'messages' field")
        else:
            msgs = sample["messages"]
            if not isinstance(msgs, list):
                issues.append("'messages' should be a list")
            elif len(msgs) == 0:
                issues.append("'messages' is empty")
            else:
                for i, msg in enumerate(msgs):
                    if "role" not in msg:
                        issues.append(f"Message {i} missing 'role' key")
                    if "content" not in msg:
                        issues.append(f"Message {i} missing 'content' key")

    elif format_type == "raw":
        if "text" not in sample:
            issues.append("Missing 'text' field")
        elif not sample["text"]:
            issues.append("'text' field is empty")

    return issues


# ============================================
# TOKEN ANALYSIS
# ============================================

def get_text_content(sample: dict, format_type: str) -> str:
    """Extract all text from a sample."""
    if format_type == "alpaca":
        parts = [sample.get("instruction", "")]
        if sample.get("input"):
            parts.append(sample["input"])
        parts.append(sample.get("output", ""))
        return " ".join(parts)

    elif format_type == "sharegpt":
        return " ".join(
            turn.get("value", "")
            for turn in sample.get("conversations", [])
        )

    elif format_type == "chatml":
        return " ".join(
            msg.get("content", "")
            for msg in sample.get("messages", [])
        )

    else:  # raw
        return sample.get("text", "")


def count_tokens(text: str, tokenizer) -> int:
    """Count tokens in text."""
    return len(tokenizer.encode(text, add_special_tokens=False))


# ============================================
# CHINCHILLA ANALYSIS
# ============================================

MODEL_PARAMS = {
    "llama-3.1-8b": 8e9,
    "llama-3.1-70b": 70e9,
    "qwen-2.5-7b": 7e9,
    "qwen-2.5-14b": 14e9,
    "qwen-2.5-32b": 32e9,
    "gemma-2-9b": 9e9,
    "gemma-2-27b": 27e9,
    "phi-4-14b": 14e9,
    "mistral-7b": 7e9,
}


def calculate_chinchilla(
    total_tokens: int,
    model_key: str,
    use_lora: bool = True,
    lora_rank: int = 16,
) -> dict:
    """Calculate Chinchilla optimality metrics."""
    base_params = MODEL_PARAMS.get(model_key, 8e9)

    if use_lora:
        # Approximate trainable params for LoRA
        # ~0.5% of base at rank 16, scales with rank
        lora_fraction = (lora_rank / 16) * 0.005
        trainable_params = base_params * lora_fraction
    else:
        trainable_params = base_params

    optimal_tokens = 20 * trainable_params
    chinchilla_fraction = total_tokens / optimal_tokens if optimal_tokens > 0 else 0

    return {
        "base_params": base_params,
        "trainable_params": trainable_params,
        "optimal_tokens": optimal_tokens,
        "actual_tokens": total_tokens,
        "chinchilla_fraction": chinchilla_fraction,
    }


def interpret_chinchilla(fraction: float) -> str:
    """Interpret Chinchilla fraction."""
    if fraction < 0.5:
        return "Dataset may be too small for optimal training. Consider data augmentation or a smaller model."
    elif fraction <= 2.0:
        return "Good range for fine-tuning. Dataset size is reasonable for this model."
    else:
        return "Large dataset relative to trainable params. Training may take longer but could yield good results."


# ============================================
# MAIN
# ============================================

def main():
    parser = argparse.ArgumentParser(description="Validate dataset for fine-tuning")
    parser.add_argument("--dataset", help="Hugging Face dataset ID")
    parser.add_argument("--file", help="Local file path (jsonl/json)")
    parser.add_argument("--split", default="train", help="Dataset split")
    parser.add_argument("--model", default="llama-3.1-8b",
                       choices=list(MODEL_PARAMS.keys()),
                       help="Target model for Chinchilla analysis")
    parser.add_argument("--lora-rank", type=int, default=16,
                       help="LoRA rank (0 for full fine-tuning)")
    parser.add_argument("--tokenizer", default="unsloth/llama-3.1-8b-unsloth-bnb-4bit",
                       help="Tokenizer for token counting")
    parser.add_argument("--max-samples", type=int, default=10000,
                       help="Max samples for token analysis")
    args = parser.parse_args()

    if not args.dataset and not args.file:
        parser.error("Must specify --dataset or --file")

    print("=" * 60)
    print("Dataset Validation Report")
    print("=" * 60)

    # Load dataset
    print("\n1. Loading Dataset")
    print("-" * 40)

    if args.dataset:
        print(f"Source: Hugging Face - {args.dataset}")
        dataset = load_dataset(args.dataset, split=args.split)
    else:
        print(f"Source: Local file - {args.file}")
        if args.file.endswith(".jsonl"):
            dataset = load_dataset("json", data_files=args.file, split="train")
        else:
            dataset = load_dataset("json", data_files=args.file, split="train")

    print(f"Total rows: {len(dataset):,}")
    print(f"Columns: {', '.join(dataset.column_names)}")

    # Detect format
    print("\n2. Format Detection")
    print("-" * 40)

    sample = dataset[0]
    format_type = detect_format(sample)
    print(f"Detected format: {format_type.upper()}")

    if format_type == "unknown":
        print("WARNING: Could not detect format. Check your data structure.")
        print(f"Sample keys: {list(sample.keys())}")

    # Validate schema
    print("\n3. Schema Validation")
    print("-" * 40)

    issues = validate_schema(sample, format_type)
    if issues:
        print("ISSUES FOUND:")
        for issue in issues:
            print(f"  - {issue}")
    else:
        print("Schema valid!")

    # Check multiple samples for consistency
    validation_count = min(100, len(dataset))
    inconsistent = 0
    for i in range(validation_count):
        sample_format = detect_format(dataset[i])
        if sample_format != format_type:
            inconsistent += 1

    if inconsistent > 0:
        print(f"WARNING: {inconsistent}/{validation_count} samples have inconsistent format")

    # Show samples
    print("\n4. Sample Data")
    print("-" * 40)

    for i in range(min(2, len(dataset))):
        print(f"\nSample {i+1}:")
        sample = dataset[i]
        sample_json = json.dumps(sample, indent=2, ensure_ascii=False)
        if len(sample_json) > 500:
            sample_json = sample_json[:500] + "\n... (truncated)"
        print(sample_json)

    # Token analysis
    print("\n5. Token Analysis")
    print("-" * 40)

    print(f"Loading tokenizer: {args.tokenizer}")
    tokenizer = AutoTokenizer.from_pretrained(args.tokenizer)

    analysis_count = min(args.max_samples, len(dataset))
    print(f"Analyzing {analysis_count:,} samples...")

    token_counts = []
    for i in range(analysis_count):
        text = get_text_content(dataset[i], format_type)
        tokens = count_tokens(text, tokenizer)
        token_counts.append(tokens)

    # Extrapolate if we sampled
    if analysis_count < len(dataset):
        avg_tokens = statistics.mean(token_counts)
        total_tokens = int(avg_tokens * len(dataset))
        print(f"(Extrapolated from {analysis_count:,} sample)")
    else:
        total_tokens = sum(token_counts)

    print(f"\n{'Metric':<25} {'Value':>15}")
    print("-" * 42)
    print(f"{'Total tokens':<25} {total_tokens:>15,}")
    print(f"{'Min sequence':<25} {min(token_counts):>15,}")
    print(f"{'Max sequence':<25} {max(token_counts):>15,}")
    print(f"{'Mean sequence':<25} {statistics.mean(token_counts):>15,.1f}")
    print(f"{'Median sequence':<25} {statistics.median(token_counts):>15,.1f}")
    print(f"{'Std deviation':<25} {statistics.stdev(token_counts):>15,.1f}")

    # Flag concerns
    long_seqs = sum(1 for t in token_counts if t > 4096)
    short_seqs = sum(1 for t in token_counts if t < 10)

    if long_seqs > 0:
        pct = 100 * long_seqs / len(token_counts)
        print(f"\nWARNING: {long_seqs} sequences ({pct:.1f}%) exceed 4096 tokens")

    if short_seqs > 0:
        pct = 100 * short_seqs / len(token_counts)
        print(f"WARNING: {short_seqs} sequences ({pct:.1f}%) are very short (<10 tokens)")

    # Chinchilla analysis
    print("\n6. Chinchilla Optimality Analysis")
    print("-" * 40)

    use_lora = args.lora_rank > 0
    chinchilla = calculate_chinchilla(
        total_tokens=total_tokens,
        model_key=args.model,
        use_lora=use_lora,
        lora_rank=args.lora_rank,
    )

    print(f"Target model: {args.model}")
    print(f"Training mode: {'LoRA (rank ' + str(args.lora_rank) + ')' if use_lora else 'Full fine-tuning'}")
    print(f"\n{'Metric':<25} {'Value':>20}")
    print("-" * 47)
    print(f"{'Base parameters':<25} {chinchilla['base_params']:>20,.0f}")
    print(f"{'Trainable parameters':<25} {chinchilla['trainable_params']:>20,.0f}")
    print(f"{'Optimal tokens':<25} {chinchilla['optimal_tokens']:>20,.0f}")
    print(f"{'Your tokens':<25} {chinchilla['actual_tokens']:>20,}")
    print(f"{'Chinchilla fraction':<25} {chinchilla['chinchilla_fraction']:>20.2f}x")

    print(f"\nInterpretation: {interpret_chinchilla(chinchilla['chinchilla_fraction'])}")

    # Recommendations
    print("\n7. Recommendations")
    print("-" * 40)

    if format_type == "sharegpt":
        print("- Use standardize_sharegpt() before training to convert to ChatML")

    if statistics.mean(token_counts) > 2048:
        print("- Consider max_seq_length=4096 or higher")

    if len(dataset) < 100:
        print("- Small dataset: use low learning rate (1e-5) and few epochs (1-3)")

    if chinchilla['chinchilla_fraction'] < 0.5:
        print("- Consider data augmentation or using a smaller model")

    print("\n" + "=" * 60)
    print("Validation Complete")
    print("=" * 60)


if __name__ == "__main__":
    main()
