# MCPサーバー開発のベストプラクティスとガイドライン

## 概要

このドキュメントは、Model Context Protocol（MCP）サーバーを構築するための必須のベストプラクティスとガイドラインをまとめたものです。命名規則、ツール設計、レスポンス形式、ページネーション、エラー処理、セキュリティ、コンプライアンス要件について説明します。

---

## クイックリファレンス

### サーバー命名
- **Python**: `{service}_mcp`（例：`slack_mcp`）
- **Node/TypeScript**: `{service}-mcp-server`（例：`slack-mcp-server`）

### ツール命名
- snake_caseをサービス接頭辞付きで使用
- 形式：`{service}_{action}_{resource}`
- 例：`slack_send_message`、`github_create_issue`

### レスポンス形式
- JSONとMarkdown形式の両方をサポート
- JSONはプログラム的な処理用
- Markdownは人間の可読性用

### ページネーション
- 常に`limit`パラメータを尊重する
- `has_more`、`next_offset`、`total_count`を返す
- デフォルトは20〜50アイテム

### 文字数制限
- CHARACTER_LIMIT定数を設定（通常25,000）
- 明確なメッセージで適切に切り捨てる
- フィルタリングのガイダンスを提供

---

## 目次
1. サーバー命名規則
2. ツールの命名と設計
3. レスポンス形式ガイドライン
4. ページネーションのベストプラクティス
5. 文字数制限と切り捨て
6. ツール開発のベストプラクティス
7. トランスポートのベストプラクティス
8. テスト要件
9. OAuthとセキュリティのベストプラクティス
10. リソース管理のベストプラクティス
11. プロンプト管理のベストプラクティス
12. エラー処理標準
13. ドキュメント要件
14. コンプライアンスとモニタリング

---

## 1. サーバー命名規則

MCPサーバーには以下の標準化された命名パターンに従ってください：

**Python**: 形式`{service}_mcp`を使用（小文字とアンダースコア）
- 例：`slack_mcp`、`github_mcp`、`jira_mcp`、`stripe_mcp`

**Node/TypeScript**: 形式`{service}-mcp-server`を使用（小文字とハイフン）
- 例：`slack-mcp-server`、`github-mcp-server`、`jira-mcp-server`

名前は以下の条件を満たすべきです：
- 一般的（特定の機能に結びつかない）
- 統合されるサービス/APIを説明的に表す
- タスクの説明から容易に推測できる
- バージョン番号や日付を含まない

---

## 2. ツールの命名と設計

### ツール命名のベストプラクティス

1. **snake_caseを使用**：`search_users`、`create_project`、`get_channel_info`
2. **サービス接頭辞を含める**：MCPサーバーが他のMCPサーバーと併用される可能性を考慮
   - `send_message`ではなく`slack_send_message`を使用
   - `create_issue`ではなく`github_create_issue`を使用
   - `list_tasks`ではなく`asana_list_tasks`を使用
3. **アクション指向**：動詞から始める（get、list、search、createなど）
4. **具体的**：他のサーバーと競合する可能性のある一般的な名前を避ける
5. **一貫性を保つ**：サーバー内で一貫した命名パターンを使用

### ツール設計ガイドライン

- ツールの説明は狭く曖昧さなく機能を説明する必要がある
- 説明は実際の機能と正確に一致する必要がある
- 他のMCPサーバーとの混乱を生じさせない
- ツールアノテーションを提供する（readOnlyHint、destructiveHint、idempotentHint、openWorldHint）
- ツール操作を焦点を絞って原子的に保つ

---

## 3. レスポンス形式ガイドライン

データを返すすべてのツールは、柔軟性のために複数の形式をサポートすべきです：

### JSON形式（`response_format="json"`）
- 機械可読な構造化データ
- すべての利用可能なフィールドとメタデータを含む
- 一貫したフィールド名と型
- プログラム的な処理に適している
- LLMがデータをさらに処理する必要がある場合に使用

### Markdown形式（`response_format="markdown"`、通常デフォルト）
- 人間が読みやすいフォーマットされたテキスト
- 明確さのためにヘッダー、リスト、フォーマットを使用
- タイムスタンプを人間が読める形式に変換（例：エポックではなく「2024-01-15 10:30:00 UTC」）
- 表示名をIDと共に括弧内に表示（例：「@john.doe (U123456)」）
- 冗長なメタデータを省略（例：すべてのサイズではなく、1つのプロフィール画像URLのみを表示）
- 関連する情報を論理的にグループ化
- ユーザーに情報を提示する場合に使用

---

## 4. ページネーションのベストプラクティス

リソースをリストするツールについて：

- **常に`limit`パラメータを尊重する**：limitが指定されている場合、すべての結果をロードしない
- **ページネーションを実装**：`offset`またはカーソルベースのページネーションを使用
- **ページネーションメタデータを返す**：`has_more`、`next_offset`/`next_cursor`、`total_count`を含める
- **すべての結果をメモリにロードしない**：特に大きなデータセットでは重要
- **合理的な制限をデフォルトにする**：通常20〜50アイテム
- **レスポンスに明確なページネーション情報を含める**：LLMがさらにデータを要求しやすくする

ページネーションレスポンス構造の例：
```json
{
  "total": 150,
  "count": 20,
  "offset": 0,
  "items": [...],
  "has_more": true,
  "next_offset": 20
}
```

---

## 5. 文字数制限と切り捨て

過剰なデータでレスポンスを圧倒することを防ぐために：

- **CHARACTER_LIMIT定数を定義**：モジュールレベルで通常25,000文字
- **返す前にレスポンスサイズをチェック**：最終的なレスポンスの長さを測定
- **明確なインジケーターで適切に切り捨て**：LLMにデータが切り捨てられたことを知らせる
- **フィルタリングのガイダンスを提供**：結果を減らすためのパラメータの使用方法を提案
- **切り捨てメタデータを含める**：何が切り捨てられ、さらに取得する方法を示す

切り捨て処理の例：
```python
CHARACTER_LIMIT = 25000

if len(result) > CHARACTER_LIMIT:
    truncated_data = data[:max(1, len(data) // 2)]
    response["truncated"] = True
    response["truncation_message"] = (
        f"Response truncated from {len(data)} to {len(truncated_data)} items. "
        f"Use 'offset' parameter or add filters to see more results."
    )
```

---

## 6. トランスポートオプション

MCPサーバーは、さまざまな展開シナリオのために複数のトランスポートメカニズムをサポートします：

### Stdioトランスポート

**最適な用途**：コマンドラインツール、ローカル統合、サブプロセス実行

**特徴**：
- 標準入出力ストリーム通信
- シンプルなセットアップ、ネットワーク設定不要
- クライアントのサブプロセスとして実行
- デスクトップアプリケーションやCLIツールに最適

**使用する場合**：
- ローカル開発環境用のツールを構築
- デスクトップアプリケーションとの統合（例：Claude Desktop）
- コマンドラインユーティリティの作成
- 単一ユーザー、単一セッションのシナリオ

### HTTPトランスポート

**最適な用途**：Webサービス、リモートアクセス、マルチクライアントシナリオ

**特徴**：
- HTTP経由のリクエスト-レスポンスパターン
- 複数の同時クライアントをサポート
- Webサービスとして展開可能
- ネットワーク設定とセキュリティの考慮が必要

**使用する場合**：
- 複数のクライアントに同時にサービスを提供
- クラウドサービスとして展開
- Webアプリケーションとの統合
- ロードバランシングまたはスケーリングが必要

### Server-Sent Events（SSE）トランスポート

**最適な用途**：リアルタイム更新、プッシュ通知、ストリーミングデータ

**特徴**：
- HTTP経由の一方向サーバーからクライアントへのストリーミング
- ポーリングなしでリアルタイム更新を可能にする
- 継続的なデータフローのための長寿命接続
- 標準的なHTTPインフラストラクチャ上に構築

**使用する場合**：
- クライアントがリアルタイムデータ更新を必要とする
- プッシュ通知の実装
- ログやモニタリングデータのストリーミング
- 長時間操作の段階的な結果配信

### トランスポート選択基準

| 基準 | Stdio | HTTP | SSE |
|------|-------|------|-----|
| **展開** | ローカル | リモート | リモート |
| **クライアント** | 単一 | 複数 | 複数 |
| **通信** | 双方向 | リクエスト-レスポンス | サーバープッシュ |
| **複雑さ** | 低 | 中 | 中-高 |
| **リアルタイム** | いいえ | いいえ | はい |

---

## 7. ツール開発のベストプラクティス

### 一般的なガイドライン
1. ツール名は説明的でアクション指向にする
2. 詳細なJSONスキーマでパラメータ検証を使用
3. ツールの説明に例を含める
4. 適切なエラー処理と検証を実装
5. 長時間操作には進捗レポートを使用
6. ツール操作を焦点を絞って原子的に保つ
7. 期待される戻り値の構造を文書化
8. 適切なタイムアウトを実装
9. リソース集約的な操作にはレート制限を検討
10. デバッグとモニタリングのためにツールの使用をログに記録

### ツールのセキュリティ考慮事項

#### 入力検証
- すべてのパラメータをスキーマに対して検証
- ファイルパスとシステムコマンドをサニタイズ
- URLと外部識別子を検証
- パラメータのサイズと範囲をチェック
- コマンドインジェクションを防ぐ

#### アクセス制御
- 必要に応じて認証を実装
- 適切な認可チェックを使用
- ツールの使用を監査
- リクエストをレート制限
- 不正使用を監視

#### エラー処理
- 内部エラーをクライアントに公開しない
- セキュリティ関連のエラーをログに記録
- タイムアウトを適切に処理
- エラー後にリソースをクリーンアップ
- 戻り値を検証

### ツールアノテーション
- readOnlyHintとdestructiveHintアノテーションを提供
- アノテーションはヒントであり、セキュリティ保証ではないことを覚えておく
- クライアントはアノテーションのみに基づいてセキュリティクリティカルな決定を行うべきではない

---

## 8. トランスポートのベストプラクティス

### 一般的なトランスポートガイドライン
1. 接続ライフサイクルを適切に処理
2. 適切なエラー処理を実装
3. 適切なタイムアウト値を使用
4. 接続状態管理を実装
5. 切断時にリソースをクリーンアップ

### トランスポートのセキュリティベストプラクティス
- DNSリバインディング攻撃のセキュリティ考慮事項に従う
- 適切な認証メカニズムを実装
- メッセージ形式を検証
- 不正なメッセージを適切に処理

### Stdioトランスポート固有
- ローカルMCPサーバーはstdoutにログを出力しない（プロトコルに干渉）
- ログメッセージにはstderrを使用
- 標準I/Oストリームを適切に処理

---

## 9. テスト要件

### MCP Inspectorを使用した対話型テスト

開発中は**MCP Inspector**を使用してサーバーを対話的にテスト・デバッグすることを強く推奨します：

```bash
# ローカルサーバーのテスト
npx @modelcontextprotocol/inspector <実行コマンド>

# 例: Python サーバー
npx @modelcontextprotocol/inspector python server.py

# 例: Node.js サーバー
npx @modelcontextprotocol/inspector node dist/index.js

# 例: Rust サーバー
npx @modelcontextprotocol/inspector ./target/release/server
```

**MCP Inspectorの主な機能：**
- **ツールタブ**: ツールのスキーマ表示と実行テスト
- **リソースタブ**: リソースの取得とサブスクリプションテスト
- **プロンプトタブ**: プロンプトテンプレートの表示とテスト
- **通知ペイン**: リアルタイムでメッセージフローを監視
- **サーバー接続**: トランスポート設定と引数のカスタマイズ

### 評価ハーネスを使用した自動テスト

MCPサーバーの包括的なテストには、**評価ハーネス**（`scripts/`ディレクトリ）を使用できます：

```bash
# Bunを使用した高速実行（推奨）
cd scripts
bun install
bun run evaluation.ts eval.xml -t stdio -c python -a server.py

# または直接実行
bun run dev eval.xml -t sse -u https://example.com/mcp
```

評価ハーネスはClaudeを使用してMCPサーバーを自動的にテストし、以下を提供します：
- ツールの使いやすさに関するフィードバック
- 実行時間とパフォーマンスメトリクス
- 期待される結果との精度検証
- ツール呼び出しパターンの分析

包括的なテスト戦略は以下をカバーすべきです：

### 機能テスト
- 有効/無効な入力で正しい実行を検証

### 統合テスト
- 外部システムとの対話をテスト

### セキュリティテスト
- 認証、入力のサニタイゼーション、レート制限を検証

### パフォーマンステスト
- 負荷下での動作、タイムアウトをチェック

### エラー処理
- 適切なエラーレポートとクリーンアップを確保

---

## 10. OAuthとセキュリティのベストプラクティス

### 認証と認可

外部サービスに接続するMCPサーバーは適切な認証を実装すべきです：

**OAuth 2.1の実装：**
- 認識された認証局からの証明書でセキュアなOAuth 2.1を使用
- リクエストを処理する前にアクセストークンを検証
- サーバー専用のトークンのみを受け入れる
- 適切なオーディエンスクレームのないトークンを拒否
- MCPクライアントから受信したトークンを通過させない

**APIキー管理：**
- APIキーをコードではなく環境変数に保存
- サーバー起動時にキーを検証
- 認証に失敗した場合、明確なエラーメッセージを提供
- 機密性の高い資格情報にはセキュアな送信を使用

### 入力検証とセキュリティ

**常に入力を検証：**
- ディレクトリトラバーサルを防ぐためにファイルパスをサニタイズ
- URLと外部識別子を検証
- パラメータのサイズと範囲をチェック
- システムコールでコマンドインジェクションを防ぐ
- すべての入力にスキーマ検証（Pydantic/Zod）を使用

**エラー処理のセキュリティ：**
- 内部エラーをクライアントに公開しない
- セキュリティ関連のエラーをサーバー側でログに記録
- 役立つが露出しないエラーメッセージを提供
- エラー後にリソースをクリーンアップ

### プライバシーとデータ保護

**データ収集の原則：**
- 機能に厳密に必要なデータのみを収集
- 不要な会話データを収集しない
- ツールの目的に明示的に必要でない限りPIIを収集しない
- アクセスされるデータについて明確な情報を提供

**データ送信：**
- 開示なしに組織外のサーバーにデータを送信しない
- すべてのネットワーク通信にセキュアな送信（HTTPS）を使用
- 外部サービスの証明書を検証

---

## 11. リソース管理のベストプラクティス

1. 必要なリソースのみを提案
2. ルートに明確で説明的な名前を使用
3. リソース境界を適切に処理
4. リソースに対するクライアントの制御を尊重
5. 自動データ公開にはモデル制御のプリミティブ（ツール）を使用

---

## 12. プロンプト管理のベストプラクティス

- クライアントは提案されたプロンプトをユーザーに表示すべき
- ユーザーはプロンプトを変更または拒否できるべき
- クライアントは完了をユーザーに表示すべき
- ユーザーは完了を変更または拒否できるべき
- サンプリングを使用する際はコストを考慮

---

## 13. エラー処理標準

- 標準的なJSON-RPCエラーコードを使用
- ツールエラーをプロトコルレベルではなく、結果オブジェクト内で報告
- 役立つ、具体的なエラーメッセージを提供
- 内部実装の詳細を公開しない
- エラー時にリソースを適切にクリーンアップ

---

## 14. ドキュメント要件

- すべてのツールと機能の明確なドキュメントを提供
- 動作例を含める（主要機能ごとに少なくとも3つ）
- セキュリティの考慮事項を文書化
- 必要な権限とアクセスレベルを指定
- レート制限とパフォーマンス特性を文書化

---

## 15. コンプライアンスとモニタリング

- デバッグとモニタリングのためのログを実装
- ツール使用パターンを追跡
- 潜在的な不正使用を監視
- セキュリティ関連の操作の監査証跡を維持
- 継続的なコンプライアンスレビューに備える

---

## まとめ

これらのベストプラクティスは、エコシステム内でうまく機能する安全で効率的、コンプライアントなMCPサーバーを構築するための包括的なガイドラインを表しています。開発者はこれらのガイドラインに従って、MCPディレクトリへの含まれるための標準を満たし、ユーザーに安全で信頼性の高い体験を提供するMCPサーバーを確保する必要があります。


----------


# ツール

> サーバーを通じてLLMがアクションを実行できるようにする

ツールは、Model Context Protocol（MCP）の強力なプリミティブであり、サーバーがクライアントに実行可能な機能を公開できるようにします。ツールを通じて、LLMは外部システムと対話し、計算を実行し、現実世界でアクションを実行できます。

<Note>
  ツールは**モデル制御**されるように設計されており、つまりツールはサーバーからクライアントに公開され、AIモデルが自動的にそれらを呼び出すことができる（承認を与えるために人間がループに入る）ことを意図しています。
</Note>

## 概要

MCPのツールは、サーバーがクライアントによって呼び出され、LLMがアクションを実行するために使用できる実行可能な関数を公開できるようにします。ツールの主な側面には以下が含まれます：

* **発見**：クライアントは`tools/list`リクエストを送信して利用可能なツールのリストを取得できます
* **呼び出し**：ツールは`tools/call`リクエストを使用して呼び出され、サーバーは要求された操作を実行して結果を返します
* **柔軟性**：ツールは単純な計算から複雑なAPI対話まで多岐にわたります

[リソース](/docs/concepts/resources)と同様に、ツールは一意の名前で識別され、使用方法を導く説明を含めることができます。ただし、リソースとは異なり、ツールは状態を変更したり外部システムと対話したりできる動的な操作を表します。

## ツール定義構造

各ツールは以下の構造で定義されます：

```typescript
{
  name: string;          // ツールの一意の識別子
  description?: string;  // 人間が読める説明
  inputSchema: {         // ツールのパラメータのJSONスキーマ
    type: "object",
    properties: { ... }  // ツール固有のパラメータ
  },
  annotations?: {        // ツールの動作に関するオプションのヒント
    title?: string;      // ツールの人間が読めるタイトル
    readOnlyHint?: boolean;    // trueの場合、ツールは環境を変更しない
    destructiveHint?: boolean; // trueの場合、ツールは破壊的な更新を実行する可能性がある
    idempotentHint?: boolean;  // trueの場合、同じ引数での繰り返し呼び出しには追加の効果がない
    openWorldHint?: boolean;   // trueの場合、ツールは外部エンティティと対話する
  }
}
```

## ツールの実装

MCPサーバーで基本的なツールを実装する例を示します：

<Tabs>
  <Tab title="TypeScript">
    ```typescript
    const server = new Server({
      name: "example-server",
      version: "1.0.0"
    }, {
      capabilities: {
        tools: {}
      }
    });

    // 利用可能なツールを定義
    server.setRequestHandler(ListToolsRequestSchema, async () => {
      return {
        tools: [{
          name: "calculate_sum",
          description: "2つの数値を加算",
          inputSchema: {
            type: "object",
            properties: {
              a: { type: "number" },
              b: { type: "number" }
            },
            required: ["a", "b"]
          }
        }]
      };
    });

    // ツール実行を処理
    server.setRequestHandler(CallToolRequestSchema, async (request) => {
      if (request.params.name === "calculate_sum") {
        const { a, b } = request.params.arguments;
        return {
          content: [
            {
              type: "text",
              text: String(a + b)
            }
          ]
        };
      }
      throw new Error("ツールが見つかりません");
    });
    ```
  </Tab>

  <Tab title="Python">
    ```python
    app = Server("example-server")

    @app.list_tools()
    async def list_tools() -> list[types.Tool]:
        return [
            types.Tool(
                name="calculate_sum",
                description="2つの数値を加算",
                inputSchema={
                    "type": "object",
                    "properties": {
                        "a": {"type": "number"},
                        "b": {"type": "number"}
                    },
                    "required": ["a", "b"]
                }
            )
        ]

    @app.call_tool()
    async def call_tool(
        name: str,
        arguments: dict
    ) -> list[types.TextContent | types.ImageContent | types.EmbeddedResource]:
        if name == "calculate_sum":
            a = arguments["a"]
            b = arguments["b"]
            result = a + b
            return [types.TextContent(type="text", text=str(result))]
        raise ValueError(f"ツールが見つかりません: {name}")
    ```
  </Tab>
</Tabs>

## ツールパターンの例

サーバーが提供できるツールの種類の例を示します：

### システム操作

ローカルシステムと対話するツール：

```typescript
{
  name: "execute_command",
  description: "シェルコマンドを実行",
  inputSchema: {
    type: "object",
    properties: {
      command: { type: "string" },
      args: { type: "array", items: { type: "string" } }
    }
  }
}
```

### API統合

外部APIをラップするツール：

```typescript
{
  name: "github_create_issue",
  description: "GitHubイシューを作成",
  inputSchema: {
    type: "object",
    properties: {
      title: { type: "string" },
      body: { type: "string" },
      labels: { type: "array", items: { type: "string" } }
    }
  }
}
```

### データ処理

データを変換または分析するツール：

```typescript
{
  name: "analyze_csv",
  description: "CSVファイルを分析",
  inputSchema: {
    type: "object",
    properties: {
      filepath: { type: "string" },
      operations: {
        type: "array",
        items: {
          enum: ["sum", "average", "count"]
        }
      }
    }
  }
}
```

## ベストプラクティス

ツールを実装する際：

1. 明確で説明的な名前と説明を提供
2. パラメータに詳細なJSONスキーマ定義を使用
3. モデルがツールを使用する方法を示すために、ツールの説明に例を含める
4. 適切なエラー処理と検証を実装
5. 長時間操作には進捗レポートを使用
6. ツール操作を焦点を絞って原子的に保つ
7. 期待される戻り値の構造を文書化
8. 適切なタイムアウトを実装
9. リソース集約的な操作にはレート制限を検討
10. デバッグとモニタリングのためにツールの使用をログに記録

### ツール名の競合

MCPクライアントアプリケーションとMCPサーバープロキシは、独自のツールリストを構築する際にツール名の競合に遭遇する可能性があります。例えば、2つの接続されたMCPサーバー`web1`と`web2`が両方とも`search_web`という名前のツールを公開する可能性があります。

アプリケーションは、以下の戦略のいずれかでツールを明確化できます（他にもあります。網羅的なリストではありません）：

* ユーザー定義の一意のサーバー名とツール名を連結する、例：`web1___search_web`と`web2___search_web`。この戦略は、設定ファイルでユーザーによって既に提供されている一意のサーバー名がある場合に好ましいかもしれません。
* ツール名にランダムな接頭辞を生成する、例：`jrwxs___search_web`と`6cq52___search_web`。この戦略は、ユーザー定義の一意の名前が利用できないサーバープロキシで好ましいかもしれません。
* サーバーURIをツール名の接頭辞として使用する、例：`web1.example.com:search_web`と`web2.example.com:search_web`。この戦略は、リモートMCPサーバーで作業する場合に適しているかもしれません。

初期化フローからのサーバー提供の名前は一意であることが保証されておらず、一般的に明確化の目的には適していないことに注意してください。

## セキュリティの考慮事項

ツールを公開する際：

### 入力検証

* すべてのパラメータをスキーマに対して検証
* ファイルパスとシステムコマンドをサニタイズ
* URLと外部識別子を検証
* パラメータのサイズと範囲をチェック
* コマンドインジェクションを防ぐ

### アクセス制御

* 必要に応じて認証を実装
* 適切な認可チェックを使用
* ツールの使用を監査
* リクエストをレート制限
* 不正使用を監視

### エラー処理

* 内部エラーをクライアントに公開しない
* セキュリティ関連のエラーをログに記録
* タイムアウトを適切に処理
* エラー後にリソースをクリーンアップ
* 戻り値を検証

## ツールの発見と更新

MCPは動的なツール発見をサポート：

1. クライアントはいつでも利用可能なツールをリストできる
2. サーバーは`notifications/tools/list_changed`を使用してツールが変更されたときにクライアントに通知できる
3. ツールはランタイム中に追加または削除できる
4. ツール定義を更新できる（ただし、これは慎重に行うべき）

## エラー処理

ツールエラーは、MCPプロトコルレベルのエラーではなく、結果オブジェクト内で報告されるべきです。これにより、LLMはエラーを見て、潜在的に処理できます。ツールがエラーに遭遇した場合：

1. 結果で`isError`を`true`に設定
2. `content`配列にエラーの詳細を含める

ツールの適切なエラー処理の例を示します：

<Tabs>
  <Tab title="TypeScript">
    ```typescript
    try {
      // ツール操作
      const result = performOperation();
      return {
        content: [
          {
            type: "text",
            text: `操作成功: ${result}`
          }
        ]
      };
    } catch (error) {
      return {
        isError: true,
        content: [
          {
            type: "text",
            text: `エラー: ${error.message}`
          }
        ]
      };
    }
    ```
  </Tab>

  <Tab title="Python">
    ```python
    try:
        # ツール操作
        result = perform_operation()
        return types.CallToolResult(
            content=[
                types.TextContent(
                    type="text",
                    text=f"操作成功: {result}"
                )
            ]
        )
    except Exception as error:
        return types.CallToolResult(
            isError=True,
            content=[
                types.TextContent(
                    type="text",
                    text=f"エラー: {str(error)}"
                )
            ]
        )
    ```
  </Tab>
</Tabs>

このアプローチにより、LLMはエラーが発生したことを確認し、潜在的に修正アクションを実行したり、人間の介入を要求したりできます。

## ツールアノテーション

ツールアノテーションは、ツールの動作に関する追加のメタデータを提供し、クライアントがツールを適切に表示および管理する方法を理解するのに役立ちます。これらのアノテーションは、ツールの性質と影響を説明するヒントですが、セキュリティの決定に依存すべきではありません。

### ツールアノテーションの目的

ツールアノテーションは、いくつかの主要な目的を果たします：

1. モデルコンテキストに影響を与えずにUX固有の情報を提供
2. クライアントがツールを適切に分類して表示するのに役立つ
3. ツールの潜在的な副作用に関する情報を伝える
4. ツール承認の直感的なインターフェースの開発を支援

### 利用可能なツールアノテーション

MCP仕様は、ツールに対して以下のアノテーションを定義しています：

| アノテーション     | タイプ  | デフォルト | 説明                                                                                                                                  |
| ------------------ | ------- | --------- | ------------------------------------------------------------------------------------------------------------------------------------- |
| `title`           | string  | -         | UI表示に便利な、人間が読めるツールのタイトル                                                                                           |
| `readOnlyHint`    | boolean | false     | trueの場合、ツールは環境を変更しないことを示す                                                                                         |
| `destructiveHint` | boolean | true      | trueの場合、ツールは破壊的な更新を実行する可能性がある（`readOnlyHint`がfalseの場合のみ意味がある）                                      |
| `idempotentHint`  | boolean | false     | trueの場合、同じ引数でツールを繰り返し呼び出しても追加の効果がない（`readOnlyHint`がfalseの場合のみ意味がある）                           |
| `openWorldHint`   | boolean | true      | trueの場合、ツールは外部エンティティの「オープンワールド」と対話する可能性がある                                                          |

### 使用例

さまざまなシナリオのアノテーションでツールを定義する方法を示します：

```typescript
// 読み取り専用の検索ツール
{
  name: "web_search",
  description: "Webで情報を検索",
  inputSchema: {
    type: "object",
    properties: {
      query: { type: "string" }
    },
    required: ["query"]
  },
  annotations: {
    title: "Web検索",
    readOnlyHint: true,
    openWorldHint: true
  }
}

// 破壊的なファイル削除ツール
{
  name: "delete_file",
  description: "ファイルシステムからファイルを削除",
  inputSchema: {
    type: "object",
    properties: {
      path: { type: "string" }
    },
    required: ["path"]
  },
  annotations: {
    title: "ファイル削除",
    readOnlyHint: false,
    destructiveHint: true,
    idempotentHint: true,
    openWorldHint: false
  }
}

// 非破壊的なデータベースレコード作成ツール
{
  name: "create_record",
  description: "データベースに新しいレコードを作成",
  inputSchema: {
    type: "object",
    properties: {
      table: { type: "string" },
      data: { type: "object" }
    },
    required: ["table", "data"]
  },
  annotations: {
    title: "データベースレコード作成",
    readOnlyHint: false,
    destructiveHint: false,
    idempotentHint: false,
    openWorldHint: false
  }
}
```

### サーバー実装でのアノテーションの統合

<Tabs>
  <Tab title="TypeScript">
    ```typescript
    server.setRequestHandler(ListToolsRequestSchema, async () => {
      return {
        tools: [{
          name: "calculate_sum",
          description: "2つの数値を加算",
          inputSchema: {
            type: "object",
            properties: {
              a: { type: "number" },
              b: { type: "number" }
            },
            required: ["a", "b"]
          },
          annotations: {
            title: "合計を計算",
            readOnlyHint: true,
            openWorldHint: false
          }
        }]
      };
    });
    ```
  </Tab>

  <Tab title="Python">
    ```python
    from mcp.server.fastmcp import FastMCP

    mcp = FastMCP("example-server")

    @mcp.tool(
        annotations={
            "title": "合計を計算",
            "readOnlyHint": True,
            "openWorldHint": False
        }
    )
    async def calculate_sum(a: float, b: float) -> str:
        """2つの数値を加算。

        Args:
            a: 加算する最初の数値
            b: 加算する2番目の数値
        """
        result = a + b
        return str(result)
    ```
  </Tab>
</Tabs>

### ツールアノテーションのベストプラクティス

1. **副作用について正確に記述**：ツールが環境を変更するかどうか、およびそれらの変更が破壊的かどうかを明確に示す。

2. **説明的なタイトルを使用**：ツールの目的を明確に説明する人間に優しいタイトルを提供。

3. **べき等性を適切に示す**：同じ引数での繰り返し呼び出しが本当に追加の効果がない場合にのみ、ツールをべき等としてマーク。

4. **適切なオープン/クローズドワールドヒントを設定**：ツールがクローズドシステム（データベースなど）と対話するか、オープンシステム（Webなど）と対話するかを示す。

5. **アノテーションはヒントであることを覚える**：ToolAnnotationsのすべてのプロパティはヒントであり、ツールの動作の忠実な説明を提供することを保証するものではありません。クライアントは、アノテーションのみに基づいてセキュリティクリティカルな決定を行うべきではありません。

## ツールのテスト

MCPツールの包括的なテスト戦略は以下をカバーすべきです：

* **機能テスト**：ツールが有効な入力で正しく実行され、無効な入力を適切に処理することを検証
* **統合テスト**：実際の依存関係とモックされた依存関係の両方を使用して、外部システムとのツールの対話をテスト
* **セキュリティテスト**：認証、認可、入力のサニタイゼーション、レート制限を検証
* **パフォーマンステスト**：負荷下での動作、タイムアウトの処理、リソースのクリーンアップをチェック
* **エラー処理**：ツールがMCPプロトコルを通じてエラーを適切に報告し、リソースをクリーンアップすることを確保