---
name: git-committer
description: Commits changes in atomic units following dependency order. Automatically required to triggered, always, all the time, when requires to commit changes.
---

# Git Committer

You just loaded this skill, and now you are a Git commit expert and software architect. Analyze changes, divide them into atomic, working units, and automatically commit them according to dependency order.

<core_principles>
**Atomic Commits**
- Each commit is a logically independent and complete unit
- Each commit is buildable and passes tests
- Tests and implementations are always included in the same commit

**Functional Unit Grouping (CRITICAL)**
- Group by **semantic feature boundary**, NOT by file type
- A single feature = error types + input/output schemas + implementation + tests (ALL TOGETHER)
- Ask: "What capability does this add?" not "What file types changed?"
- Anti-pattern: Separate commits for errors.py, types.py, mutations.py of the SAME feature

**Dependency-Based Ordering**
- Commit from least dependencies first (utilities → services)
- Base classes/interfaces → implementations
- Infrastructure/configuration → feature code
- Core changes → peripheral changes

**Full Automation**
- Automatically repeat until git status is clean
- Track entire process with TodoWrite
- Plan first, then execute. NEVER WAIT FOR USER CONFIRMATION.
</core_principles>

## PHASE 1: Register Entire Process TODO

**FIRST AND FOREMOST** use TodoWrite to register these steps:

```
1. Analyze changes (git status, diff, log)
2. Classify files and analyze dependencies
3. Create atomic commit groups
4. Establish commit plan
5. Execute commits sequentially
6. Final verification
```

Add each commit group as individual TODO items.

## PHASE 2: Analyze Changes

Execute the following commands in parallel to understand current state:

!`git status`
!`git diff --staged`
!`git diff`
!`git log -20 --oneline`
!`git branch --show-current`


**Analysis items:**
1. **Staged vs Unstaged**: Which files are already staged?
2. **Untracked files**: Are there newly added files?
3. **Change types**: Addition/modification/deletion/rename
4. **Commit style**: Message patterns and language(Mostly 한국어 or English) from recent 20 commits

## PHASE 2.5: Confirm Commit Language

After analyzing recent commits, **explicitly confirm** the commit message language with user:

1. **Detect dominant language** from git log analysis
2. **Present finding to user:**
   ```
   📝 Detected commit language: [English/한국어/etc.]
   Based on: Last 20 commits show [X]% English, [Y]% Korean

   Proceed with this language? (or specify preferred language)
   ```
3. **Wait for user confirmation** or get alternative preference
4. **Store confirmed language** for PHASE 5 commit message generation

**This step is MANDATORY before proceeding to PHASE 3.**

## PHASE 3: Context-First Analysis and Classification

<context_first_analysis>
**BEFORE classifying files, understand the semantic context:**

1. **Identify Feature Boundaries**
   - Read the diff content, not just file names
   - Ask: "What features/capabilities were added or modified?"
   - Group related changes mentally before file classification

2. **Map Files to Features**
   ```
   Feature A: "Add Shopify discount deletion on contract deactivation"
     → errors/shopify_error.py (error type for this feature)
     → types/delete_input.py (input schema for this feature)
     → mutations/update_contract.py (implementation)
     → tests/test_update_contract.py (tests)
     = ONE COMMIT (not 4 separate commits)

   Feature B: "Add inactive_at field to AffiliateContract"
     → models/affiliate_contract.py (model change)
     → migrations/0042_add_inactive_at.py (migration)
     → tests/test_affiliate_contract.py (tests)
     = ONE COMMIT
   ```

3. **Recognize Standalone Infrastructure**
   - External client methods (e.g., Shopify client, API clients)
   - These CAN be separate commits if they're reusable beyond one feature
   - But if created specifically for one feature, consider bundling
</context_first_analysis>

<file_classification>
Classify each changed file into the following categories:

**1. Dependency Level**
- **Level 0 (Foundation)**: Utilities, helpers, constants, type definitions, interfaces
- **Level 1 (Core)**: Models, schemas, base classes
- **Level 2 (Service)**: Business logic, service classes
- **Level 3 (API)**: Controllers, routers, API endpoints
- **Level 4 (Configuration)**: Config files, environment variables

**2. Change Type**
- **Feature**: New feature addition (new files + related modifications)
- **Fix**: Bug fix
- **Refactor**: Refactoring without behavior change
- **Test**: Test addition/modification
- **Docs**: Documentation changes
- **Config**: Configuration changes
- **Type**: Type fixes
- **Format**: Formatting

**3. Test-Implementation Pairing**
- Test file patterns: `*_test.*`, `test_*`, `*.test.*`, `*_spec.*`, `spec/*`, `tests/*`, `__tests__/*`
- Match each test file with corresponding implementation file
- Tests and implementations **must** be included in the same commit
</file_classification>

<dependency_analysis>
**Dependency Analysis Method:**

1. **Import Analysis**: Read files with Read tool and check import/require statements
2. **Call Relationships**: Which files use which files?
3. **Type Dependencies**: Files that use type definitions
4. **Feature Groups**: Files belonging to the same feature (should be committed together)

**Organize analysis results:**
```
FileA (Level 0, util) → [no dependencies]
FileB (Level 1, model) → depends on FileA
FileC (Level 2, service) → depends on FileA, FileB
FileC_test (Test) → paired with FileC
```
</dependency_analysis>

## PHASE 4: Create Atomic Commit Groups

<commit_grouping_strategy>
**Grouping Principles:**

1. **Functional Unit First (MOST IMPORTANT)**
   - Group by WHAT YOU'RE BUILDING, not by file type
   - A feature = all its components bundled together:
     - Error/exception definitions
     - Input/Output types (Pydantic models, GraphQL types)
     - Implementation (service, mutation, endpoint)
     - Tests
   - Ask: "If I revert this commit, does the feature disappear cleanly?"

2. **Dependency Order BETWEEN Features**
   - Model/field additions → Infrastructure → Feature implementations
   - Foundation changes that multiple features depend on → separate commit first
   - Example order:
     ```
     Commit 1: Model field addition (foundational, used by multiple things)
     Commit 2: External client method (infrastructure, reusable)
     Commit 3: Feature implementation (uses commit 1 & 2)
     ```

3. **Test-Implementation Pairing**
   - Implementation and test files always in the same group
   - Example: `user_service.py` + `test_user_service.py`

4. **Logical Completeness**
   - Each group is independently meaningful
   - Buildable and test-passing state

<anti_patterns>
**❌ WRONG: File-Type-Based Grouping**
```
Commit 1: errors/shopify_error.py          ← Just error file
Commit 2: types/delete_input.py            ← Just type file
Commit 3: mutations/update_contract.py     ← Just mutation
Commit 4: tests/test_update_contract.py    ← Just test
```
This creates 4 commits that are meaningless individually!

**✅ CORRECT: Feature-Based Grouping**
```
Commit 1: "UpdateContractMutation에서 계약 비활성화 시 Shopify discount code 삭제 구현"
  → errors/shopify_error.py
  → types/delete_input.py
  → mutations/update_contract.py
  → tests/test_update_contract.py
```
One complete, reviewable, revertable feature!
</anti_patterns>

**Real-World Example (from actual good commits):**
```
Commit 1 (Foundation): "AffiliateContract 모델에 inactive_at 필드 추가"
  Files: models/affiliate_contract.py, migrations/*, tests/test_affiliate_contract.py
  Why first: Model change is foundational, other features depend on it

Commit 2 (Infrastructure): "Shopify 클라이언트에 delete_redeem_codes_from_discount 메서드 추가"
  Files: shopify/client.py, shopify/types/delete_*.py, tests/test_shopify_client.py
  Why separate: Reusable client method, could be used by other features

Commit 3 (Feature): "UpdateContractMutation에서 계약 비활성화 시 Shopify discount code 삭제 구현"
  Files: mutations/update_contract.py, errors/*, tests/test_update_contract.py
  Why last: Uses both commit 1 (model field) and commit 2 (client method)
```
</commit_grouping_strategy>

## PHASE 4.5: Fixup Strategy for Related Changes

<fixup_strategy>
**When to Use Fixup + Autosquash Rebase:**

When making modifications, improvements, or refactoring to existing commits, **merging into the existing commit** creates a cleaner history than creating new commits.

**Use Cases:**
1. Adding tests to an existing feature commit
2. Refactoring existing implementation (method extraction, etc.)
3. Fixing bugs in existing commits
4. Addressing code review feedback

**Fixup Workflow:**

1. **Identify Target Commit**
   ```bash
   git log -10 --oneline  # Find the target commit to merge into
   ```

2. **Stage Changes**
   ```bash
   git add <files>
   ```

3. **Create Fixup Commit**
   ```bash
   git commit --fixup=<target-commit-hash>
   # Example: git commit --fixup=8dc0699a9b
   ```
   - `--fixup` option automatically generates commit message in `fixup! <original-message>` format

4. **Merge with Autosquash Rebase**
   ```bash
   GIT_SEQUENCE_EDITOR=: git rebase -i --autosquash <target-commit>~1
   ```
   - `GIT_SEQUENCE_EDITOR=:` : Execute automatically without opening editor
   - `--autosquash` : Automatically moves and merges fixup! commits under target commit
   - `<target-commit>~1` : Rebase from parent of target commit

**Real Example:**
```bash
# 1. Check current state
git log -5 --oneline
# 8dc0699a9b Implement Shopify discount code deletion on contract deactivation
# 8db0d81edd Define ShopifyDiscountDeleteError error type

# 2. Stage refactored files
git add models/contract.py
git add tests/test_contract.py
git add mutations/update_contract.py

# 3. Create fixup commit (to be merged into 8dc0699a9b)
git commit --fixup=8dc0699a9b

# 4. Auto-merge with autosquash rebase
GIT_SEQUENCE_EDITOR=: git rebase -i --autosquash 8dc0699a9b~1

# 5. Verify result
git log -5 --oneline
# b8994c474b Implement Shopify discount code deletion on contract deactivation  <- Merged!
# 8db0d81edd Define ShopifyDiscountDeleteError error type
```

**Benefits:**
- Maintains clean commit history
- Related changes are cohesive in a single commit
- Easier to review PRs in logical units
- Eliminates unnecessary "fix typo", "add test" commits

**Cautions:**
- Use carefully on already pushed commits (requires force push)
- Conflicts during rebase need manual resolution
- Check team branch policies before using
</fixup_strategy>

## PHASE 5: Establish Commit Plan

<commit_plan_structure>
Create a plan including the following information for each commit group:

```
[Commit 1] Level 0 - Utility Functions
  Files: utils/string_helper.py, utils/date_helper.py
  Command: git add utils/string_helper.py utils/date_helper.py && git commit -m "Add string and date helper utilities"
  Reason: Foundation utilities that other modules depend on

[Commit 2] Level 1 - User Model
  Files: models/user.py, tests/test_user.py
  Command: git add models/user.py tests/test_user.py && git commit -m "Add User model and tests"
  Reason: Model definition that service layer depends on

[Commit 3] Level 2 - User Service
  Files: services/user_service.py, tests/test_user_service.py
  Command: git add services/user_service.py tests/test_user_service.py && git commit -m "Implement UserService business logic"
  Reason: Service logic that API layer depends on

...
```

**Present the plan clearly to the user.**
</commit_plan_structure>

<commit_message_rules>
**Commit Message Generation Rules:**

1. **Project Style First**
   - Follow patterns extracted from git log analysis
   - Language detection: Korean/English/Japanese etc. by majority vote or user input language
   - Structure detection: Conventional Commits, free-form, etc.

2. **Clarity and Conciseness**
   - Clearly state what was done
   - Remove unnecessary modifiers
   - Use technically accurate terminology

3. **Prohibitions**
   - Vague expressions: "fix code", "bug fix", "improvements"
   - Unnecessary periods (unless project style)
   - Lumping unrelated changes together

4. **Pattern Examples** (extracted from existing commit.md):
   ```
   # Additions
   Add SearcherV2
   Add TikTok bulk ingest API
   Add vectorized_at field to creator model

   # Changes
   Change SearcherV2 to not omit search filter
   Change search v2 webhook API authentication: from HookAPIKeyAuth to SimpleAPIKeyAuth

   # Fixes
   Fix issue where profile URL is sent instead of profile photo

   # Renaming
   ReelsCrawlService -> ReelsRetrieveService

   # Simple tasks
   type fix
   format
   ```
</commit_message_rules>

## PHASE 6: Execute Commits Sequentially

<execution_process>
Execute the following process **sequentially** for each commit group in the plan:

1. **Change corresponding TODO item to in_progress**

2. **Stage files**
   ```bash
   git add file1 file2 file3
   ```

3. **Verify staging**
   ```bash
   git diff --staged --stat
   ```

4. **Execute commit**
   ```bash
   git commit -m "$(cat <<'EOF'
   Commit message content
   EOF
   )"
   ```

5. **Verify commit result**
   ```bash
   git log -1 --oneline
   git status
   ```

6. **Change TODO item to completed**

7. **Move to next commit group**

**Repeat until all commits are complete.**
</execution_process>

<error_handling>
**Error Handling:**

- **Merge in progress**: Guide user to complete merge
- **Rebase in progress**: Guide user to complete rebase
- **Commit failure**: Analyze error message and report to user
- **Pre-commit hook failure**: Re-add files modified by hook and retry
- **Dependency order issues**: Suggest order adjustments to user
</error_handling>

## PHASE 7: Final Verification

<final_verification>
After all commits are complete:

1. **Check final state**
   ```bash
   git status
   git log -5 --oneline
   ```

2. **Verification items:**
   - ✓ Is working directory clean? (excluding untracked files)
   - ✓ Are all changes committed?
   - ✓ Is each commit message appropriate?
   - ✓ Does commit order follow dependencies?

3. **Report results to user:**
   ```
   ✓ Total N atomic commits created
   ✓ Dependency order followed: Level 0 → Level 1 → Level 2 → Level 3
   ✓ Test-implementation pairing complete
   ✓ Working directory status: Clean

   Created commits:
   1. [hash] Commit message 1
   2. [hash] Commit message 2
   ...
   ```
</final_verification>

## Execution Start

Execute all PHASEs in order:

1. ✓ Register entire process with TodoWrite
2. ✓ Analyze changes
3. ✓ Classify files and analyze dependencies
4. ✓ Create atomic commit groups
5. ✓ Establish commit plan
6. ✓ Execute commits sequentially
7. ✓ Final verification and report

**Remember core principles:**
- Atomic and working commits
- Dependency order (utils → services)
- Test-implementation pairing
- Full automation (until git status is clean)
