# Cloudflare Bindings to Bun: Replacement Guide

Complete guide for replacing Cloudflare Workers bindings (KV, R2, D1, Durable Objects) with standard solutions in Bun.

## KV Namespace → Key-Value Stores

Cloudflare KV is a globally distributed key-value store. Replace with Redis, Upstash, or similar.

### Redis Replacement (Recommended)

**Cloudflare KV:**
```typescript
export default {
  async fetch(request: Request, env: Env) {
    // Get
    const value = await env.MY_KV.get('key');
    const json = await env.MY_KV.get('key', { type: 'json' });

    // Put
    await env.MY_KV.put('key', 'value');
    await env.MY_KV.put('key', 'value', { expirationTtl: 3600 });

    // Delete
    await env.MY_KV.delete('key');

    // List
    const keys = await env.MY_KV.list();
  },
};
```

**Bun with Redis:**
```typescript
import { Redis } from 'ioredis';

const redis = new Redis(process.env.REDIS_URL || 'redis://localhost:6379');

Bun.serve({
  async fetch(request: Request) {
    // Get
    const value = await redis.get('key');
    const json = JSON.parse(await redis.get('key') || 'null');

    // Put
    await redis.set('key', 'value');
    await redis.setex('key', 3600, 'value'); // With TTL

    // Delete
    await redis.del('key');

    // List (scan pattern)
    const keys = await redis.keys('*');
    // Or use SCAN for production:
    // const stream = redis.scanStream();
  },
});
```

**Installation:**
```bash
bun add ioredis
```

### Upstash Redis (Edge-Compatible)

For edge-like performance:

```typescript
import { Redis } from '@upstash/redis';

const redis = new Redis({
  url: process.env.UPSTASH_REDIS_REST_URL!,
  token: process.env.UPSTASH_REDIS_REST_TOKEN!,
});

// Same API as ioredis
await redis.set('key', 'value');
const value = await redis.get('key');
```

### API Comparison

| KV Operation | Redis Equivalent |
|--------------|------------------|
| `get(key)` | `redis.get(key)` |
| `get(key, {type: 'json'})` | `JSON.parse(await redis.get(key))` |
| `put(key, value)` | `redis.set(key, value)` |
| `put(key, value, {expirationTtl: n})` | `redis.setex(key, n, value)` |
| `delete(key)` | `redis.del(key)` |
| `list()` | `redis.keys('*')` or `redis.scan()` |

## R2 Bucket → Object Storage

Cloudflare R2 is S3-compatible object storage. Replace with AWS S3, MinIO, or similar.

### AWS S3 Replacement

**Cloudflare R2:**
```typescript
export default {
  async fetch(request: Request, env: Env) {
    // Get object
    const object = await env.MY_BUCKET.get('file.txt');
    const text = await object?.text();
    const json = await object?.json();

    // Put object
    await env.MY_BUCKET.put('file.txt', 'content', {
      httpMetadata: {
        contentType: 'text/plain',
      },
    });

    // Delete object
    await env.MY_BUCKET.delete('file.txt');

    // List objects
    const list = await env.MY_BUCKET.list();
  },
};
```

**Bun with AWS S3:**
```typescript
import { S3Client, GetObjectCommand, PutObjectCommand, DeleteObjectCommand, ListObjectsV2Command } from '@aws-sdk/client-s3';

const s3 = new S3Client({
  region: process.env.AWS_REGION || 'us-east-1',
  credentials: {
    accessKeyId: process.env.AWS_ACCESS_KEY_ID!,
    secretAccessKey: process.env.AWS_SECRET_ACCESS_KEY!,
  },
});

const BUCKET = 'my-bucket';

Bun.serve({
  async fetch(request: Request) {
    // Get object
    const getCmd = new GetObjectCommand({
      Bucket: BUCKET,
      Key: 'file.txt',
    });
    const response = await s3.send(getCmd);
    const text = await response.Body?.transformToString();
    const json = JSON.parse(text || '{}');

    // Put object
    const putCmd = new PutObjectCommand({
      Bucket: BUCKET,
      Key: 'file.txt',
      Body: 'content',
      ContentType: 'text/plain',
    });
    await s3.send(putCmd);

    // Delete object
    const delCmd = new DeleteObjectCommand({
      Bucket: BUCKET,
      Key: 'file.txt',
    });
    await s3.send(delCmd);

    // List objects
    const listCmd = new ListObjectsV2Command({
      Bucket: BUCKET,
    });
    const list = await s3.send(listCmd);
  },
});
```

**Installation:**
```bash
bun add @aws-sdk/client-s3
```

### MinIO (Self-Hosted S3-Compatible)

```typescript
import { S3Client } from '@aws-sdk/client-s3';

const s3 = new S3Client({
  endpoint: 'http://localhost:9000',
  region: 'us-east-1',
  credentials: {
    accessKeyId: 'minioadmin',
    secretAccessKey: 'minioadmin',
  },
  forcePathStyle: true, // Required for MinIO
});

// Same API as AWS S3
```

## D1 Database → SQL Databases

Cloudflare D1 is SQLite at the edge. Replace with PostgreSQL, MySQL, or SQLite.

### PostgreSQL Replacement

**Cloudflare D1:**
```typescript
export default {
  async fetch(request: Request, env: Env) {
    // Query
    const result = await env.DB.prepare('SELECT * FROM users WHERE id = ?')
      .bind(1)
      .first();

    // Execute
    await env.DB.prepare('INSERT INTO users (name) VALUES (?)')
      .bind('Alice')
      .run();

    // Batch
    const batch = await env.DB.batch([
      env.DB.prepare('INSERT INTO users (name) VALUES (?)').bind('Bob'),
      env.DB.prepare('INSERT INTO users (name) VALUES (?)').bind('Charlie'),
    ]);
  },
};
```

**Bun with PostgreSQL:**
```typescript
import { Pool } from 'pg';

const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
});

Bun.serve({
  async fetch(request: Request) {
    // Query
    const result = await pool.query(
      'SELECT * FROM users WHERE id = $1',
      [1]
    );
    const user = result.rows[0];

    // Execute
    await pool.query(
      'INSERT INTO users (name) VALUES ($1)',
      ['Alice']
    );

    // Transaction (for batch)
    const client = await pool.connect();
    try {
      await client.query('BEGIN');
      await client.query('INSERT INTO users (name) VALUES ($1)', ['Bob']);
      await client.query('INSERT INTO users (name) VALUES ($1)', ['Charlie']);
      await client.query('COMMIT');
    } catch (e) {
      await client.query('ROLLBACK');
      throw e;
    } finally {
      client.release();
    }
  },
});
```

**Installation:**
```bash
bun add pg
```

### Prisma ORM (Recommended for Type Safety)

```typescript
import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

Bun.serve({
  async fetch(request: Request) {
    // Query
    const user = await prisma.user.findUnique({
      where: { id: 1 },
    });

    // Insert
    await prisma.user.create({
      data: { name: 'Alice' },
    });

    // Batch
    await prisma.$transaction([
      prisma.user.create({ data: { name: 'Bob' } }),
      prisma.user.create({ data: { name: 'Charlie' } }),
    ]);
  },
});
```

### Bun's SQLite (For D1-like Experience)

```typescript
import { Database } from 'bun:sqlite';

const db = new Database('mydb.sqlite');

Bun.serve({
  async fetch(request: Request) {
    // Query
    const query = db.query('SELECT * FROM users WHERE id = ?');
    const user = query.get(1);

    // Execute
    db.run('INSERT INTO users (name) VALUES (?)', ['Alice']);

    // Batch (transaction)
    db.transaction(() => {
      db.run('INSERT INTO users (name) VALUES (?)', ['Bob']);
      db.run('INSERT INTO users (name) VALUES (?)', ['Charlie']);
    })();
  },
});
```

## Durable Objects → Traditional Architecture

Durable Objects provide stateful serverless objects. Replace with traditional architecture patterns.

### Session Management (Durable Object → Redis)

**Cloudflare Durable Objects:**
```typescript
export class SessionDurableObject {
  state: DurableObjectState;
  sessions: Map<string, any>;

  constructor(state: DurableObjectState) {
    this.state = state;
    this.sessions = new Map();
  }

  async fetch(request: Request) {
    const sessionId = new URL(request.url).searchParams.get('id');
    const session = this.sessions.get(sessionId!);

    return Response.json({ session });
  }
}
```

**Bun with Redis:**
```typescript
import { Redis } from 'ioredis';

const redis = new Redis();

Bun.serve({
  async fetch(request: Request) {
    const url = new URL(request.url);
    const sessionId = url.searchParams.get('id');

    const session = await redis.get(`session:${sessionId}`);

    return Response.json({
      session: session ? JSON.parse(session) : null
    });
  },
});
```

### Realtime Collaboration (Durable Object → WebSocket Server)

**Cloudflare Durable Objects:**
```typescript
export class RoomDurableObject {
  state: DurableObjectState;
  connections: Set<WebSocket>;

  async fetch(request: Request) {
    const [client, server] = Object.values(new WebSocketPair());

    server.accept();
    this.connections.add(server);

    server.addEventListener('message', (msg) => {
      for (const conn of this.connections) {
        if (conn !== server) conn.send(msg.data);
      }
    });

    return new Response(null, { status: 101, webSocket: client });
  }
}
```

**Bun WebSocket Server:**
```typescript
const rooms = new Map<string, Set<any>>();

Bun.serve({
  fetch(request, server) {
    const url = new URL(request.url);
    const roomId = url.searchParams.get('room');

    const success = server.upgrade(request, {
      data: { roomId },
    });

    if (success) return undefined;
    return new Response('WebSocket upgrade failed', { status: 400 });
  },

  websocket: {
    open(ws) {
      const { roomId } = ws.data;
      if (!rooms.has(roomId)) rooms.set(roomId, new Set());
      rooms.get(roomId)!.add(ws);
    },

    message(ws, message) {
      const { roomId } = ws.data;
      const room = rooms.get(roomId);

      if (room) {
        for (const conn of room) {
          if (conn !== ws) conn.send(message);
        }
      }
    },

    close(ws) {
      const { roomId } = ws.data;
      rooms.get(roomId)?.delete(ws);
    },
  },
});
```

## Service Bindings → HTTP Calls

**Cloudflare Service Bindings:**
```typescript
export default {
  async fetch(request: Request, env: Env) {
    const response = await env.MY_SERVICE.fetch(request);
    return response;
  },
};
```

**Bun (standard HTTP):**
```typescript
Bun.serve({
  async fetch(request: Request) {
    const response = await fetch('http://my-service.internal/api', {
      method: request.method,
      headers: request.headers,
      body: request.body,
    });

    return response;
  },
});
```

## Queue → Message Queues

**Cloudflare Queues:**
```typescript
export default {
  async fetch(request: Request, env: Env) {
    await env.MY_QUEUE.send({ id: 123, data: 'value' });
  },

  async queue(batch: MessageBatch, env: Env) {
    for (const message of batch.messages) {
      await processMessage(message.body);
    }
  },
};
```

**Bun with BullMQ (Redis-based):**
```typescript
import { Queue, Worker } from 'bullmq';

const queue = new Queue('my-queue', {
  connection: { host: 'localhost', port: 6379 },
});

Bun.serve({
  async fetch(request: Request) {
    await queue.add('job', { id: 123, data: 'value' });
    return new Response('Queued');
  },
});

// Worker (separate process)
const worker = new Worker('my-queue', async (job) => {
  await processMessage(job.data);
}, {
  connection: { host: 'localhost', port: 6379 },
});
```

## Summary Table

| Cloudflare Binding | Bun Replacement | Package |
|--------------------|-----------------|---------|
| KV Namespace | Redis, Upstash | `ioredis`, `@upstash/redis` |
| R2 Bucket | S3, MinIO | `@aws-sdk/client-s3` |
| D1 Database | PostgreSQL, MySQL, SQLite | `pg`, `mysql2`, `bun:sqlite`, `@prisma/client` |
| Durable Objects | Redis + Architecture | `ioredis` + design patterns |
| Service Bindings | HTTP fetch | Built-in `fetch` |
| Queues | BullMQ, RabbitMQ | `bullmq`, `amqplib` |
| Analytics Engine | ClickHouse, TimescaleDB | Database-specific clients |
| Email (MailChannels) | Resend, SendGrid | `resend`, `@sendgrid/mail` |

## Environment Variables

All binding references should use environment variables:

```bash
# .env
REDIS_URL=redis://localhost:6379
DATABASE_URL=postgresql://user:pass@localhost:5432/db
AWS_ACCESS_KEY_ID=...
AWS_SECRET_ACCESS_KEY=...
AWS_REGION=us-east-1
S3_BUCKET=my-bucket
```

Access in code:
```typescript
const redis = new Redis(process.env.REDIS_URL);
```

This makes testing and deployment across environments easier.
