# Cloudflare Workers to Bun: Deployment Strategies

Guide for transitioning from edge deployment (Cloudflare Workers) to traditional server deployment (Bun).

## Deployment Architecture Comparison

### Cloudflare Workers

- **Location**: Global edge network (~300 locations)
- **Execution**: V8 isolates (shared infrastructure)
- **Scaling**: Automatic, per-request
- **Cold Start**: ~0ms (isolates)
- **Pricing**: Pay-per-request
- **State**: Ephemeral (Durable Objects for state)
- **Limits**: CPU time limits, memory limits

### Bun Server

- **Location**: Self-hosted or cloud regions
- **Execution**: Dedicated containers/VMs
- **Scaling**: Manual or auto-scaling groups
- **Cold Start**: Minimal (~10ms with Bun)
- **Pricing**: Server/container costs
- **State**: Can maintain in-memory state
- **Limits**: Based on server resources

## Deployment Options

### 1. Docker Deployment (Recommended)

Use the `bun-deploy` skill for complete Docker deployment guide.

**Basic Dockerfile:**
```dockerfile
FROM oven/bun:1-alpine AS deps
WORKDIR /app
COPY package.json bun.lockb ./
RUN bun install --frozen-lockfile --production

FROM oven/bun:1-alpine
WORKDIR /app
COPY --from=deps /app/node_modules ./node_modules
COPY . .

EXPOSE 3000
CMD ["bun", "run", "src/index.ts"]
```

**Deploy to:**
- Docker Hub / GitHub Container Registry
- AWS ECS / Fargate
- Google Cloud Run
- Azure Container Instances
- DigitalOcean App Platform
- Fly.io
- Railway

### 2. Platform-as-a-Service (PaaS)

#### Railway

```bash
# Install Railway CLI
npm install -g @railway/cli

# Login and deploy
railway login
railway init
railway up
```

**railway.json:**
```json
{
  "build": {
    "builder": "NIXPACKS"
  },
  "deploy": {
    "startCommand": "bun run src/index.ts",
    "restartPolicyType": "ON_FAILURE"
  }
}
```

#### Fly.io

```bash
# Install Fly CLI
curl -L https://fly.io/install.sh | sh

# Initialize and deploy
fly launch
fly deploy
```

**fly.toml:**
```toml
app = "my-bun-app"

[build]
  image = "oven/bun:1"

[[services]]
  internal_port = 3000
  protocol = "tcp"

  [[services.ports]]
    port = 80
    handlers = ["http"]

  [[services.ports]]
    port = 443
    handlers = ["tls", "http"]
```

### 3. Cloud Platforms

#### AWS (ECS/Fargate)

```bash
# Build and push Docker image
docker build -t my-bun-app .
docker tag my-bun-app:latest ${AWS_ACCOUNT_ID}.dkr.ecr.${REGION}.amazonaws.com/my-bun-app:latest
docker push ${AWS_ACCOUNT_ID}.dkr.ecr.${REGION}.amazonaws.com/my-bun-app:latest

# Deploy via ECS
aws ecs update-service --cluster my-cluster --service my-service --force-new-deployment
```

**task-definition.json:**
```json
{
  "family": "my-bun-app",
  "networkMode": "awsvpc",
  "requiresCompatibilities": ["FARGATE"],
  "cpu": "256",
  "memory": "512",
  "containerDefinitions": [
    {
      "name": "app",
      "image": "${AWS_ACCOUNT_ID}.dkr.ecr.${REGION}.amazonaws.com/my-bun-app:latest",
      "portMappings": [
        {
          "containerPort": 3000,
          "protocol": "tcp"
        }
      ],
      "environment": [
        {
          "name": "NODE_ENV",
          "value": "production"
        }
      ]
    }
  ]
}
```

#### Google Cloud Run

```bash
# Build and deploy
gcloud builds submit --tag gcr.io/${PROJECT_ID}/my-bun-app
gcloud run deploy my-bun-app \
  --image gcr.io/${PROJECT_ID}/my-bun-app \
  --platform managed \
  --region us-central1 \
  --allow-unauthenticated
```

#### Azure Container Instances

```bash
# Create resource group
az group create --name myResourceGroup --location eastus

# Deploy container
az container create \
  --resource-group myResourceGroup \
  --name my-bun-app \
  --image myregistry.azurecr.io/my-bun-app:latest \
  --dns-name-label my-bun-app \
  --ports 3000
```

### 4. Kubernetes (For Production at Scale)

See `bun-deploy` skill for complete Kubernetes manifests.

**Basic deployment.yaml:**
```yaml
apiVersion: apps/v1
kind: Deployment
metadata:
  name: bun-app
spec:
  replicas: 3
  selector:
    matchLabels:
      app: bun-app
  template:
    metadata:
      labels:
        app: bun-app
    spec:
      containers:
      - name: app
        image: my-bun-app:latest
        ports:
        - containerPort: 3000
        resources:
          requests:
            memory: "128Mi"
            cpu: "100m"
          limits:
            memory: "512Mi"
            cpu: "500m"
---
apiVersion: v1
kind: Service
metadata:
  name: bun-app
spec:
  selector:
    app: bun-app
  ports:
  - port: 80
    targetPort: 3000
  type: LoadBalancer
```

## CDN and Edge Caching

Since Bun runs on origin servers, add a CDN for edge caching:

### Cloudflare (as CDN)

Keep using Cloudflare for caching:

```typescript
// Set cache headers in Bun
Bun.serve({
  fetch(request) {
    return new Response('Hello', {
      headers: {
        'Cache-Control': 'public, max-age=3600',
        'CDN-Cache-Control': 'max-age=86400', // Cloudflare respects this
      }
    });
  },
});
```

### Fastly / Akamai / AWS CloudFront

Similar cache header strategies work across CDNs:

```typescript
const headers = {
  'Cache-Control': 'public, max-age=3600, s-maxage=86400',
  'Surrogate-Control': 'max-age=604800',
  'Vary': 'Accept-Encoding',
};
```

## Multi-Region Deployment

### Global Distribution

**Option 1: Deploy to multiple regions**
```
- us-east-1 (AWS Virginia)
- eu-west-1 (AWS Ireland)
- ap-southeast-1 (AWS Singapore)
```

Use GeoDNS (Route 53, Cloudflare) to route users to nearest region.

**Option 2: Edge caching with single origin**
```
Cloudflare CDN (global) → Bun server (single region)
```

Most traffic served from edge cache, origin handles cache misses.

## Scaling Strategies

### Horizontal Scaling

**Auto-scaling group (AWS):**
```yaml
# Auto Scaling Group configuration
MinSize: 2
MaxSize: 10
TargetTrackingScaling:
  - Type: TargetTrackingScaling
    TargetValue: 70.0
    PredefinedMetricSpecification:
      PredefinedMetricType: ASGAverageCPUUtilization
```

**Kubernetes HPA:**
```yaml
apiVersion: autoscaling/v2
kind: HorizontalPodAutoscaler
metadata:
  name: bun-app-hpa
spec:
  scaleTargetRef:
    apiVersion: apps/v1
    kind: Deployment
    name: bun-app
  minReplicas: 2
  maxReplicas: 10
  metrics:
  - type: Resource
    resource:
      name: cpu
      target:
        type: Utilization
        averageUtilization: 70
```

### Vertical Scaling

Increase container resources:
```yaml
resources:
  requests:
    memory: "256Mi"  # Increased from 128Mi
    cpu: "200m"      # Increased from 100m
  limits:
    memory: "1Gi"    # Increased from 512Mi
    cpu: "1000m"     # Increased from 500m
```

## Monitoring and Observability

### Application Performance Monitoring (APM)

**Datadog:**
```typescript
import tracer from 'dd-trace';

tracer.init({
  service: 'bun-app',
  env: 'production',
});

Bun.serve({
  fetch(request) {
    const span = tracer.scope().active();
    span?.setTag('http.url', request.url);

    return new Response('Hello');
  },
});
```

**New Relic:**
```typescript
import newrelic from 'newrelic';

Bun.serve({
  fetch(request) {
    return newrelic.startWebTransaction(request.url, async () => {
      return new Response('Hello');
    });
  },
});
```

### Logging

**Structured logging:**
```typescript
import pino from 'pino';

const logger = pino({
  level: process.env.LOG_LEVEL || 'info',
  transport: {
    target: 'pino-pretty',
    options: {
      colorize: true
    }
  }
});

Bun.serve({
  fetch(request) {
    logger.info({ url: request.url, method: request.method }, 'Request received');

    return new Response('Hello');
  },
});
```

**Log aggregation:**
- CloudWatch Logs (AWS)
- Cloud Logging (GCP)
- Azure Monitor
- Datadog Logs
- Elasticsearch/Logstash/Kibana (ELK)

## Health Checks and Load Balancing

**Health check endpoint:**
```typescript
Bun.serve({
  fetch(request) {
    const url = new URL(request.url);

    if (url.pathname === '/health') {
      return Response.json({
        status: 'healthy',
        uptime: process.uptime(),
        timestamp: Date.now(),
      });
    }

    // Your app logic
  },
});
```

**Load balancer configuration (ALB):**
```yaml
HealthCheck:
  Path: /health
  Protocol: HTTP
  Port: 3000
  HealthyThresholdCount: 2
  UnhealthyThresholdCount: 3
  Interval: 30
  Timeout: 5
```

## CI/CD Pipeline

### GitHub Actions

```yaml
name: Deploy to Production

on:
  push:
    branches: [main]

jobs:
  deploy:
    runs-on: ubuntu-latest

    steps:
      - uses: actions/checkout@v4

      - uses: oven-sh/setup-bun@v1
        with:
          bun-version: latest

      - name: Install dependencies
        run: bun install --frozen-lockfile

      - name: Run tests
        run: bun test

      - name: Build Docker image
        run: docker build -t my-bun-app:${{ github.sha }} .

      - name: Push to registry
        run: |
          echo ${{ secrets.DOCKER_PASSWORD }} | docker login -u ${{ secrets.DOCKER_USERNAME }} --password-stdin
          docker tag my-bun-app:${{ github.sha }} myregistry/my-bun-app:latest
          docker push myregistry/my-bun-app:latest

      - name: Deploy to production
        run: |
          # Deploy command (e.g., kubectl, aws ecs update-service)
          kubectl set image deployment/bun-app app=myregistry/my-bun-app:latest
```

## Cost Comparison

### Cloudflare Workers

- **Free Tier**: 100k requests/day
- **Paid**: $5/month for 10M requests
- **Cost Model**: Pay-per-request

### Bun on Cloud

**AWS Fargate (example):**
- **Instance**: 0.25 vCPU, 0.5GB RAM
- **Cost**: ~$15/month (running 24/7)
- **Traffic**: Unlimited requests (within instance capacity)
- **Scaling**: Additional containers as needed

**DigitalOcean (example):**
- **Basic Droplet**: 1 vCPU, 1GB RAM
- **Cost**: $6/month
- **Traffic**: 1TB included

**Trade-offs:**
- Workers: Better for low/variable traffic
- Bun: Better for consistent traffic or high request volumes

## Migration Deployment Strategy

### Zero-Downtime Migration

**Phase 1: Shadow deployment**
- Deploy Bun server
- Keep Cloudflare Worker running
- Send duplicate requests to both (via Worker)
- Compare responses

**Phase 2: Gradual rollout**
- Route 10% traffic to Bun
- Monitor metrics (latency, errors)
- Gradually increase to 100%

**Phase 3: Full cutover**
- Route 100% traffic to Bun
- Keep Worker as backup for 1 week
- Decommission Worker

### DNS Strategy

```
1. Worker: worker.example.com (all traffic)
2. Add Bun: api.example.com
3. Test Bun thoroughly
4. Update DNS: api.example.com → Bun server
5. Deprecate worker.example.com
```

## Best Practices

1. **Use CDN**: Add Cloudflare/Fastly in front for edge caching
2. **Multi-region**: Deploy to multiple regions for low latency
3. **Auto-scaling**: Configure HPA or auto-scaling groups
4. **Monitoring**: Set up APM and logging from day one
5. **Health checks**: Implement robust health endpoints
6. **Graceful shutdown**: Handle SIGTERM properly
7. **Container optimization**: Use multi-stage Docker builds
8. **Security**: Use secrets management, not env files
9. **CI/CD**: Automate testing and deployment
10. **Rollback plan**: Keep previous versions for quick rollback

## Summary

| Aspect | Cloudflare Workers | Bun Deployment |
|--------|-------------------|----------------|
| **Complexity** | Low | Medium-High |
| **Control** | Limited | Full |
| **Cost (low traffic)** | Lower | Higher (minimum server cost) |
| **Cost (high traffic)** | Higher (per-request) | Lower (fixed server cost) |
| **Latency** | Lower (global edge) | Higher (regional origin) |
| **Cold Start** | ~0ms | Minimal (~10ms) |
| **State** | Limited (DO) | Flexible |
| **CPU Limits** | 10-50ms | Unlimited |

Choose Bun deployment when you need:
- Full control over infrastructure
- Long-running computations
- Complex state management
- Cost predictability at scale
- Integration with existing services
