# Cloudflare Workers to Bun: Runtime API Mapping

Complete mapping of Cloudflare Workers runtime APIs to Bun equivalents.

## Core APIs (Identical)

These Web Standard APIs work the same in both:

| API | Status | Notes |
|-----|--------|-------|
| `Request` | ✅ Identical | Standard Web API |
| `Response` | ✅ Identical | Standard Web API |
| `Headers` | ✅ Identical | Standard Web API |
| `URL` | ✅ Identical | Standard Web API |
| `URLSearchParams` | ✅ Identical | Standard Web API |
| `fetch()` | ✅ Identical | Standard Web API |
| `crypto.*` | ✅ Identical | Web Crypto API |
| `TextEncoder` | ✅ Identical | Standard Web API |
| `TextDecoder` | ✅ Identical | Standard Web API |
| `AbortController` | ✅ Identical | Standard Web API |
| `ReadableStream` | ✅ Identical | Streams API |
| `WritableStream` | ✅ Identical | Streams API |

## Worker Entry Points

### Module Worker (Modern)

**Cloudflare:**
```typescript
export default {
  async fetch(
    request: Request,
    env: Env,
    ctx: ExecutionContext
  ): Promise<Response> {
    return new Response('Hello World');
  },
};
```

**Bun:**
```typescript
Bun.serve({
  port: 3000,
  async fetch(request: Request): Promise<Response> {
    return new Response('Hello World');
  },
});
```

### Service Worker (Legacy)

**Cloudflare:**
```typescript
addEventListener('fetch', (event: FetchEvent) => {
  event.respondWith(handleRequest(event.request));
});

async function handleRequest(request: Request): Promise<Response> {
  return new Response('Hello World');
}
```

**Bun:**
```typescript
Bun.serve({
  port: 3000,
  async fetch(request: Request): Promise<Response> {
    return new Response('Hello World');
  },
});
```

## ExecutionContext Methods

Cloudflare's `ctx` parameter doesn't exist in Bun. Replace patterns:

### ctx.waitUntil()

**Cloudflare:**
```typescript
export default {
  async fetch(request: Request, env: Env, ctx: ExecutionContext) {
    // Don't wait for this to complete
    ctx.waitUntil(
      logAnalytics(request)
    );

    return new Response('OK');
  },
};
```

**Bun:**
```typescript
Bun.serve({
  async fetch(request: Request) {
    // Fire and forget
    logAnalytics(request).catch(console.error);

    return new Response('OK');
  },
});
```

### ctx.passThroughOnException()

**Cloudflare:**
```typescript
export default {
  async fetch(request: Request, env: Env, ctx: ExecutionContext) {
    ctx.passThroughOnException();
    // If exception, fall through to origin
  },
};
```

**Bun (no equivalent, handle errors explicitly):**
```typescript
Bun.serve({
  async fetch(request: Request) {
    try {
      // Your code
      return new Response('OK');
    } catch (error) {
      // Forward to origin or handle error
      console.error(error);
      return new Response('Error', { status: 500 });
    }
  },
});
```

## Cache API

### Cloudflare Workers Cache

**Cloudflare:**
```typescript
export default {
  async fetch(request: Request) {
    const cache = caches.default;
    let response = await cache.match(request);

    if (!response) {
      response = await fetch(request);
      await cache.put(request, response.clone());
    }

    return response;
  },
};
```

**Bun (implement with Redis or in-memory):**
```typescript
import { Redis } from 'ioredis';

const redis = new Redis();

Bun.serve({
  async fetch(request: Request) {
    const cacheKey = new URL(request.url).pathname;
    const cached = await redis.get(cacheKey);

    if (cached) {
      return new Response(cached, {
        headers: { 'X-Cache': 'HIT' }
      });
    }

    const response = await fetch(request);
    const data = await response.text();

    await redis.setex(cacheKey, 3600, data);

    return new Response(data, {
      headers: { 'X-Cache': 'MISS' }
    });
  },
});
```

## WebSocket

### Cloudflare Workers

**Cloudflare:**
```typescript
export default {
  async fetch(request: Request) {
    const upgradeHeader = request.headers.get('Upgrade');

    if (upgradeHeader === 'websocket') {
      const pair = new WebSocketPair();
      const [client, server] = Object.values(pair);

      server.accept();
      server.addEventListener('message', (event) => {
        server.send(`Echo: ${event.data}`);
      });

      return new Response(null, {
        status: 101,
        webSocket: client,
      });
    }

    return new Response('Expected WebSocket', { status: 400 });
  },
};
```

**Bun:**
```typescript
Bun.serve({
  fetch(request, server) {
    const success = server.upgrade(request);

    if (success) {
      return undefined;
    }

    return new Response('Expected WebSocket', { status: 400 });
  },

  websocket: {
    message(ws, message) {
      ws.send(`Echo: ${message}`);
    },
    open(ws) {
      console.log('Client connected');
    },
    close(ws) {
      console.log('Client disconnected');
    },
  },
});
```

## HTMLRewriter

Cloudflare's HTMLRewriter doesn't have a direct Bun equivalent.

**Cloudflare:**
```typescript
export default {
  async fetch(request: Request) {
    const response = await fetch(request);

    return new HTMLRewriter()
      .on('h1', {
        element(element) {
          element.setInnerContent('Modified Title');
        },
      })
      .transform(response);
  },
};
```

**Bun (use cheerio or jsdom):**
```typescript
import * as cheerio from 'cheerio';

Bun.serve({
  async fetch(request: Request) {
    const response = await fetch(request);
    const html = await response.text();

    const $ = cheerio.load(html);
    $('h1').text('Modified Title');

    return new Response($.html(), {
      headers: { 'Content-Type': 'text/html' }
    });
  },
});
```

## Cloudflare-Specific APIs (No Direct Equivalent)

### Runtime APIs

| Cloudflare API | Bun Alternative | Notes |
|----------------|-----------------|-------|
| `navigator.userAgent` | `request.headers.get('user-agent')` | Same in both |
| `caches` | Redis, in-memory cache | Not built-in |
| `HTMLRewriter` | cheerio, jsdom | NPM packages |
| `WebSocketPair` | `Bun.serve({ websocket })` | Different API |
| `ScheduledEvent` | node-cron, cron jobs | NPM packages |
| `DurableObjectNamespace` | Database + Redis | Different architecture |

## Performance APIs

### Response Headers for Performance

**Same in both:**
```typescript
return new Response('Hello', {
  headers: {
    'Cache-Control': 'public, max-age=3600',
    'CDN-Cache-Control': 'max-age=86400',
  }
});
```

### Compression

**Cloudflare (automatic):**
```typescript
// Cloudflare auto-compresses responses
return new Response(largeData);
```

**Bun (manual with Hono middleware):**
```typescript
import { Hono } from 'hono';
import { compress } from 'hono/compress';

const app = new Hono();
app.use('*', compress());

app.get('/', (c) => c.text(largeData));

export default app;
```

## Environment Access

**Cloudflare (via env parameter):**
```typescript
export default {
  async fetch(request: Request, env: Env) {
    const apiKey = env.API_KEY;
    const db = env.DB;
  },
};
```

**Bun (via process.env):**
```typescript
Bun.serve({
  async fetch(request: Request) {
    const apiKey = process.env.API_KEY;
    // Database connections initialized separately
  },
});
```

## Request Context

### Getting Client IP

**Cloudflare:**
```typescript
const clientIp = request.headers.get('CF-Connecting-IP');
```

**Bun:**
```typescript
// Behind proxy (nginx, cloudflare)
const clientIp = request.headers.get('X-Forwarded-For')?.split(',')[0];

// Direct connection
Bun.serve({
  fetch(request, server) {
    const clientIp = server.requestIP(request);
  },
});
```

### Getting Request Country/Region

**Cloudflare:**
```typescript
const country = request.cf?.country;
const city = request.cf?.city;
```

**Bun (use GeoIP library):**
```typescript
import geoip from 'geoip-lite';

const clientIp = request.headers.get('X-Forwarded-For')?.split(',')[0];
const geo = geoip.lookup(clientIp);

const country = geo?.country;
const city = geo?.city;
```

## Streaming Responses

**Same API in both:**
```typescript
const stream = new ReadableStream({
  start(controller) {
    controller.enqueue(new TextEncoder().encode('chunk 1\n'));
    controller.enqueue(new TextEncoder().encode('chunk 2\n'));
    controller.close();
  }
});

return new Response(stream, {
  headers: { 'Content-Type': 'text/plain' }
});
```

## FormData Handling

**Same in both:**
```typescript
const formData = await request.formData();
const file = formData.get('file') as File;
const name = formData.get('name') as string;
```

## JSON Handling

**Same in both:**
```typescript
// Parse request JSON
const data = await request.json();

// Return JSON response
return Response.json({ message: 'Success' });
```

## Error Handling

**Same pattern in both:**
```typescript
try {
  const result = await riskyOperation();
  return Response.json({ result });
} catch (error) {
  console.error(error);
  return new Response('Internal Server Error', {
    status: 500
  });
}
```

## Summary

**High Compatibility (✅):**
- Standard Web APIs (fetch, Request, Response, etc.)
- JSON/FormData handling
- Streaming responses
- WebSocket (different API, same capability)

**Needs Replacement (⚠️):**
- ExecutionContext methods → Fire-and-forget promises
- Cache API → Redis or in-memory cache
- HTMLRewriter → cheerio/jsdom
- Bindings → Standard databases/services

**No Equivalent (❌):**
- Edge-specific features (distributed execution)
- CF-specific headers → GeoIP libraries
- Durable Objects → Traditional databases + architecture

Most Worker code can be migrated with minimal changes since both use standard Web APIs.
