#!/bin/bash
# ABOUTME: Create a Linear issue using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ] || [ -z "$LINEAR_TEAM_ID" ]; then
    echo "Error: LINEAR_API_KEY and LINEAR_TEAM_ID must be set"
    exit 1
fi

# Parse arguments
TITLE=""
DESCRIPTION=""
PROJECT_ID=""
ASSIGNEE_ID=""
PRIORITY=0
STATE_ID=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --title "Issue Title" [OPTIONS]

Create a Linear issue

Required:
    --title TEXT          Issue title

Optional:
    --description TEXT    Issue description
    --project-id ID       Project ID to assign to
    --assignee-id ID      User ID to assign to
    --priority NUM        Priority (0=none, 1=urgent, 2=high, 3=medium, 4=low)
    --state-id ID         Workflow state ID
    --help                Show this help

Examples:
    $(basename "$0") --title "Fix login bug"
    $(basename "$0") --title "Add feature" --description "Details here" --priority 2
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --title)
            TITLE="$2"
            shift 2
            ;;
        --description)
            DESCRIPTION="$2"
            shift 2
            ;;
        --project-id)
            PROJECT_ID="$2"
            shift 2
            ;;
        --assignee-id)
            ASSIGNEE_ID="$2"
            shift 2
            ;;
        --priority)
            PRIORITY="$2"
            shift 2
            ;;
        --state-id)
            STATE_ID="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$TITLE" ]; then
    echo "Error: --title is required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation IssueCreate($input: IssueCreateInput!) {
  issueCreate(input: $input) {
    success
    issue {
      id
      identifier
      title
      url
    }
  }
}'

# Build input object
INPUT="{\"teamId\":\"$LINEAR_TEAM_ID\",\"title\":\"$TITLE\""

if [ -n "$DESCRIPTION" ]; then
    # Escape description for JSON
    DESC_ESCAPED=$(echo "$DESCRIPTION" | jq -Rs .)
    INPUT="$INPUT,\"description\":$DESC_ESCAPED"
fi

if [ -n "$PROJECT_ID" ]; then
    INPUT="$INPUT,\"projectId\":\"$PROJECT_ID\""
fi

if [ -n "$ASSIGNEE_ID" ]; then
    INPUT="$INPUT,\"assigneeId\":\"$ASSIGNEE_ID\""
fi

if [ "$PRIORITY" != "0" ]; then
    INPUT="$INPUT,\"priority\":$PRIORITY"
fi

if [ -n "$STATE_ID" ]; then
    INPUT="$INPUT,\"stateId\":\"$STATE_ID\""
fi

INPUT="$INPUT}"

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"input\":$INPUT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error creating issue:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.issueCreate.success,
  id: .data.issueCreate.issue.id,
  identifier: .data.issueCreate.issue.identifier,
  title: .data.issueCreate.issue.title,
  url: .data.issueCreate.issue.url
}'
