#!/bin/bash
# ABOUTME: Create a label for a team using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ] || [ -z "$LINEAR_TEAM_ID" ]; then
    echo "Error: LINEAR_API_KEY and LINEAR_TEAM_ID must be set"
    exit 1
fi

# Parse arguments
NAME=""
DESCRIPTION=""
COLOR=""

show_help() {
    cat << EOF
Usage: $(basename "$0") --name "Label Name" [OPTIONS]

Create a label for the configured team

Required:
    --name TEXT           Label name

Optional:
    --description TEXT    Label description
    --color TEXT          Label color (hex code without #)
    --help                Show this help

Examples:
    $(basename "$0") --name "bug"
    $(basename "$0") --name "feature" --description "New feature" --color "00FF00"
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --name)
            NAME="$2"
            shift 2
            ;;
        --description)
            DESCRIPTION="$2"
            shift 2
            ;;
        --color)
            COLOR="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$NAME" ]; then
    echo "Error: --name is required"
    show_help
    exit 1
fi

# Build GraphQL mutation
MUTATION='mutation IssueLabelCreate($input: IssueLabelCreateInput!) {
  issueLabelCreate(input: $input) {
    success
    issueLabel {
      id
      name
      description
      color
      createdAt
    }
  }
}'

# Build input object
INPUT="{\"teamId\":\"$LINEAR_TEAM_ID\",\"name\":\"$NAME\""

if [ -n "$DESCRIPTION" ]; then
    DESC_ESCAPED=$(echo "$DESCRIPTION" | jq -Rs .)
    INPUT="$INPUT,\"description\":$DESC_ESCAPED"
fi

if [ -n "$COLOR" ]; then
    INPUT="$INPUT,\"color\":\"$COLOR\""
fi

INPUT="$INPUT}"

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$MUTATION" | jq -Rs .),\"variables\":{\"input\":$INPUT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error creating label:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '{
  success: .data.issueLabelCreate.success,
  id: .data.issueLabelCreate.issueLabel.id,
  name: .data.issueLabelCreate.issueLabel.name,
  description: .data.issueLabelCreate.issueLabel.description,
  color: .data.issueLabelCreate.issueLabel.color
}'
