#!/bin/bash
# ABOUTME: Get authenticated user information using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ]; then
    echo "Error: LINEAR_API_KEY must be set"
    exit 1
fi

show_help() {
    cat << EOF
Usage: $(basename "$0") [OPTIONS]

Get information about the authenticated user

Optional:
    --help                Show this help

Examples:
    $(basename "$0")
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

# Build GraphQL query
QUERY='query Viewer {
  viewer {
    id
    name
    email
    displayName
    active
    admin
    createdAt
    organization {
      id
      name
      urlKey
    }
  }
}'

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$QUERY" | jq -Rs .)}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error getting viewer:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '.data.viewer'
