#!/bin/bash
# ABOUTME: List attachments for a Linear issue using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ]; then
    echo "Error: LINEAR_API_KEY must be set"
    exit 1
fi

# Parse arguments
ISSUE_ID=""
LIMIT=50

show_help() {
    cat << EOF
Usage: $(basename "$0") --issue-id ID [OPTIONS]

List attachments for a Linear issue

Required:
    --issue-id ID         Issue ID to get attachments for

Optional:
    --limit NUM           Number of attachments to return (default: 50)
    --help                Show this help

Examples:
    $(basename "$0") --issue-id "abc123"
    $(basename "$0") --issue-id "abc123" --limit 20
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --issue-id)
            ISSUE_ID="$2"
            shift 2
            ;;
        --limit)
            LIMIT="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

if [ -z "$ISSUE_ID" ]; then
    echo "Error: --issue-id is required"
    show_help
    exit 1
fi

# Build GraphQL query
QUERY='query IssueAttachments($id: String!, $first: Int) {
  issue(id: $id) {
    id
    identifier
    title
    attachments(first: $first) {
      nodes {
        id
        title
        subtitle
        url
        createdAt
        creator {
          id
          name
          email
        }
      }
    }
  }
}'

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$QUERY" | jq -Rs .),\"variables\":{\"id\":\"$ISSUE_ID\",\"first\":$LIMIT}}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error listing attachments:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result
echo "$RESPONSE" | jq '.data.issue.attachments.nodes'
