#!/bin/bash
# ABOUTME: List Linear issues with filters using GraphQL API

set -e

# Source environment variables
if [ -f "$(dirname "$0")/../../.env" ]; then
    source "$(dirname "$0")/../../.env"
fi

# Validate required environment variables
if [ -z "$LINEAR_API_KEY" ] || [ -z "$LINEAR_TEAM_ID" ]; then
    echo "Error: LINEAR_API_KEY and LINEAR_TEAM_ID must be set"
    exit 1
fi

# Parse arguments
ASSIGNEE_ID=""
PROJECT_ID=""
STATE=""
LIMIT=50
AFTER=""

show_help() {
    cat << EOF
Usage: $(basename "$0") [OPTIONS]

List Linear issues with optional filters and pagination

Optional:
    --assignee-id ID      Filter by assignee user ID
    --project-id ID       Filter by project ID
    --state NAME          Filter by state name (e.g., "In Progress", "Backlog")
    --limit NUM           Number of issues to return (default: 50)
    --after CURSOR        Pagination cursor for next page
    --help                Show this help

Examples:
    $(basename "$0")
    $(basename "$0") --state "In Progress" --limit 10
    $(basename "$0") --project-id "abc123"
    $(basename "$0") --after "cursor123" --limit 25
EOF
}

while [[ $# -gt 0 ]]; do
    case $1 in
        --assignee-id)
            ASSIGNEE_ID="$2"
            shift 2
            ;;
        --project-id)
            PROJECT_ID="$2"
            shift 2
            ;;
        --state)
            STATE="$2"
            shift 2
            ;;
        --limit)
            LIMIT="$2"
            shift 2
            ;;
        --after)
            AFTER="$2"
            shift 2
            ;;
        --help)
            show_help
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

# Build GraphQL query
QUERY='query Issues($filter: IssueFilter, $first: Int, $after: String) {
  issues(filter: $filter, first: $first, after: $after) {
    nodes {
      id
      identifier
      title
      description
      priority
      url
      createdAt
      updatedAt
      assignee {
        id
        name
        email
      }
      state {
        id
        name
        type
      }
      project {
        id
        name
      }
    }
    pageInfo {
      hasNextPage
      endCursor
    }
  }
}'

# Build filter object
FILTER="{\"team\":{\"id\":{\"eq\":\"$LINEAR_TEAM_ID\"}}}"

if [ -n "$ASSIGNEE_ID" ]; then
    FILTER=$(echo "$FILTER" | jq --arg aid "$ASSIGNEE_ID" '.assignee.id.eq = $aid')
fi

if [ -n "$PROJECT_ID" ]; then
    FILTER=$(echo "$FILTER" | jq --arg pid "$PROJECT_ID" '.project.id.eq = $pid')
fi

if [ -n "$STATE" ]; then
    FILTER=$(echo "$FILTER" | jq --arg state "$STATE" '.state.name.eq = $state')
fi

# Build variables object
VARIABLES="{\"filter\":$FILTER,\"first\":$LIMIT"
if [ -n "$AFTER" ]; then
    VARIABLES="$VARIABLES,\"after\":\"$AFTER\""
fi
VARIABLES="$VARIABLES}"

# Make API request
RESPONSE=$(curl -s -X POST https://api.linear.app/graphql \
  -H "Authorization: $LINEAR_API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"query\":$(echo "$QUERY" | jq -Rs .),\"variables\":$VARIABLES}")

# Check for errors
if echo "$RESPONSE" | jq -e '.errors' > /dev/null 2>&1; then
    echo "Error listing issues:"
    echo "$RESPONSE" | jq '.errors'
    exit 1
fi

# Output result with pagination info
echo "$RESPONSE" | jq '{
  issues: .data.issues.nodes,
  pageInfo: .data.issues.pageInfo
}'
